package com.biz.crm.sfa.business.template.action.ordinary.local.service.internal;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.template.action.ordinary.local.entity.ActionDisplayEntity;
import com.biz.crm.sfa.business.template.action.ordinary.local.entity.ActionOrdinaryEntity;
import com.biz.crm.sfa.business.template.action.ordinary.local.model.ActionDisplayConditionModel;
import com.biz.crm.sfa.business.template.action.ordinary.local.repository.ActionDisplayExecuteRepository;
import com.biz.crm.sfa.business.template.action.ordinary.local.repository.ActionDisplayProductRepository;
import com.biz.crm.sfa.business.template.action.ordinary.local.repository.ActionDisplayRepository;
import com.biz.crm.sfa.business.template.action.ordinary.local.service.ActionDisplayExecuteService;
import com.biz.crm.sfa.business.template.action.ordinary.local.service.ActionDisplayProductService;
import com.biz.crm.sfa.business.template.action.ordinary.local.service.ActionDisplayService;
import com.biz.crm.sfa.business.template.action.ordinary.sdk.service.ActionOrdinaryVoService;
import com.biz.crm.sfa.business.template.action.ordinary.sdk.vo.ActionDisplayVo;
import com.biz.crm.sfa.business.template.action.ordinary.sdk.vo.ActionOrdinaryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static java.time.temporal.ChronoUnit.DAYS;

/**
 * 普通活动表服务实现类
 *
 * @author ning.zhang
 * @date 2022-07-28 17:56:37
 */
@Slf4j
@Service("actionDisplayService")
public class ActionDisplayServiceImpl implements ActionDisplayService {

  @Autowired
  private ActionDisplayRepository actionDisplayRepository;
  @Autowired
  private ActionDisplayProductRepository actionDisplayProductRepository;
  @Autowired
  private ActionDisplayProductService actionDisplayProductService;
  @Autowired
  private ActionDisplayExecuteRepository actionDisplayExecuteRepository;
  @Autowired
  private ActionDisplayExecuteService actionDisplayExecuteService;
  @Autowired
  private ActionOrdinaryVoService actionOrdinaryVoService;

  @Override
  @Transactional
  public ActionDisplayEntity update(ActionDisplayEntity entity) {
    this.updateValidation(entity);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    ActionDisplayConditionModel conditionModel = new ActionDisplayConditionModel();
    conditionModel.setTenantCode(TenantUtils.getTenantCode());
    conditionModel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    conditionModel.setActionCodes(Sets.newHashSet(entity.getActionCode()));
    List<ActionDisplayEntity> entities = this.actionDisplayRepository.findByActionDisplayConditionModel(conditionModel);
    if (!CollectionUtils.isEmpty(entities)) {
      Set<String> displayIds = entities.stream().map(ActionDisplayEntity::getId).collect(Collectors.toSet());
      this.actionDisplayRepository.removeByIds(displayIds);
      this.actionDisplayProductRepository.deleteByDisplayIds(displayIds);
    }
    this.actionDisplayRepository.save(entity);
    this.actionDisplayProductService.update(entity);
    return entity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> actionCodes) {
    this.actionDisplayRepository.updateDelFlagByActionCodes(actionCodes, TenantUtils.getTenantCode());
    this.actionDisplayExecuteRepository.deleteByActionCodes(actionCodes, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void enableBatch(List<String> actionCodes) {
    this.actionDisplayRepository.updateEnableStatusByActionCodes(actionCodes, EnableStatusEnum.ENABLE, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void disableBatch(List<String> actionCodes) {
    this.actionDisplayRepository.updateEnableStatusByActionCodes(actionCodes, EnableStatusEnum.DISABLE, TenantUtils.getTenantCode());
  }

  @Override
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "陈列活动执行信息生成定时任务")
  @Transactional
  public void executeTask() {
    Date nowDate = new Date();
    ActionDisplayConditionModel conditionModel = new ActionDisplayConditionModel();
    conditionModel.setTenantCode(TenantUtils.getTenantCode());
    conditionModel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    conditionModel.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    conditionModel.setInValidTime(BooleanEnum.TRUE.getCapital());
    conditionModel.setNowDate(nowDate);
    List<ActionDisplayEntity> entities = this.actionDisplayRepository.findByActionDisplayConditionModel(conditionModel);
    if (CollectionUtils.isEmpty(entities)) {
      log.info("无可生成执行信息的陈列活动");
      return;
    }
    entities = entities.stream().filter(entity -> {
      //没有生成过执行信息的活动,需要生成执行信息
      if (Objects.isNull(entity.getLastExecuteDate())) {
        return true;
      }
      long days = DAYS.between(entity.getLastExecuteDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate(), LocalDate.now());
      return days >= entity.getActionFrequency();
    }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      log.info("无可生成执行信息的陈列活动");
      return;
    }
    log.info("可生成执行信息的陈列活动数量:{}", entities.size());
    Set<String> actionCodes = entities.stream().map(ActionDisplayEntity::getActionCode).collect(Collectors.toSet());
    List<ActionOrdinaryVo> actionOrdinaryList = this.actionOrdinaryVoService.findByActionCodes(actionCodes);
    actionOrdinaryList.forEach(actionOrdinaryVo -> {
      this.actionDisplayExecuteService.create(actionOrdinaryVo.getActionCode(), actionOrdinaryVo.getScopeList());
    });
    entities.forEach(entity -> {
      entity.setLastExecuteDate(nowDate);
    });
    this.actionDisplayRepository.updateBatchById(entities);
  }

  /**
   * 在修改ActionOrdinary模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param entity 检查对象
   */
  private void updateValidation(ActionDisplayEntity entity) {
    Validate.notNull(entity, "进行当前操作时，信息对象必须传入!");
    entity.setId(null);
    entity.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(entity.getActionCode(), "缺失活动编码");
    Validate.notNull(entity.getActionFrequency(), "缺失活动频率");
    Validate.isTrue(entity.getActionFrequency() > 0, "活动频率数据错误");
    Validate.isTrue(!CollectionUtils.isEmpty(entity.getProductList()), "缺失陈列商品");
  }
}
