package com.biz.crm.sfa.business.template.action.ordinary.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.template.action.ordinary.local.entity.ActionOrdinaryEntity;
import com.biz.crm.sfa.business.template.action.ordinary.local.model.ActionOrdinaryConditionModel;
import com.biz.crm.sfa.business.template.action.ordinary.local.repository.ActionOrdinaryRepository;
import com.biz.crm.sfa.business.template.action.ordinary.local.service.ActionOrdinaryRequireService;
import com.biz.crm.sfa.business.template.action.ordinary.local.service.ActionOrdinaryScopeService;
import com.biz.crm.sfa.business.template.action.ordinary.local.service.ActionOrdinaryService;
import com.biz.crm.sfa.business.template.action.ordinary.sdk.dto.ActionOrdinaryDto;
import com.biz.crm.sfa.business.template.action.ordinary.sdk.register.ActionOrdinaryRegister;
import com.biz.crm.sfa.business.template.action.ordinary.sdk.vo.ActionOrdinaryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

/**
 * 普通活动表服务实现类
 *
 * @author ning.zhang
 * @date 2022-07-28 17:56:39
 */
@Slf4j
@Service("actionOrdinaryService")
public class ActionOrdinaryServiceImpl implements ActionOrdinaryService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private ActionOrdinaryRepository actionOrdinaryRepository;
  @Autowired(required = false)
  private List<ActionOrdinaryRegister> registers;
  @Autowired
  private ActionOrdinaryRequireService actionOrdinaryRequireService;
  @Autowired
  private ActionOrdinaryScopeService actionOrdinaryScopeService;

  @Override
  @Transactional
  public ActionOrdinaryEntity create(ActionOrdinaryDto dto) {
    this.createValidation(dto);
    ActionOrdinaryEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, ActionOrdinaryEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.actionOrdinaryRepository.save(entity);
    //绑定关联数据
    this.actionOrdinaryRequireService.update(entity);
    this.actionOrdinaryScopeService.update(entity);
    //新增普通活动事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      ActionOrdinaryVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, ActionOrdinaryVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public ActionOrdinaryEntity update(ActionOrdinaryDto dto) {
    this.updateValidation(dto);
    ActionOrdinaryConditionModel model = new ActionOrdinaryConditionModel();
    model.setIds(Sets.newHashSet(dto.getId()));
    model.setTenantCode(dto.getTenantCode());
    List<ActionOrdinaryEntity> entities = this.actionOrdinaryRepository.findByActionOrdinaryConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "普通活动信息不存在");
    ActionOrdinaryEntity oldEntity = entities.get(0);
    Validate.isTrue(oldEntity.getActionCode().equals(dto.getActionCode()), "活动编码不能编辑");
    Validate.isTrue(oldEntity.getTypeKey().equals(dto.getTypeKey()), "活动类型不能编辑");
    Validate.isTrue(oldEntity.getActionStartDate().getTime() > System.currentTimeMillis(), "只能编辑未开始活动");
    ActionOrdinaryEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, ActionOrdinaryEntity.class, HashSet.class, ArrayList.class);
    this.actionOrdinaryRepository.updateById(updateEntity);
    //绑定关联数据
    this.actionOrdinaryRequireService.update(updateEntity);
    this.actionOrdinaryScopeService.update(updateEntity);
    //更新普通活动事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      ActionOrdinaryVo oldVo = this.nebulaToolkitService.copyObjectByBlankList(oldEntity, ActionOrdinaryVo.class, HashSet.class, ArrayList.class);
      ActionOrdinaryVo newVo = this.nebulaToolkitService.copyObjectByBlankList(updateEntity, ActionOrdinaryVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<ActionOrdinaryEntity> entities = this.actionOrdinaryRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    entities.forEach(entity -> {
      Validate.isTrue(entity.getActionStartDate().getTime() > System.currentTimeMillis(), "只能删除未开始活动");
    });
    this.actionOrdinaryRepository.updateDelFlagByIds(ids);
    //删除普通活动事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      List<ActionOrdinaryVo> voList = (List<ActionOrdinaryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, ActionOrdinaryEntity.class
          , ActionOrdinaryVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    ActionOrdinaryConditionModel model = new ActionOrdinaryConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<ActionOrdinaryEntity> entities = this.actionOrdinaryRepository.findByActionOrdinaryConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.actionOrdinaryRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用普通活动事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      List<ActionOrdinaryVo> voList = (List<ActionOrdinaryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, ActionOrdinaryEntity.class
          , ActionOrdinaryVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    ActionOrdinaryConditionModel model = new ActionOrdinaryConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<ActionOrdinaryEntity> entities = this.actionOrdinaryRepository.findByActionOrdinaryConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.actionOrdinaryRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //启用普通活动事件通知
    if (!CollectionUtils.isEmpty(registers)) {
      List<ActionOrdinaryVo> voList = (List<ActionOrdinaryVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, ActionOrdinaryEntity.class
          , ActionOrdinaryVo.class, HashSet.class, ArrayList.class);
      registers.forEach(register -> {
        register.onDisable(voList);
      });
    }
  }


  /**
   * 在创建ActionOrdinary模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(ActionOrdinaryDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setActionCode(generateCodeService.generateCode("AO", 1).get(0));
    Validate.notBlank(dto.getActionName(), "缺失活动名称");
    Validate.notBlank(dto.getTypeKey(), "缺失活动类型key");
    Validate.notNull(dto.getActionStartDate(), "缺失活动开始时间");
    Validate.notNull(dto.getActionEndDate(), "缺失活动结束时间");
    Validate.isTrue(dto.getActionStartDate().getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于活动开始时间");
    Date nowDate = new Date();
    Validate.isTrue(nowDate.getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于当前时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRequireList()), "缺失活动要求");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失活动范围");
  }

  /**
   * 在修改ActionOrdinary模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(ActionOrdinaryDto dto) {

    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getActionCode(), "缺失活动编码");
    Validate.notBlank(dto.getActionName(), "缺失活动名称");
    Validate.notBlank(dto.getTypeKey(), "缺失活动类型key");
    Validate.notNull(dto.getActionStartDate(), "缺失活动开始时间");
    Validate.notNull(dto.getActionEndDate(), "缺失活动结束时间");
    Validate.isTrue(dto.getActionStartDate().getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于活动开始时间");
    Date nowDate = new Date();
    Validate.isTrue(nowDate.getTime() < dto.getActionEndDate().getTime(), "活动结束时间需大于当前时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRequireList()), "缺失活动要求");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失活动范围");
  }
}
