package com.biz.crm.sfa.business.template.action.tpm.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.sfa.business.template.action.tpm.local.entity.ActionTpmExecuteEntity;
import com.biz.crm.sfa.business.template.action.tpm.local.entity.ActionTpmProtocolEntity;
import com.biz.crm.sfa.business.template.action.tpm.local.model.ActionTpmBasicModel;
import com.biz.crm.sfa.business.template.action.tpm.local.model.ActionTpmModel;
import com.biz.crm.sfa.business.template.action.tpm.local.model.ActionTpmProtocolModel;
import com.biz.crm.sfa.business.template.action.tpm.local.repository.ActionTpmExecuteRepository;
import com.biz.crm.sfa.business.template.action.tpm.local.service.ActionTpmModelService;
import com.biz.crm.sfa.business.template.action.tpm.sdk.dto.ActionTpmExecuteDto;
import com.biz.crm.sfa.business.template.action.tpm.sdk.register.ActionTpmExecuteRegister;
import com.biz.crm.tpm.business.activities.sdk.service.ActivitiesDetailService;
import com.biz.crm.tpm.business.activities.sdk.vo.ActivitiesDetailVo;
import com.biz.crm.tpm.business.activities.sdk.vo.ActivitiesVo;
import com.biz.crm.tpm.business.budget.sdk.enums.StrategySettingExecutorOprtType;
import com.biz.crm.tpm.business.budget.sdk.enums.StrategySettingType;
import com.biz.crm.tpm.business.budget.sdk.strategy.setting.BusinessStrategySettingExecutor;
import com.biz.crm.tpm.business.budget.sdk.vo.CostTypeDetailVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * TPM活动表单(ActionTpmModel)服务接口实现类
 *
 * @author ning.zhang
 * @date 2022/7/1
 */
@Slf4j
@Service
public class ActionTpmModelServiceImpl implements ActionTpmModelService {

  @Autowired
  private ActionTpmExecuteRepository actionTpmExecuteRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private List<ActionTpmExecuteRegister> registers;
  @Autowired
  private ActivitiesDetailService activitiesDetailService;
  @Autowired(required = false)
  private List<BusinessStrategySettingExecutor> businessStrategySettingExecutors;

  @Override
  public ActionTpmModel findByActionTpmExecuteDto(ActionTpmExecuteDto dto) {
    if (Objects.isNull(dto) || StringUtils.isAnyBlank(dto.getActionCode(), dto.getActionDetailCode(), dto.getTerminalCode())) {
      return null;
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<ActionTpmExecuteEntity> entities = this.actionTpmExecuteRepository.findByActionTpmExecuteDto(dto);
    ActionTpmModel actionTpmModel = new ActionTpmModel();
    actionTpmModel.setParentCode(dto.getParentCode());
    actionTpmModel.setDynamicKey(dto.getDynamicKey());
    actionTpmModel.setDisplayProtocols(this.buildProtocolInfo(entities));
    actionTpmModel.setRelationMap(this.buildRelationMap(entities));
    this.buildActionDetails(dto, actionTpmModel, entities);
    return actionTpmModel;
  }

  /**
   * 构建活动执行表单活动详细信息
   *
   * @param dto            参数dto
   * @param actionTpmModel 活动执行model
   */
  private void buildActionDetails(ActionTpmExecuteDto dto, ActionTpmModel actionTpmModel, List<ActionTpmExecuteEntity> entities) {
    ActionTpmBasicModel basicModel = new ActionTpmBasicModel();
    basicModel.setTerminalCode(dto.getTerminalCode());
    basicModel.setTerminalName(dto.getTerminalName());
    ActivitiesDetailVo detailVo = this.activitiesDetailService.findDetailByActivitiesDetailCode(dto.getActionDetailCode());
    if (Objects.isNull(detailVo)) {
      return;
    }

    ActivitiesVo activitiesVo = ObjectUtils.defaultIfNull(detailVo.getActivities(), new ActivitiesVo());
    CostTypeDetailVo costTypeDetailVo = ObjectUtils.defaultIfNull(detailVo.getCostTypeDetailVo(), new CostTypeDetailVo());
    basicModel.setActionCode(detailVo.getActivitiesCode());
    basicModel.setActionDetailCode(detailVo.getActivitiesDetailCode());
    basicModel.setActionDescribe(activitiesVo.getRemark());
    basicModel.setActionEndTime(activitiesVo.getEndTime());
    basicModel.setActionName(detailVo.getActivitiesName());
    basicModel.setActionStartTime(activitiesVo.getBeginTime());
    basicModel.setPayType(detailVo.getPayType());
    basicModel.setPayTypeName(detailVo.getPayTypeName());
    basicModel.setStoreUsedAmount(ObjectUtils.defaultIfNull(detailVo.getTerminalAmount(), BigDecimal.ZERO));
    basicModel.setTotalAmount(ObjectUtils.defaultIfNull(detailVo.getApplyAmount(), BigDecimal.ZERO));
    basicModel.setUsedAmount(ObjectUtils.defaultIfNull(detailVo.getTotalAmount(), BigDecimal.ZERO));
    BusinessStrategySettingExecutor executor = BusinessStrategySettingExecutor.getExecutor(businessStrategySettingExecutors, StrategySettingType.ACTIVITY.name());
    basicModel.setIsCollectData(Objects.nonNull(executor) && executor.matchedOprtType(costTypeDetailVo.getSettingStrategies(), StrategySettingExecutorOprtType.COLLECT_DATA.name(),true)
        ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital());
    basicModel.setIsCollectDistributionOrder(Objects.nonNull(executor) && executor.matchedOprtType(costTypeDetailVo.getSettingStrategies(), StrategySettingExecutorOprtType.COLLECT_ORDER.name(),true)
        ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital());
    basicModel.setIsSignDisplayAgreement(Objects.nonNull(executor) && executor.matchedOprtType(costTypeDetailVo.getSettingStrategies(), StrategySettingExecutorOprtType.SIGN_DISPLAY_AGREEMENT.name(),true)
        ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital());
    actionTpmModel.setBasicInfo(basicModel);
    actionTpmModel.setActivitiesDetailCollects(ObjectUtils.defaultIfNull(detailVo.getActivitiesDetailCollects(),Lists.newArrayList()));
    actionTpmModel.setFieldsCollectTemplates(ObjectUtils.defaultIfNull(detailVo.getFieldsCollectTemplates(),Lists.newArrayList()));
    actionTpmModel.setFieldsCollectTemplateStruct(detailVo.getFieldsCollectTemplateStruct());
    actionTpmModel.setPhotoRequires(ObjectUtils.defaultIfNull(detailVo.getPhotoRequires(),Lists.newArrayList()));
    //过滤非当前批次的所有TPM数据
    List<String> executeCodes = !CollectionUtils.isEmpty(entities)
        ? entities.stream().map(ActionTpmExecuteEntity::getExecuteCode).collect(Collectors.toList()) : Lists.newArrayList();
    actionTpmModel.setFieldsCollectTemplates(actionTpmModel.getFieldsCollectTemplates().stream().filter(o -> executeCodes.contains(o.getBtNo())).collect(Collectors.toList()));
    actionTpmModel.setActivitiesDetailCollects(actionTpmModel.getActivitiesDetailCollects().stream().filter(o -> executeCodes.contains(o.getBtNo())).collect(Collectors.toList()));
  }

  /**
   * 构建执行表单协议数据
   *
   * @param entities 活动执行实体信息
   * @return 执行表单协议数据
   */
  private List<ActionTpmProtocolModel> buildProtocolInfo(List<ActionTpmExecuteEntity> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    List<String> executeIds = entities.stream().map(ActionTpmExecuteEntity::getId).collect(Collectors.toList());
    List<ActionTpmExecuteEntity> detailEntities = this.actionTpmExecuteRepository.findByIds(executeIds);
    List<ActionTpmProtocolModel> list = Lists.newArrayList();
    detailEntities.forEach(entity -> {
      if (!CollectionUtils.isEmpty(entity.getDisplayProtocols())) {
        list.addAll(this.nebulaToolkitService.copyCollectionByBlankList(entity.getDisplayProtocols()
            , ActionTpmProtocolEntity.class, ActionTpmProtocolModel.class, HashSet.class, ArrayList.class));
      }
    });
    return list;
  }

  /**
   * 构建活动执行上层关联数据映射(key:上层业务key,value:上层业务数据)
   *
   * @param entities 活动执行实体信息
   * @return 活动执行上层关联数据映射
   */
  private Map<String, List<JSONObject>> buildRelationMap(List<ActionTpmExecuteEntity> entities) {
    Map<String, List<JSONObject>> relationMap = Maps.newHashMap();
    if (CollectionUtils.isEmpty(entities)) {
      return relationMap;
    }
    Set<String> executeCodes = entities.stream().map(ActionTpmExecuteEntity::getExecuteCode).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(registers)) {
      return relationMap;
    }
    registers.forEach(register -> {
      Map<String, List<JSONObject>> map = register.onRequestByExecuteCodes(executeCodes);
      if (Objects.isNull(map)) {
        return;
      }
      //汇总所有的执行编码关联的数据
      relationMap.put(register.relationKey(), map.values().stream().flatMap(Collection::stream).collect(Collectors.toList()));
    });
    return relationMap;
  }
}
