package com.biz.crm.sfa.business.inventory.check.local.service.internal;

import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.sfa.business.inventory.check.local.entity.InventoryCheckDetailEntity;
import com.biz.crm.sfa.business.inventory.check.local.entity.InventoryCheckEntity;
import com.biz.crm.sfa.business.inventory.check.local.repository.InventoryCheckDetailRepository;
import com.biz.crm.sfa.business.inventory.check.local.repository.InventoryCheckRepository;
import com.biz.crm.sfa.business.inventory.check.local.service.InventoryCheckService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Set;

/**
 * 库存盘点接口实现类
 */
@Service
public class InventoryCheckServiceImpl implements InventoryCheckService {

  @Autowired
  private InventoryCheckRepository inventoryCheckRepository;

  @Autowired
  private InventoryCheckDetailRepository inventoryCheckDetailRepository;

  @Autowired
  private LoginUserService loginUserService;

  @Override
  public InventoryCheckEntity create(InventoryCheckEntity inventoryCheckEntity) {
    this.validateCreateData(inventoryCheckEntity);
    if (!StringUtils.isAnyBlank(inventoryCheckEntity.getParentCode(), inventoryCheckEntity.getDynamicKey())) {
      InventoryCheckEntity existEntity = this.inventoryCheckRepository.findByParentCodeAndDynamicKey(inventoryCheckEntity.getParentCode(), inventoryCheckEntity.getDynamicKey());
      Validate.isTrue(ObjectUtils.isEmpty(existEntity), "此业务已经提交过");
    }
    this.inventoryCheckRepository.save(inventoryCheckEntity);
    inventoryCheckEntity.getDetails().forEach(detail -> detail.setCheckId(inventoryCheckEntity.getId()));
    this.inventoryCheckDetailRepository.saveBatch(inventoryCheckEntity.getDetails());
    return inventoryCheckEntity;
  }

  @Override
  public InventoryCheckEntity update(InventoryCheckEntity inventoryCheckEntity) {
    this.validateCreateData(inventoryCheckEntity);
    Validate.notNull(inventoryCheckEntity.getId(), "待更新的数据不存在！");
    this.inventoryCheckRepository.saveOrUpdate(inventoryCheckEntity);
    this.inventoryCheckDetailRepository.deleteByCheckId(inventoryCheckEntity.getId());
    inventoryCheckEntity.getDetails().forEach(detail -> detail.setCheckId(inventoryCheckEntity.getId()));
    this.inventoryCheckDetailRepository.saveBatch(inventoryCheckEntity.getDetails());
    return inventoryCheckEntity;
  }

  @Override
  public Page<InventoryCheckEntity> findByConditions(Pageable pageable, InventoryCheckEntity inventoryCheckEntity) {
    return this.inventoryCheckRepository.findByConditions(pageable, inventoryCheckEntity);
  }

  @Override
  public InventoryCheckEntity findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    InventoryCheckEntity inventoryCheckEntity = this.inventoryCheckRepository.getById(id);
    if (ObjectUtils.isEmpty(inventoryCheckEntity)) {
      return null;
    }
    inventoryCheckEntity.setDetails(this.inventoryCheckDetailRepository.findByCheckId(inventoryCheckEntity.getId()));
    return inventoryCheckEntity;
  }

  @Override
  public InventoryCheckEntity findLatestByClientTypeAndClientCode(String clientType, String clientCode) {
    if (StringUtils.isAnyBlank(clientCode, clientType)) {
      return null;
    }
    List<InventoryCheckEntity> latestByClientTypeAndClientCode = this.inventoryCheckRepository.findLatestByClientTypeAndClientCode(clientType, clientCode);
    if (CollectionUtils.isEmpty(latestByClientTypeAndClientCode)) {
      return null;
    }
    return latestByClientTypeAndClientCode.get(0);
  }


  /**
   * 校验库存盘点创建数据
   *
   * @param inventoryCheckEntity 库存盘点实体
   */
  private void validateCreateData(InventoryCheckEntity inventoryCheckEntity) {
    Validate.notNull(inventoryCheckEntity, "库存盘点实体不能为空！");
    Validate.notNull(inventoryCheckEntity.getClientCode(), "站点编码不能为空");
    Validate.notNull(inventoryCheckEntity.getClientType(), "站点类别不能为空");
    Validate.notEmpty(inventoryCheckEntity.getDetails(), "库存盘点详情不能为空");
    inventoryCheckEntity.getDetails().forEach(this::validateDetail);
    inventoryCheckEntity.setProductLevelQuantify(String.valueOf(this.getProductLevelNumber(inventoryCheckEntity.getDetails())));
    //构造基础数据
    LoginUserDetails loginUserDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    inventoryCheckEntity.setUserCode(loginUserDetails.getAccount());
    inventoryCheckEntity.setUserName(loginUserDetails.getRealName());
    inventoryCheckEntity.setPostCode(loginUserDetails.getPostCode());
    inventoryCheckEntity.setPostName(loginUserDetails.getPostName());
    inventoryCheckEntity.setTenantCode(loginUserDetails.getTenantCode());
  }

  /**
   * 校验库存盘点创建数据
   *
   * @param inventoryCheckDetail 库存盘点详情
   */
  private void validateDetail(InventoryCheckDetailEntity inventoryCheckDetail) {
    Validate.notNull(inventoryCheckDetail, "库存盘点详情实体不能为空！");
    inventoryCheckDetail.setId(null);
    Validate.notNull(inventoryCheckDetail.getProductCode(), "产品编码不能为空");
    Validate.notNull(inventoryCheckDetail.getProductLevelCode(), "产品层级编码不能为空");
    Validate.notEmpty(inventoryCheckDetail.getQuantify(), "库存盘点数量不能为空");
  }

  /**
   * 获取产品层级数量
   *
   * @param detailEntities 库存盘点详情列表
   * @return 产品层级数量
   */
  private int getProductLevelNumber(List<InventoryCheckDetailEntity> detailEntities) {
    Set<String> produceLevelCodes = Sets.newHashSet();
    detailEntities.forEach(detail -> {
      if (StringUtils.isNotBlank(detail.getProductLevelCode())) {
        produceLevelCodes.add(detail.getProductLevelCode());
      }
    });
    return produceLevelCodes.size();
  }
}
