package com.biz.crm.sfa.business.template.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.utils.DistanceUtil;
import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.OperationStrategy;
import com.biz.crm.common.form.sdk.module.ModuleRegister;
import com.biz.crm.sfa.business.client.sdk.strategy.ClientRouteStrategy;
import com.biz.crm.sfa.business.client.sdk.vo.ClientRouteInfoVo;
import com.biz.crm.sfa.business.step.sdk.enums.VisitTypeEnum;
import com.biz.crm.sfa.business.step.sdk.listerner.StepFormIntegralListener;
import com.biz.crm.sfa.business.step.sdk.register.NecessaryFormRegister;
import com.biz.crm.sfa.business.step.sdk.service.StepFormVoService;
import com.biz.crm.sfa.business.step.sdk.strategy.ExecutePlanCallBackStrategy;
import com.biz.crm.sfa.business.step.sdk.vo.StepFormVo;
import com.biz.crm.sfa.business.template.sdk.abstracts.AbstractDynamicTemplateModel;
import com.biz.crm.sfa.business.template.sdk.dto.DynamicTemplateDto;
import com.biz.crm.sfa.business.template.sdk.dto.DynamicTemplateModelDto;
import com.biz.crm.sfa.business.template.sdk.service.DynamicTemplateService;
import com.biz.crm.sfa.business.template.sdk.strategy.DynamicTemplateOperationStrategy;
import com.biz.crm.sfa.business.template.sdk.vo.DynamicTemplateVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 动态表单模板基础服务实现类
 *
 * @author songjingen
 */
@Service
public class DynamicTemplateServiceImpl implements DynamicTemplateService {

  @Order
  @Autowired(required = false)
  private List<ModuleRegister> moduleRegisters;
  @Order
  @Autowired(required = false)
  private List<OperationStrategy<? extends DynamicForm>> operationStrategies;
  @Autowired(required = false)
  private List<DynamicTemplateOperationStrategy> dynamicTemplateOperationStrategies;
  @Autowired
  private List<ExecutePlanCallBackStrategy> completeStatusCallBackStrategies;
  @Autowired(required = false)
  private List<NecessaryFormRegister> necessaryFormRegisters;
  @Autowired(required = false)
  private List<StepFormIntegralListener> stepFormIntegralListeners;
  @Autowired(required = false)
  private StepFormVoService stepFormVoService;
  @Autowired(required = false)
  private List<ClientRouteStrategy> clientRouteStrategies;

  @Override
  public Page<DynamicTemplateVo> findByConditions(Pageable pageable, DynamicTemplateDto dto) {
    if (CollectionUtils.isEmpty(moduleRegisters) || CollectionUtils.isEmpty(operationStrategies)) {
      return new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    }
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    if (dto == null) {
      dto = new DynamicTemplateDto();
    }
    //1.====获取模块分组信息
    List<DynamicTemplateVo> templates = Lists.newArrayList();
    for (ModuleRegister moduleRegister : moduleRegisters) {
      String moduleCode = moduleRegister.moduleCode();
      String moduleName = moduleRegister.moduleName();
      //2.====获取动态模板策略信息
      for (OperationStrategy<? extends DynamicForm> operationStrategy : operationStrategies) {
        String moduleCodeItem = operationStrategy.moduleCode();
        if (!StringUtils.equals(moduleCode, moduleCodeItem)) {
          continue;
        }
        String dynamicFormCode = operationStrategy.dynamicFormCode();
        String dynamicFormName = operationStrategy.dynamicFormName();
        Class<?> dynamicClass = operationStrategy.dynamicFormClass();
        DynamicTemplateVo template = new DynamicTemplateVo();
        template.setModuleCode(moduleCode);
        template.setModuleName(moduleName);
        template.setDynamicFormCode(dynamicFormCode);
        template.setDynamicFormName(dynamicFormName);
        template.setDynamicFormSimpleClass(dynamicClass.getSimpleName());
        templates.add(template);
      }
    }

    if (CollectionUtils.isEmpty(templates)) {
      return new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    }

    //3.====根据筛选条件过滤数据
    if (StringUtils.isNotBlank(dto.getModuleCode())) {
      String moduleCode = dto.getModuleCode();
      templates = templates.stream().filter(e -> StringUtils.equals(moduleCode, e.getModuleCode())).collect(Collectors.toList());
    }
    if (StringUtils.isNotBlank(dto.getModuleName())) {
      String moduleName = dto.getModuleName();
      templates = templates.stream().filter(e -> StringUtils.indexOf(e.getModuleName(), moduleName) >= 0).collect(Collectors.toList());
    }
    if (StringUtils.isNotBlank(dto.getDynamicFormCode())) {
      String dynamicFormCode = dto.getDynamicFormCode();
      templates = templates.stream().filter(e -> StringUtils.equals(dynamicFormCode, e.getDynamicFormCode())).collect(Collectors.toList());
    }
    if (StringUtils.isNotBlank(dto.getDynamicFormName())) {
      String dynamicFormName = dto.getDynamicFormName();
      templates = templates.stream().filter(e -> StringUtils.indexOf(e.getDynamicFormName(), dynamicFormName) >= 0).collect(Collectors.toList());
    }
    long totalSize = templates.size();
    templates = templates.stream().skip(pageable.getPageSize() * pageable.getPageNumber() > 0 ? pageable.getPageNumber() - 1 : 0)
        .limit(pageable.getPageSize()).collect(Collectors.toList());
    Page<DynamicTemplateVo> result = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    result.setRecords(templates);
    result.setTotal(totalSize);
    return result;
  }

  /**
   * 1、验证信息，如果是步骤表单提交则验证parentCode和dynamicKey字段；反之则不校验
   * 2、调用表单操作策略进行创建表单
   * 3、如果是步骤表单提交，则验证拜访状态，然后调取执行回调策略，更改完成状态以及执行计划状态；反之则不进行任何操作
   *
   * @param jsonObject
   */
  @Transactional
  @Override
  public void handleCreateForm(JSONObject jsonObject) {
    Validate.notNull(jsonObject, "创建表单信息时，入参json对象不能为空！");
    String dynamicFormCode = jsonObject.getString("dynamicFormCode");
    Validate.notBlank(dynamicFormCode, "创建表单信息时，表单编码（表单编码）不能为空！");
    Boolean stepFormOrNot = jsonObject.getBoolean("stepFormOrNot");
    Validate.notNull(stepFormOrNot, "创建表单信息时，是否为步骤表单参数不能为空！");
    String parentCode = jsonObject.getString("parentCode");
    String dynamicKey = jsonObject.getString("dynamicKey");
    //1、======
    if (stepFormOrNot) {
      Validate.notBlank(parentCode, "创建表单信息时，表单关联父级编码（明细唯一编码）不能为空！");
      Validate.notBlank(dynamicKey, "创建表单信息时，表单关联动态key（步骤编码）不能为空！");
      //验证步骤定位
      this.validateLongitudeAndLatitude(jsonObject);
    }
    //2、======
    Validate.isTrue(!CollectionUtils.isEmpty(dynamicTemplateOperationStrategies), "未查询到表单模型服务的操作策略");
    for (DynamicTemplateOperationStrategy dynamicTemplateOperationStrategy : dynamicTemplateOperationStrategies) {
      if (dynamicTemplateOperationStrategy.dynamicFormCode().equals(dynamicFormCode)) {
        dynamicTemplateOperationStrategy.onDynamicTemplateCreate(jsonObject, dynamicKey, parentCode);
        break;
      }
    }
    //3、======
    if (stepFormOrNot) {
      String visitType = jsonObject.getString("visitType");
      Validate.notBlank(visitType, "回调完成状态时，拜访类型不能为空！");
      Validate.notNull(VisitTypeEnum.getByDictCode(visitType), "回调更改完成状态时，未匹配到对应的拜访类型！");
      //表单完成状态更新
      if (!CollectionUtils.isEmpty(completeStatusCallBackStrategies)) {
        for (ExecutePlanCallBackStrategy executePlanCallBackStrategy : completeStatusCallBackStrategies) {
          if (executePlanCallBackStrategy.visitType().equals(visitType)) {
            executePlanCallBackStrategy.onSave(parentCode, dynamicKey, dynamicFormCode);
            break;
          }
        }
      }
      //判断首尾步骤更改执行状态
      if (!CollectionUtils.isEmpty(necessaryFormRegisters)) {
        NecessaryFormRegister firstNecessaryFormRegister = necessaryFormRegisters.stream().filter(NecessaryFormRegister::first).sorted().collect(Collectors.toList()).get(0);
        NecessaryFormRegister lastNecessaryFormRegister = necessaryFormRegisters.stream().filter(NecessaryFormRegister::last).sorted().collect(Collectors.toList()).get(0);
        Validate.notNull(firstNecessaryFormRegister, "更改执行计划状态时，未获取到系统预设的首位必做步骤表单！");
        Validate.notNull(lastNecessaryFormRegister, "更改执行计划状态时，未获取到系统预设的末位必做步骤表单！");
        for (ExecutePlanCallBackStrategy executePlanCallBackStrategy : completeStatusCallBackStrategies) {
          if (executePlanCallBackStrategy.visitType().equals(visitType)) {
            if (firstNecessaryFormRegister.dynamicFormCode().equals(dynamicFormCode)) {
              executePlanCallBackStrategy.onUpdateExecuteStatus(false, parentCode);
            }
            if (lastNecessaryFormRegister.dynamicFormCode().equals(dynamicFormCode)) {
              executePlanCallBackStrategy.onUpdateExecuteStatus(true, parentCode);
            }
            break;
          }
        }
      }
    }
    //4、======
    if (!CollectionUtils.isEmpty(stepFormIntegralListeners)) {
      for (StepFormIntegralListener stepFormIntegralListener : stepFormIntegralListeners) {
        stepFormIntegralListener.onCreateIntegral(parentCode, dynamicKey, dynamicFormCode);
      }
    }

  }

  @Transactional
  @Override
  public void handleUpdateForm(JSONObject jsonObject) {
    Validate.notNull(jsonObject, "编辑表单信息时，入参json对象不能为空！");
    String dynamicFormCode = jsonObject.getString("dynamicFormCode");
    Validate.notBlank(dynamicFormCode, "编辑表单信息时，表单编码（表单编码）不能为空！");
    Boolean stepFormOrNot = jsonObject.getBoolean("stepFormOrNot");
    Validate.notNull(stepFormOrNot, "编辑表单信息时，是否为步骤表单参数不能为空！");
    String parentCode = jsonObject.getString("parentCode");
    String dynamicKey = jsonObject.getString("dynamicKey");
    if (stepFormOrNot) {
      Validate.notBlank(parentCode, "创建表单信息时，表单关联父级编码（明细唯一编码）不能为空！");
      Validate.notBlank(dynamicKey, "创建表单信息时，表单关联动态key（步骤编码）不能为空！");
      //验证步骤定位
      this.validateLongitudeAndLatitude(jsonObject);
    }
    Validate.isTrue(!CollectionUtils.isEmpty(dynamicTemplateOperationStrategies), "未查询到表单模型服务的操作策略");
    for (DynamicTemplateOperationStrategy dynamicTemplateOperationStrategy : dynamicTemplateOperationStrategies) {
      if (dynamicTemplateOperationStrategy.dynamicFormCode().equals(dynamicFormCode)) {
        dynamicTemplateOperationStrategy.onDynamicTemplateModify(jsonObject, dynamicKey, parentCode);
        break;
      }
    }
  }

  @Override
  public AbstractDynamicTemplateModel findByDynamicTemplateModelDto(DynamicTemplateModelDto model) {
    if (model == null || StringUtils.isAnyBlank(model.getDynamicFormCode(), model.getDynamicKey(), model.getParentCode())) {
      return null;
    }
    if (CollectionUtils.isEmpty(dynamicTemplateOperationStrategies)) {
      return null;
    }
    for (DynamicTemplateOperationStrategy dynamicTemplateOperationStrategy : dynamicTemplateOperationStrategies) {
      if (dynamicTemplateOperationStrategy.dynamicFormCode().equals(model.getDynamicFormCode())) {
        return dynamicTemplateOperationStrategy.findByParentCode(model.getDynamicKey(), model.getParentCode());
      }
    }
    return null;
  }

  /**
   * 验证是否需要定位
   *
   * @param jsonObject
   */
  private void validateLongitudeAndLatitude(JSONObject jsonObject) {
    String stepCode = jsonObject.getString("dynamicKey");
    List<StepFormVo> stepCodes = this.stepFormVoService.findByStepCodes(Sets.newHashSet(stepCode));
    Validate.isTrue(!CollectionUtils.isEmpty(stepCodes), "未查询到可执行的步骤！");
    StepFormVo stepFormVo = stepCodes.get(0);
    if (!stepFormVo.getLocateType()) {
      return;
    }
    BigDecimal longitude = jsonObject.getBigDecimal("longitude");
    BigDecimal latitude = jsonObject.getBigDecimal("latitude");
    String clientCode = jsonObject.getString("clientCode");
    String clientType = jsonObject.getString("clientType");
    String clientRoute = jsonObject.getString("clientRoute");
    Validate.notNull(latitude, "验证是否定位时，纬度不能为空！");
    Validate.notNull(longitude, "验证是否定位时，经度不能为空！");
    Validate.notBlank(clientCode, "验证是否定位时，客户编码不能为空！");
    Validate.notBlank(clientType, "验证是否定位时，客户类型不能为空！");
    Validate.notBlank(clientRoute, "验证是否定位时，客户维度不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(clientRouteStrategies), "未查询到客户维度注册器！");
    List<ClientRouteStrategy> currentClientRouteStrategies = clientRouteStrategies.stream().filter(clientRouteStrategy -> clientRouteStrategy.getKey().equals(clientRoute)).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(currentClientRouteStrategies), "无法识别当前客户的来源维度！");
    ClientRouteInfoVo clientRouteInfoVo = new ClientRouteInfoVo();
    for (ClientRouteStrategy clientRouteStrategy : clientRouteStrategies) {
      if (clientRoute.equals(clientRouteStrategy.getKey())) {
        List<ClientRouteInfoVo> clientRouteInfoVos = clientRouteStrategy.onFindByClientCodesAndType(Lists.newArrayList(clientCode), clientType);
        Validate.notNull(!CollectionUtils.isEmpty(clientRouteInfoVos), "验证是否定位时，未查询到客户信息");
        clientRouteInfoVo = clientRouteInfoVos.get(0);
        break;
      }
    }
    //查看客户信息
    BigDecimal longitudeParam = clientRouteInfoVo.getLongitude();
    BigDecimal latitudeParam = clientRouteInfoVo.getLatitude();
    Validate.notNull(longitudeParam, "当前站点没有配置位置（经度）信息");
    Validate.notNull(latitudeParam, "当前站点没有配置位置（纬度）信息");
    double distance = DistanceUtil.calculatePointDistance(latitude.doubleValue(), longitude.doubleValue(), latitudeParam.doubleValue(), longitudeParam.doubleValue());
    Validate.isTrue(BigDecimal.valueOf(distance).compareTo(BigDecimal.valueOf(stepFormVo.getDistance())) <= 0, "验证是否定位时，超过步骤限制距离：%s米", stepFormVo.getDistance());
  }
}
