package com.biz.crm.sfa.business.travel.local.service.internal;

import cn.hutool.core.date.DateUtil;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetails;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.travel.local.entity.TravelApplyEntity;
import com.biz.crm.sfa.business.travel.local.model.TravelApplyConditionModel;
import com.biz.crm.sfa.business.travel.local.repository.TravelApplyRepository;
import com.biz.crm.sfa.business.travel.local.service.TravelApplyService;
import com.biz.crm.sfa.business.travel.local.service.TravelApplyUserService;
import com.biz.crm.sfa.business.travel.sdk.constant.TravelConstant;
import com.biz.crm.sfa.business.travel.sdk.dto.TravelApplyDto;
import com.biz.crm.sfa.business.travel.sdk.dto.TravelApplyUserDto;
import com.biz.crm.workflow.sdk.constant.enums.ActApproveStatusEnum;
import com.biz.crm.workflow.sdk.dto.StartProcessDto;
import com.biz.crm.workflow.sdk.listener.ProcessListener;
import com.biz.crm.workflow.sdk.vo.response.CommitWorkflowResponse;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.Collections;
import java.util.Objects;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 出差申请表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-31 15:20:33
 */
@Slf4j
@Service("travelApplyService")
public class TravelApplyServiceImpl implements TravelApplyService {

  @Autowired private TravelApplyRepository travelApplyRepository;
  @Autowired private GenerateCodeService generateCodeService;
  @Autowired private LoginUserService loginUserService;
  @Autowired private NebulaToolkitService nebulaToolkitService;
  @Autowired private TravelApplyUserService travelApplyUserService;
  @Autowired private NebulaNetEventClient nebulaNetEventClient;
  @Autowired private UserVoService userVoService;
  @Autowired private UserFeignVoService userFeignVoService;

  @Override
  @Transactional
  public TravelApplyEntity create(TravelApplyDto dto) {
    this.createValidation(dto);
    this.validateApply(dto);
    TravelApplyEntity entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, TravelApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setApplyCode(this.generateCodeService.generateCode("TA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setProcessStatus(ActApproveStatusEnum.APPROVING.getCode());
    List<UserVo> userVoList =
        this.userFeignVoService.findByUserNames(Collections.singletonList(entity.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap =
          userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPostCode(userVo.getPositionCode());
      entity.setPostName(userVo.getPositionName());
    }
    this.travelApplyRepository.save(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    // 持久化出差人员信息
    this.travelApplyUserService.update(dto);
    entity.setProcessNumber(this.commitProcess(dto));
    this.travelApplyRepository.updateById(entity);
    return entity;
  }

  @Override
  @Transactional
  public TravelApplyEntity update(TravelApplyDto dto) {
    this.updateValidation(dto);
    this.validateApply(dto);
    TravelApplyEntity applyEntity = this.travelApplyRepository.getById(dto.getId());
    Validate.notNull(applyEntity, "出差申请记录不存在!");
    Validate.isTrue(
        ActApproveStatusEnum.REJECTED.getCode().equals(applyEntity.getProcessStatus())
            || ActApproveStatusEnum.INTERRUPT.getCode().equals(applyEntity.getProcessStatus()),
        "该申请不支持编辑");
    // 驳回状态直接创建新的申请,追回的使用原来的申请继续提交
    if (ActApproveStatusEnum.REJECTED.getCode().equals(applyEntity.getProcessStatus())) {
      dto.setId(null);
    }
    TravelApplyEntity entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, TravelApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setApplyCode(
        StringUtils.isNotBlank(entity.getId())
            ? applyEntity.getApplyCode()
            : this.generateCodeService.generateCode("TA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setProcessStatus(ActApproveStatusEnum.APPROVING.getCode());
    List<UserVo> userVoList =
        this.userFeignVoService.findByUserNames(Collections.singletonList(entity.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap =
          userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPostCode(userVo.getPositionCode());
      entity.setPostName(userVo.getPositionName());
    }
    this.travelApplyRepository.saveOrUpdate(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    // 持久化出差人员信息
    this.travelApplyUserService.update(dto);
    entity.setProcessNumber(this.commitProcess(dto));
    this.travelApplyRepository.updateById(entity);
    return entity;
  }

  /**
   * 加班申请工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 加班申请信息
   */
  private String commitProcess(TravelApplyDto dto) {
    StartProcessDto startProcessDto = new StartProcessDto();
    startProcessDto.setProcessKey(TravelConstant.TRAVEL_APPLY_PROCESS_KEY);
    startProcessDto.setBusinessId(dto.getId());
    startProcessDto.setBusinessNo(dto.getApplyCode());
    startProcessDto.setRemark(dto.getProcessRemark());
    startProcessDto.setFormType(TravelConstant.TRAVEL_APPLY_PROCESS_FORM_TYPE);
    startProcessDto.setProcessTitle(TravelConstant.TRAVEL_APPLY_PROCESS_NAME);
    startProcessDto.setMenuCode(dto.getCompetenceCode());
    Map<String, Object> maps = Maps.newHashMap();
    maps.put(
        "duration",
        DateUtil.betweenDay(
                DateUtil.parse(dto.getBeginTime()), DateUtil.parse(dto.getEndTime()), true)
            + 1);
    startProcessDto.setVariables(maps);
    SerializableBiConsumer<ProcessListener, StartProcessDto> sf = ProcessListener::onStartProcess;
    CommitWorkflowResponse response =
        (CommitWorkflowResponse)
            this.nebulaNetEventClient.directPublish(startProcessDto, ProcessListener.class, sf);
    Validate.isTrue(StringUtils.isNotEmpty(response.getProcessInstanceId()), "发起流程失败！");
    return response.getProcessInstanceId();
  }

  /**
   * 在创建travelApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(TravelApplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getTravelAddress(), "缺失出差地点");
    Validate.notBlank(dto.getTravelObjective(), "缺失出差目的");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getTravelUserList()), "缺失出差人员");
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setApplyDate(LocalDateTime.now().format(TravelConstant.YYYY_MM_DD_HH_MM_SS));
  }

  /**
   * 在修改travelApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(TravelApplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getTravelAddress(), "缺失出差地点");
    Validate.notBlank(dto.getTravelObjective(), "缺失出差目的");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getTravelUserList()), "缺失出差人员");
    LoginUserDetails loginDetails = this.loginUserService.getLoginDetails(LoginUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setApplyDate(LocalDateTime.now().format(TravelConstant.YYYY_MM_DD_HH_MM_SS));
  }

  /**
   * 校验申请信息
   *
   * @param dto 参数dto
   */
  private void validateApply(TravelApplyDto dto) {
    TravelApplyConditionModel model = new TravelApplyConditionModel();
    model.setBeginTime(dto.getBeginTime());
    model.setEndTime(dto.getEndTime());
    model.setTenantCode(dto.getTenantCode());
    model.setTravelUserNames(
        dto.getTravelUserList().stream()
            .map(TravelApplyUserDto::getUserName)
            .collect(Collectors.toList()));
    model.setExcludeProcessStatusList(
        Lists.newArrayList(
            ActApproveStatusEnum.REJECTED.getCode(), ActApproveStatusEnum.INTERRUPT.getCode()));
    List<TravelApplyEntity> applyEntities =
        this.travelApplyRepository.findByTravelApplyConditionModel(model);
    if (CollectionUtils.isEmpty(applyEntities)) {
      return;
    }
    // 排除自身之后再比较
    List<TravelApplyEntity> existsApplyList =
        applyEntities.stream()
            .filter(entity -> !entity.getId().equals(dto.getId()))
            .collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(existsApplyList)) {
      TravelApplyEntity applyEntity = existsApplyList.get(0);
      String existsUserName = applyEntity.getTravelUserList().get(0).getUserName();
      UserVo userVo =
          ObjectUtils.defaultIfNull(
              this.userVoService.findByUserName(existsUserName), new UserVo());
      String beginTime =
          applyEntity.getBeginTime().compareTo(dto.getBeginTime()) >= 0
              ? applyEntity.getBeginTime()
              : dto.getBeginTime();
      throw new IllegalArgumentException(
          String.format(
              "用户[%s-%s]已存在[%s]的出差申请记录", existsUserName, userVo.getFullName(), beginTime));
    }
  }
}
