package com.biz.crm.sfa.business.travel.local.service.internal;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.travel.local.entity.TravelApplyEntity;
import com.biz.crm.sfa.business.travel.local.model.TravelApplyConditionModel;
import com.biz.crm.sfa.business.travel.local.repository.TravelApplyRepository;
import com.biz.crm.sfa.business.travel.local.service.TravelApplyService;
import com.biz.crm.sfa.business.travel.local.service.TravelApplyUserService;
import com.biz.crm.sfa.business.travel.sdk.constant.TravelConstant;
import com.biz.crm.sfa.business.travel.sdk.dto.TravelApplyDto;
import com.biz.crm.sfa.business.travel.sdk.dto.TravelApplyUserDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 出差申请表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-31 15:20:33
 */
@Slf4j
@Service("travelApplyService")
public class TravelApplyServiceImpl implements TravelApplyService {

  @Autowired
  private TravelApplyRepository travelApplyRepository;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private TravelApplyUserService travelApplyUserService;
  @Autowired
  private UserVoService userVoService;
  @Autowired
  private UserFeignVoService userFeignVoService;
  @Autowired
  private ProcessBusinessService processBusinessService;
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;
  @Value("${workflow.process.key.travel}")
  private String processKey;
  @Override
  @Transactional
  public TravelApplyEntity create(TravelApplyDto dto) {
    this.createValidation(dto);
    this.validateApply(dto);
    TravelApplyEntity entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, TravelApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setApplyCode(this.generateCodeService.generateCode("TA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    List<UserVo> userVoList =
        this.userFeignVoService.findByUserNames(Collections.singletonList(entity.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap =
          userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPostCode(userVo.getPositionCode());
      entity.setPostName(userVo.getPositionName());
    }
    this.travelApplyRepository.save(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    // 持久化出差人员信息
    this.travelApplyUserService.update(dto);
    this.commitProcess(dto);
    return entity;
  }

  @Override
  @Transactional
  public TravelApplyEntity update(TravelApplyDto dto) {
    this.updateValidation(dto);
    this.validateApply(dto);
    TravelApplyEntity applyEntity = this.travelApplyRepository.getById(dto.getId());
    Validate.notNull(applyEntity, "出差申请记录不存在!");
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(TravelConstant.TRAVEL_APPLY_PROCESS_FORM_TYPE);
    businessMappingDto.setBusinessNo(applyEntity.getApplyCode());
    ProcessBusinessMappingVo businessMappingVo = this.processBusinessMappingService.findSignalByConditions(businessMappingDto);
    if (Objects.nonNull(businessMappingVo)) {
      Validate.isTrue(ProcessStatusEnum.REJECT.getDictCode().equals(businessMappingVo.getProcessStatus())
          || ProcessStatusEnum.RECOVER.getDictCode().equals(businessMappingVo.getProcessStatus()), "该审批状态申请不支持编辑");
      //驳回状态直接创建新的申请,追回的使用原来的申请继续提交
      if (ProcessStatusEnum.REJECT.getDictCode().equals(businessMappingVo.getProcessStatus())) {
        dto.setId(null);
      }
    }
    TravelApplyEntity entity =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, TravelApplyEntity.class, HashSet.class, ArrayList.class, "recordIds");
    entity.setApplyCode(
        StringUtils.isNotBlank(entity.getId())
            ? applyEntity.getApplyCode()
            : this.generateCodeService.generateCode("TA", 1).get(0));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    List<UserVo> userVoList =
        this.userFeignVoService.findByUserNames(Collections.singletonList(entity.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap =
          userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPostCode(userVo.getPositionCode());
      entity.setPostName(userVo.getPositionName());
    }
    this.travelApplyRepository.saveOrUpdate(entity);
    dto.setId(entity.getId());
    dto.setApplyCode(entity.getApplyCode());
    // 持久化出差人员信息
    this.travelApplyUserService.update(dto);
    this.commitProcess(dto);
    return entity;
  }

  /**
   * 加班申请工作流进行审批
   *
   * @param dto 加班申请信息
   */
  private void commitProcess(TravelApplyDto dto) {
    ProcessBusinessDto processBusiness = new ProcessBusinessDto();
    processBusiness.setBusinessNo(dto.getApplyCode());
    processBusiness.setBusinessCode(TravelConstant.TRAVEL_APPLY_PROCESS_FORM_TYPE);
    processBusiness.setProcessKey(processKey);
    processBusiness.setProcessTitle(TravelConstant.TRAVEL_APPLY_PROCESS_NAME);
    JSONObject extData = new JSONObject();
    extData.put("duration", DateUtil.betweenDay(
        DateUtil.parse(dto.getBeginTime()), DateUtil.parse(dto.getEndTime()), true)
        + 1);
    processBusiness.setExtData(extData.toJSONString());
    this.processBusinessService.processStart(processBusiness);
  }

  /**
   * 在创建travelApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(TravelApplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getTravelAddress(), "缺失出差地点");
    Validate.notBlank(dto.getTravelObjective(), "缺失出差目的");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getTravelUserList()), "缺失出差人员");
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setApplyDate(LocalDateTime.now().format(TravelConstant.YYYY_MM_DD_HH_MM_SS));
  }

  /**
   * 在修改travelApply模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(TravelApplyDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getBeginTime(), "缺失开始时间");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getTravelAddress(), "缺失出差地点");
    Validate.notBlank(dto.getTravelObjective(), "缺失出差目的");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getBeginTime()), "非法的开始时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(
        DateTimeValidateUtil.validateDate(dto.getEndTime()), "非法的结束时间格式，可用的格式：[yyyy-MM-dd]");
    Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) <= 0, "开始时间不能大于结束时间");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getTravelUserList()), "缺失出差人员");
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setApplyDate(LocalDateTime.now().format(TravelConstant.YYYY_MM_DD_HH_MM_SS));
  }

  /**
   * 校验申请信息
   *
   * @param dto 参数dto
   */
  private void validateApply(TravelApplyDto dto) {
    TravelApplyConditionModel model = new TravelApplyConditionModel();
    model.setBeginTime(dto.getBeginTime());
    model.setEndTime(dto.getEndTime());
    model.setTenantCode(dto.getTenantCode());
    model.setTravelUserNames(
        dto.getTravelUserList().stream()
            .map(TravelApplyUserDto::getUserName)
            .collect(Collectors.toList()));
    List<TravelApplyEntity> applyEntities =
        this.travelApplyRepository.findByTravelApplyConditionModel(model);
    if (CollectionUtils.isEmpty(applyEntities)) {
      return;
    }
    List<String> applyCodes = applyEntities.stream().map(TravelApplyEntity::getApplyCode).collect(Collectors.toList());
    ProcessBusinessMappingDto businessMappingDto = new ProcessBusinessMappingDto();
    businessMappingDto.setBusinessCode(TravelConstant.TRAVEL_APPLY_PROCESS_FORM_TYPE);
    businessMappingDto.setBusinessNos(applyCodes);
    List<ProcessBusinessMappingVo> businessMappingVos = this.processBusinessMappingService.findMultiByByConditions(businessMappingDto);
    Map<String, String> processStatusMap = CollectionUtils.isEmpty(businessMappingVos) ? Maps.newHashMap() : businessMappingVos.stream()
        .collect(Collectors.toMap(ProcessBusinessMappingVo::getBusinessNo, ProcessBusinessMappingVo::getProcessStatus, (a, b) -> b));
    // 排除自身剩下审批中或者审批通过的数据(没有找到流程记录说明没有审批流程,认为是有效数据)
    List<TravelApplyEntity> existsApplyList = applyEntities.stream()
        .filter(entity -> {
          String processStatus = processStatusMap.getOrDefault(entity.getApplyCode(), ProcessStatusEnum.PASS.getDictCode());
          return !entity.getId().equals(dto.getId()) && (ProcessStatusEnum.COMMIT.getDictCode().equals(processStatus) || ProcessStatusEnum.PASS.getDictCode().equals(processStatus));
        }).collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(existsApplyList)) {
      TravelApplyEntity applyEntity = existsApplyList.get(0);
      String existsUserName = applyEntity.getTravelUserList().get(0).getUserName();
      UserVo userVo = ObjectUtils.defaultIfNull(this.userVoService.findByUserName(existsUserName), new UserVo());
      String beginTime = applyEntity.getBeginTime().compareTo(dto.getBeginTime()) >= 0 ? applyEntity.getBeginTime() : dto.getBeginTime();
      throw new IllegalArgumentException(
          String.format(
              "用户[%s-%s]已存在[%s]的出差申请记录", existsUserName, userVo.getFullName(), beginTime));
    }
  }
}
