package com.biz.crm.sfa.business.visit.plan.line.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.sfa.visit.plan.line.sdk.constant.VisitPlanLineConstant;
import com.biz.crm.business.sfa.visit.plan.line.sdk.dto.VisitPlanLineQueryDto;
import com.biz.crm.sfa.business.visit.plan.line.local.entity.VisitPlanLine;
import com.biz.crm.sfa.business.visit.plan.line.local.entity.VisitPlanLineClientInfo;
import com.biz.crm.sfa.business.visit.plan.line.local.entity.VisitPlanLineRangeMapping;
import com.biz.crm.sfa.business.visit.plan.line.local.repository.VisitPlanLineRangeMappingRepository;
import com.biz.crm.sfa.business.visit.plan.line.local.repository.VisitPlanLineRepository;
import com.biz.crm.sfa.business.visit.plan.line.local.service.VisitPlanLineClientInfoService;
import com.biz.crm.sfa.business.visit.plan.line.local.service.VisitPlanLineService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 拜访计划线路组表(VisitPlanLine)表服务实现类
 *
 * @author songjingen
 * @since 2022-05-07 14:08:18
 */
@Service("visitPlanLineService")
public class VisitPlanLineServiceImpl implements VisitPlanLineService {

  @Autowired
  private VisitPlanLineRepository visitPlanLineRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private VisitPlanLineClientInfoService visitPlanLineClientInfoService;

  @Autowired
  private VisitPlanLineRangeMappingRepository visitPlanLineRangeMappingRepository;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      实体对象
   * @return
   */
  @Override
  public Page<VisitPlanLine> findByConditions(Pageable pageable, VisitPlanLineQueryDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new VisitPlanLineQueryDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    //todo 小程序获取数据需取上下文的人员账号 待完善
    return this.visitPlanLineRepository.findByConditions(pageable, dto);
  }

  /**
   * 新增数据
   *
   * @param visitPlanLine 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public VisitPlanLine create(VisitPlanLine visitPlanLine) {
    this.createValidate(visitPlanLine);
    this.visitPlanLineRepository.saveOrUpdate(visitPlanLine);
    //保存客户信息
    List<VisitPlanLineClientInfo> visitPlanLineClientInfos = visitPlanLine.getVisitPlanLineClientInfos();
    visitPlanLineClientInfos.stream().forEach(visitPlanLineClientInfo -> visitPlanLineClientInfo.setLineCode(visitPlanLine.getLineCode()));
    this.visitPlanLineClientInfoService.createBatch(visitPlanLineClientInfos);
    return visitPlanLine;
  }

  /**
   * 修改新据
   *
   * @param visitPlanLine 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public VisitPlanLine update(VisitPlanLine visitPlanLine) {
    Validate.notBlank(visitPlanLine.getId(), "修改时，主键不能为空！");
    VisitPlanLine planLine = this.visitPlanLineRepository.getById(visitPlanLine.getId());
    Validate.notNull(planLine, "修改时，未查询到线路组信息！");
    BeanUtils.copyProperties(visitPlanLine, planLine, "id", "lineCode", "tenantCode", "delFlag", "enableStatus", "createAccount", "createTime", "createName");
    this.updateValidate(planLine);
    this.visitPlanLineRepository.saveOrUpdate(visitPlanLine);
    //修改客户信息
    this.visitPlanLineClientInfoService.deleteByLineCode(planLine.getLineCode());
    List<VisitPlanLineClientInfo> visitPlanLineClientInfos = visitPlanLine.getVisitPlanLineClientInfos();
    visitPlanLineClientInfos.stream().forEach(visitPlanLineClientInfo -> visitPlanLineClientInfo.setLineCode(planLine.getLineCode()));
    this.visitPlanLineClientInfoService.createBatch(visitPlanLineClientInfos);
    return visitPlanLine;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    //验证是否存在和拜访计划绑定的关联关联，如果存在则不允许禁用
    List<VisitPlanLine> visitPlanLines = this.visitPlanLineRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanLines), "删除数据时，未查询到需要删除的数据！");
    List<String> lineCodes = visitPlanLines.stream().map(VisitPlanLine::getLineCode).collect(Collectors.toList());
    List<VisitPlanLineRangeMapping> rangeMappingList = this.visitPlanLineRangeMappingRepository.findByLineCodes(lineCodes);
    Validate.isTrue(CollectionUtils.isEmpty(rangeMappingList), "删除数据时，该线路组在拜访计划中，不允许删除！");
    this.visitPlanLineRepository.updateDelFlagByIds(idList, DelFlagStatusEnum.DELETE);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    this.visitPlanLineRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    //验证是否存在和拜访计划绑定的关联关联，如果存在则不允许禁用
    List<VisitPlanLine> visitPlanLines = this.visitPlanLineRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanLines), "禁用数据时，未查询到需要禁用的数据！");
    List<String> lineCodes = visitPlanLines.stream().map(VisitPlanLine::getLineCode).collect(Collectors.toList());
    List<VisitPlanLineRangeMapping> rangeMappingList = this.visitPlanLineRangeMappingRepository.findByLineCodes(lineCodes);
    Validate.isTrue(CollectionUtils.isEmpty(rangeMappingList), "禁用数据时，该线路组在拜访计划中，不允许停用！");
    this.visitPlanLineRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
  }

  /**
   * 创建验证
   *
   * @param visitPlanLine
   */
  private void createValidate(VisitPlanLine visitPlanLine) {
    Validate.notNull(visitPlanLine, "新增时，对象信息不能为空！");
    visitPlanLine.setId(null);
    visitPlanLine.setTenantCode(TenantUtils.getTenantCode());
    visitPlanLine.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    visitPlanLine.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    visitPlanLine.setLineCode(this.generateCodeService.generateCode(VisitPlanLineConstant.LINE_CODE, 1).get(0));
    Validate.notBlank(visitPlanLine.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(visitPlanLine.getLineCode(), "新增数据时，线路组编码不能为空！");
    Validate.notBlank(visitPlanLine.getLineName(), "新增数据时，线路组名称不能为空！");
    Validate.notBlank(visitPlanLine.getUserName(), "新增数据时，人员账号不能为空！");
    Validate.notBlank(visitPlanLine.getFullName(), "新增数据时，人员姓名不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanLine.getVisitPlanLineClientInfos()), "新增数据时，客户信息不能为空！");
    //校验一个人员账号不允许线路组名称重复
    List<VisitPlanLine> visitPlanLines = this.visitPlanLineRepository.findByUserNameAndLineName(visitPlanLine.getUserName(), visitPlanLine.getLineName());
    Validate.isTrue(CollectionUtils.isEmpty(visitPlanLines), "新增数据时，当前人员存在相同的线路组！");
  }

  /**
   * 修改验证
   *
   * @param visitPlanLine
   */
  private void updateValidate(VisitPlanLine visitPlanLine) {
    Validate.notNull(visitPlanLine, "修改数据时，对象信息不能为空！");
    Validate.notBlank(visitPlanLine.getId(), "修改数据时，不能为空！");
    Validate.notBlank(visitPlanLine.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(visitPlanLine.getLineCode(), "修改数据时，线路组编码不能为空！");
    Validate.notBlank(visitPlanLine.getLineName(), "修改数据时，线路组名称不能为空！");
    Validate.notBlank(visitPlanLine.getUserName(), "修改数据时，人员账号不能为空！");
    Validate.notBlank(visitPlanLine.getFullName(), "修改数据时，人员姓名不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanLine.getVisitPlanLineClientInfos()), "修改数据时，客户信息不能为空！");
    //校验一个人员账号不允许线路组名称重复
    List<VisitPlanLine> visitPlanLines = this.visitPlanLineRepository.findByUserNameAndLineName(visitPlanLine.getUserName(), visitPlanLine.getLineName());
    if (!CollectionUtils.isEmpty(visitPlanLines)) {
      visitPlanLines.forEach(visitPlanLine1 -> Validate.isTrue(visitPlanLine1.getId().equals(visitPlanLine.getId()), "修改数据时，当前人员存在相同的线路组！"));
      ;
    }

  }
}

