package com.biz.crm.sfa.business.visit.plan.local.service.internal;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.visit.plan.local.entity.VisitPlan;
import com.biz.crm.sfa.business.visit.plan.local.repository.VisitPlanRepository;
import com.biz.crm.sfa.business.visit.plan.local.service.VisitPlanDetailService;
import com.biz.crm.sfa.business.visit.plan.local.service.VisitPlanService;
import com.biz.crm.sfa.business.visit.plan.sdk.abstracts.VisitPlanRouteRangeAbstract;
import com.biz.crm.sfa.business.visit.plan.sdk.constant.VisitPlanConstant;
import com.biz.crm.sfa.business.visit.plan.sdk.dto.VisitPlanEventDto;
import com.biz.crm.sfa.business.visit.plan.sdk.dto.VisitPlanQueryDto;
import com.biz.crm.sfa.business.visit.plan.sdk.enums.VisitPlanTypeEnum;
import com.biz.crm.sfa.business.visit.plan.sdk.enums.VisitStatusEnum;
import com.biz.crm.sfa.business.visit.plan.sdk.model.VisitPlanRouteRangeModel;
import com.biz.crm.sfa.business.visit.plan.sdk.observer.VisitPlanDetailObserver;
import com.biz.crm.sfa.business.visit.plan.sdk.register.VisitPlanRouteRegister;
import com.biz.crm.sfa.business.visit.plan.sdk.strategy.VisitPlanRouteStrategy;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 拜访计划制定表(VisitPlan)表服务实现类
 *
 * @author songjingen
 * @since 2022-05-06 11:15:39
 */
@Service("visitPlanService")
public class VisitPlanServiceImpl implements VisitPlanService {

  @Autowired
  private VisitPlanRepository visitPlanRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private List<VisitPlanRouteRegister> visitPlanRouteRegisters;

  @Autowired(required = false)
  private List<VisitPlanRouteStrategy> visitPlanRouteStrategies;

  @Autowired(required = false)
  private List<VisitPlanDetailObserver> visitPlanDetailObservers;

  @Autowired
  private VisitPlanDetailService visitPlanDetailService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  /**
   * 分页查询数据
   *
   * @param pageable          分页对象
   * @param visitPlanQueryDto 实体对象
   * @return
   */
  @Override
  public Page<VisitPlan> findByConditions(Pageable pageable, VisitPlanQueryDto visitPlanQueryDto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(visitPlanQueryDto)) {
      visitPlanQueryDto = new VisitPlanQueryDto();
    }
    visitPlanQueryDto.setTenantCode(TenantUtils.getTenantCode());
    Page<VisitPlan> conditions = this.visitPlanRepository.findByConditions(pageable, visitPlanQueryDto);
    List<VisitPlan> records = conditions.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      return new Page<>();
    }
    return conditions;
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public VisitPlan findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    VisitPlan visitPlan = this.visitPlanRepository.getById(id);
    if (visitPlan == null) {
      return null;
    }
    //获取维度模块的拜访信息
    if (!CollectionUtils.isEmpty(visitPlanRouteStrategies)) {
      for (VisitPlanRouteStrategy visitPlanRouteStrategy : visitPlanRouteStrategies) {
        if (visitPlanRouteStrategy.getKey().equals(visitPlan.getVisitRouteType())) {
          VisitPlanRouteRangeAbstract visitPlanRouteRangeAbstract = visitPlanRouteStrategy.onFindRouteRange(visitPlan.getVisitPlanCode());
          visitPlan.setVisitPlanRouteRangeAbstract(visitPlanRouteRangeAbstract);
          break;
        }
      }
    }
    return visitPlan;
  }

  /**
   * 新增数据
   *
   * @param jsonObject 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public VisitPlan create(JSONObject jsonObject) {
    //解析拜访计划数据
    VisitPlan visitPlan = this.getVisitPlan(jsonObject);
    this.createValidate(visitPlan);
    jsonObject.put(VisitPlanConstant.VISIT_PLAN_CODE_FLAG, visitPlan.getVisitPlanCode());
    //通知维度模块创建信息
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanRouteStrategies), "进行创建计划时，未查询到维度策略实现！");
    for (VisitPlanRouteStrategy visitPlanRouteStrategy : visitPlanRouteStrategies) {
      if (StringUtils.equals(visitPlanRouteStrategy.getKey(), visitPlan.getVisitRouteType())) {
        VisitPlanRouteRangeModel visitPlanRouteRangeModel = visitPlanRouteStrategy.onCreate(jsonObject);
        visitPlan.setStartDate(visitPlanRouteRangeModel.getStartDate());
        visitPlan.setEndDate(visitPlanRouteRangeModel.getEndDate());
        break;
      }
    }
    //新增拜访计划数据
    this.visitPlanRepository.saveOrUpdate(visitPlan);
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanDetailObservers), "进行创建计划时，未查询到拜访计划明细观察者实现类！");
    Map<String, List<String>> listMap = new HashMap<>();
    listMap.put(visitPlan.getVisitRouteType(), Lists.newArrayList(visitPlan.getVisitPlanCode()));
    for (VisitPlanDetailObserver visitPlanDetailObserver : visitPlanDetailObservers) {
      visitPlanDetailObserver.onCreate(listMap);
    }
    //记录操作日志
//    VisitPlanEventDto visitPlanEventDto = new VisitPlanEventDto();
//    visitPlanEventDto.setOriginal(null);
//    visitPlanEventDto.setNewest(JsonUtils.toJSONObject(visitPlan));
//    SerializableBiConsumer<VisitPlanLogEventListener, VisitPlanEventDto> onCreate = VisitPlanLogEventListener::onCreate;
//    this.nebulaNetEventClient.publish(visitPlanEventDto, VisitPlanLogEventListener.class, onCreate);
    return visitPlan;
  }

  /**
   * 修改新据
   *
   * @param jsonObject 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public VisitPlan update(JSONObject jsonObject) {
    //解析拜访计划数据
    VisitPlan visitPlan = this.getVisitPlan(jsonObject);
    Validate.notBlank(visitPlan.getId(), "修改时，主键不能为空！");
    VisitPlan old = visitPlanRepository.getById(visitPlan.getId());
    Validate.notNull(old, "修改时，未查询到拜访计划信息！");
    VisitPlanEventDto visitPlanEventDto = new VisitPlanEventDto();
    visitPlanEventDto.setOriginal(JsonUtils.toJSONObject(old));
    BeanUtils.copyProperties(visitPlan, old, "id", "visitPlanCode", "tenantCode", "delFlag", "enableStatus", "createAccount", "createTime", "createName");
    this.updateValidate(old);
    jsonObject.put(VisitPlanConstant.VISIT_PLAN_CODE_FLAG, old.getVisitPlanCode());
    //通知维度模块创建信息
    Validate.notNull(visitPlanRouteStrategies, "进行创建计划时，未查询到维度策略实现！");
    for (VisitPlanRouteStrategy visitPlanRouteStrategy : visitPlanRouteStrategies) {
      if (StringUtils.equals(visitPlanRouteStrategy.getKey(), old.getVisitRouteType())) {
        VisitPlanRouteRangeModel visitPlanRouteRangeModel = visitPlanRouteStrategy.onUpdate(jsonObject);
        old.setStartDate(visitPlanRouteRangeModel.getStartDate());
        old.setEndDate(visitPlanRouteRangeModel.getEndDate());
        break;
      }
    }
    //修改拜访计划数据
    this.visitPlanRepository.saveOrUpdate(old);
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanDetailObservers), "进行创建计划时，未查询到拜访计划明细观察者实现类！");
    Map<String, List<String>> listMap = new HashMap<>();
    listMap.put(visitPlan.getVisitRouteType(), Lists.newArrayList(visitPlan.getVisitPlanCode()));
    for (VisitPlanDetailObserver visitPlanDetailObserver : visitPlanDetailObservers) {
      visitPlanDetailObserver.onUpdate(listMap);
    }
    //记录操作日志
//    visitPlanEventDto.setNewest(JsonUtils.toJSONObject(old));
//    SerializableBiConsumer<VisitPlanLogEventListener, VisitPlanEventDto> onUpdate = VisitPlanLogEventListener::onUpdate;
//    this.nebulaNetEventClient.publish(visitPlanEventDto, VisitPlanLogEventListener.class, onUpdate);
    return visitPlan;
  }


  /**
   * 解析出拜访计划信息
   *
   * @param jsonObject
   * @return
   */
  private VisitPlan getVisitPlan(JSONObject jsonObject) {
    Validate.notNull(jsonObject, "未获取到请求信息");
    // 获取拜访计划的配置
    JSONObject templateJson = jsonObject.getJSONObject(VisitPlanConstant.VISIT_PLAN_FLAG);
    Validate.notNull(jsonObject, "未获取到拜访计划信息");
    //校验拜访计划信息是否携带具体维度数据
    boolean hasRouteData = false;
    for (VisitPlanRouteRegister routeRegister : visitPlanRouteRegisters) {
      //循环维度绑定服务，获取对应维度的数据，获取到任一数据则将携带维度数据置为true，并退出循环
      Object policyData = jsonObject.get(routeRegister.getKey());
      if (Objects.nonNull(policyData)) {
        hasRouteData = true;
        break;
      }
    }
    Validate.isTrue(hasRouteData, "创建拜访计划时需要指定维度内容！");
    VisitPlan visitPlan = JsonUtils.json2Obj(templateJson.toJSONString(), VisitPlan.class);
    return visitPlan;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<VisitPlan> list = this.visitPlanRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(list), "删除数据时，未查询到数据！");
    this.visitPlanRepository.updateDelFlagByIds(idList, DelFlagStatusEnum.DELETE);
    //删除拜访计划明细数据
    Set<String> visitPlanCodes = list.stream().map(VisitPlan::getVisitPlanCode).collect(Collectors.toSet());
    this.visitPlanDetailService.deleteByVisitPlanCodesAndVisitStatus(visitPlanCodes, VisitStatusEnum.NOT_VISIT.getDictCode());
    //记录操作日志
//    VisitPlanEventDto visitPlanEventDto = new VisitPlanEventDto();
//    visitPlanEventDto.setOriginal(JsonUtils.toJSONObject(list));
//    visitPlanEventDto.setNewest(null);
//    SerializableBiConsumer<VisitPlanLogEventListener, VisitPlanEventDto> onDelete = VisitPlanLogEventListener::onDelete;
//    this.nebulaNetEventClient.publish(visitPlanEventDto, VisitPlanLogEventListener.class, onDelete);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    List<VisitPlan> list = this.visitPlanRepository.findByIds(idList);
    List<VisitPlan> disable = list.stream().filter(visitPlan -> EnableStatusEnum.DISABLE.getCode().equals(visitPlan.getEnableStatus())).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(disable), "启用数据时，未查询到要启用的数据！");
    this.visitPlanRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
    //同步更新拜访计划明细数据
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanDetailObservers), "进行创建计划时，未查询到拜访计划明细观察者实现类！");
    //根据拜访维度类型分组 k-维度类型，v-拜访计划编码集合
    Map<String, List<String>> map = list.stream().collect(Collectors.groupingBy(VisitPlan::getVisitRouteType, Collectors.mapping(VisitPlan::getVisitPlanCode, Collectors.toList())));
    for (VisitPlanDetailObserver visitPlanDetailObserver : visitPlanDetailObservers) {
      visitPlanDetailObserver.onUpdate(map);
    }
    //记录操作日志
//    VisitPlanEventDto visitPlanEventDto = new VisitPlanEventDto();
//    visitPlanEventDto.setOriginal(JsonUtils.toJSONObject(disable));
//    SerializableBiConsumer<VisitPlanLogEventListener, VisitPlanEventDto> onEnable = VisitPlanLogEventListener::onEnable;
//    this.nebulaNetEventClient.publish(visitPlanEventDto, VisitPlanLogEventListener.class, onEnable);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    List<VisitPlan> list = this.visitPlanRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(list), "启用数据时，未查询到数据！");
    this.visitPlanRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
    //删除拜访计划明细数据
    Set<String> visitPlanCodes = list.stream().map(VisitPlan::getVisitPlanCode).collect(Collectors.toSet());
    this.visitPlanDetailService.deleteByVisitPlanCodesAndVisitStatus(visitPlanCodes, VisitStatusEnum.NOT_VISIT.getDictCode());
    //记录操作日志
//    VisitPlanEventDto visitPlanEventDto = new VisitPlanEventDto();
//    visitPlanEventDto.setOriginal(JsonUtils.toJSONObject(list));
//    SerializableBiConsumer<VisitPlanLogEventListener, VisitPlanEventDto> onDisable = VisitPlanLogEventListener::onDisable;
//    this.nebulaNetEventClient.publish(visitPlanEventDto, VisitPlanLogEventListener.class, onDisable);
  }

  /**
   * 创建验证
   *
   * @param visitPlan
   */
  private void createValidate(VisitPlan visitPlan) {
    Validate.notNull(visitPlan, "新增时，对象信息不能为空！");
    visitPlan.setId(null);
    visitPlan.setVisitPlanCode(generateCodeService.generateCode(VisitPlanConstant.VISIT_PLAN_CODE, 1).get(0));
    visitPlan.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    visitPlan.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    visitPlan.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(visitPlan.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(visitPlan.getVisitPlanType(), "新增数据时，拜访计划类型不能为空！");
    VisitPlanTypeEnum visitPlanTypeEnum = VisitPlanTypeEnum.getByDictCode(visitPlan.getVisitPlanType());
    Validate.notNull(visitPlanTypeEnum, "新增数据时，拜访计划类型不存在！");
    Validate.notBlank(visitPlan.getVisitRouteType(), "新增数据时，拜访维度类型不能为空！");
    boolean hasRouteData = false;
    for (VisitPlanRouteRegister routeRegister : visitPlanRouteRegisters) {
      //循环维度绑定服务，获取对应维度的数据，获取到任一数据则将携带维度数据置为true，并退出循环
      if (StringUtils.equals(routeRegister.getKey(), visitPlan.getVisitRouteType())) {
        hasRouteData = true;
        break;
      }
    }
    Validate.notNull(hasRouteData, "新增数据时，拜访维度类型不存在！");
  }

  /**
   * 修改验证
   *
   * @param visitPlan
   */
  private void updateValidate(VisitPlan visitPlan) {
    Validate.notNull(visitPlan, "修改时，对象信息不能为空！");
    Validate.notBlank(visitPlan.getId(), "修改数据时，id不能为空！");
    Validate.notBlank(visitPlan.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(visitPlan.getVisitPlanType(), "修改数据时，拜访计划类型不能为空！");
    VisitPlanTypeEnum visitPlanTypeEnum = VisitPlanTypeEnum.getByDictCode(visitPlan.getVisitPlanType());
    Validate.notNull(visitPlanTypeEnum, "修改数据时，拜访计划类型不存在！");
    Validate.notBlank(visitPlan.getVisitRouteType(), "修改数据时，拜访维度类型不能为空！");
    boolean hasRouteData = false;
    for (VisitPlanRouteRegister routeRegister : visitPlanRouteRegisters) {
      //循环维度绑定服务，获取对应维度的数据，获取到任一数据则将携带维度数据置为true，并退出循环
      if (StringUtils.equals(routeRegister.getKey(), visitPlan.getVisitRouteType())) {
        hasRouteData = true;
        break;
      }
    }
    Validate.notNull(hasRouteData, "新增数据时，拜访维度类型不存在！");

  }
}

