package com.biz.crm.sfa.business.visit.plan.local.service.internal;


import com.biz.crm.sfa.business.visit.plan.local.entity.VisitPlanDetail;
import com.biz.crm.sfa.business.visit.plan.local.entity.VisitPlanDetailException;
import com.biz.crm.sfa.business.visit.plan.local.repository.VisitPlanDetailRepository;
import com.biz.crm.sfa.business.visit.plan.local.service.VisitPlanDetailExceptionService;
import com.biz.crm.sfa.business.visit.plan.local.service.VisitPlanDetailService;
import com.biz.crm.sfa.business.visit.plan.sdk.enums.VisitStatusEnum;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 拜访计划明细表(VisitPlanDetail)表服务实现类
 *
 * @author songjingen
 * @since 2022-05-13 13:59:26
 */
@Service("visitPlanDetailService")
public class VisitPlanDetailServiceImpl implements VisitPlanDetailService {

  @Autowired
  private VisitPlanDetailRepository visitPlanDetailRepository;
  @Autowired
  private VisitPlanDetailExceptionService visitPlanDetailExceptionService;

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.visitPlanDetailRepository.removeByIds(idList);
  }

  @Override
  public void deleteByVisitPlanCodesAndVisitStatus(Set<String> visitPlanCodes, String visitStatus) {
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanCodes), "删除数据时，拜访计划编码集合不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(visitPlanCodes), "删除数据时，拜访状态不能为空！");
    this.visitPlanDetailRepository.deleteByVisitPlanCodesAndVisitStatus(visitPlanCodes, visitStatus);
  }

  @Override
  public Map<String, Integer> findScheduleByUserNameAndVisitDate(String userName, Date visitDate) {
    if (StringUtils.isBlank(userName) || visitDate == null) {
      return new HashMap<>(0);
    }
    List<VisitPlanDetail> totalList = this.visitPlanDetailRepository.findByVisitDateAndUserName(visitDate, userName);
    if (CollectionUtils.isEmpty(totalList)) {
      return new HashMap<>(0);
    }
    //筛选已完成数据
    List<VisitPlanDetail> completedList = totalList.stream().filter(visitPlanDetail -> VisitStatusEnum.VISIT_TO_COMPLETE.getDictCode().equals(visitPlanDetail.getVisitStatus())).collect(Collectors.toList());
    HashMap<String, Integer> hashMap = new HashMap<>(2);
    hashMap.put("completed", CollectionUtils.isEmpty(completedList) ? 0 : completedList.size());
    hashMap.put("total", totalList.size());
    return hashMap;
  }

  @Override
  public List<VisitPlanDetail> findByVisitDateAndUserName(Date visitDate, String userName) {
    if (visitDate == null || StringUtils.isBlank(userName)) {
      return new ArrayList<>(0);
    }
    return this.visitPlanDetailRepository.findByVisitDateAndUserName(visitDate, userName);
  }

  @Override
  public void deleteByVisitPlanCodes(Set<String> visitPlanCodes) {
    this.visitPlanDetailRepository.deleteByVisitPlanCodes(visitPlanCodes);
  }


  @Override
  @Transactional
  @DynamicTaskService(cornExpression = "0 5 0 * * ?", taskDesc = "拜访计划定时任务")
  public void executeTask() {
    try {
      Date before = DateUtils.addDays(new Date(), -1);
      String beforeStr = DateFormatUtils.format(before, "yyyy-MM-dd");
      Date beforeDate = DateUtils.parseDate(beforeStr, "yyyy-MM-dd");
      List<VisitPlanDetail> list = this.visitPlanDetailRepository.findByVisitDateAndVisitStatus(beforeDate, VisitStatusEnum.IN_THE_VISIT.getDictCode());
      if (CollectionUtils.isEmpty(list)) {
        return;
      }
      ArrayList<VisitPlanDetailException> exceptions = new ArrayList<>();
      for (VisitPlanDetail visitPlanDetail : list) {
        visitPlanDetail.setVisitStatus(VisitStatusEnum.EXCEPTION.getDictCode());
        VisitPlanDetailException visitPlanDetailException = new VisitPlanDetailException();
        visitPlanDetailException.setVisitPlanDetailId(visitPlanDetail.getId());
        visitPlanDetailException.setRemark("计划已过期，更新为异常！");
        exceptions.add(visitPlanDetailException);
      }
      //更新拜访状态
      this.visitPlanDetailRepository.saveOrUpdateBatch(list);
      //增加异常记录
      this.visitPlanDetailExceptionService.createBatch(exceptions);
    } catch (ParseException e) {
      e.printStackTrace();
    }
  }

  @Override
  public List<VisitPlanDetail> findByClientCodesAndClientTypeAndVisitStatus(Set<String> clientCodes, String clientType, String visitStatus) {
    if (CollectionUtils.isEmpty(clientCodes) || StringUtils.isAnyBlank(clientType, visitStatus)) {
      return new ArrayList<>(0);
    }
    return this.visitPlanDetailRepository.findByClientCodesAndClientTypeAndVisitStatus(clientCodes, clientType, visitStatus);
  }

  @Override
  public VisitPlanDetail findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.visitPlanDetailRepository.getById(id);
  }

  @Override
  public List<VisitPlanDetail> findByClientCodesAndUserNameAndVisitDate(List<String> clientCodes, String userName, Date visitDate) {
    if (CollectionUtils.isEmpty(clientCodes) || StringUtils.isBlank(userName) || visitDate == null) {
      return new ArrayList<>(0);
    }
    return this.visitPlanDetailRepository.findByClientCodesAndUserNameAndVisitDate(clientCodes, userName, visitDate);
  }

}

