package com.biz.crm.sfa.business.visit.plan.rate.local.service.internal;


import cn.hutool.core.bean.BeanUtil;
import com.biz.crm.sfa.business.client.sdk.enums.ClientTypeEnum;
import com.biz.crm.sfa.business.client.sdk.strategy.ClientRouteStrategy;
import com.biz.crm.sfa.business.client.sdk.vo.ClientRouteInfoVo;
import com.biz.crm.sfa.business.visit.plan.rate.local.entity.VisitPlanRateClientInfo;
import com.biz.crm.sfa.business.visit.plan.rate.local.entity.VisitPlanRateRange;
import com.biz.crm.sfa.business.visit.plan.rate.local.repository.VisitPlanRateRangeRepository;
import com.biz.crm.sfa.business.visit.plan.rate.local.service.VisitPlanRateClientInfoService;
import com.biz.crm.sfa.business.visit.plan.rate.local.service.VisitPlanRateRangeService;
import com.biz.crm.sfa.business.visit.plan.rate.sdk.vo.VisitPlanRateClientInfoVo;
import com.biz.crm.sfa.business.visit.plan.rate.sdk.vo.VisitPlanRateRouteRangeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 频率拜访信息表(VisitPlanRateRange)表服务实现类
 *
 * @author songjingen
 * @since 2022-05-12 14:44:02
 */
@Service("visitPlanRateRangeService")
public class VisitPlanRateRangeServiceImpl implements VisitPlanRateRangeService {

  @Autowired
  private VisitPlanRateRangeRepository visitPlanRateRangeRepository;

  @Autowired
  private VisitPlanRateClientInfoService visitPlanRateClientInfoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private List<ClientRouteStrategy> visitPlanClientRouteStrategies;

  /**
   * 新增数据
   *
   * @param visitPlanRateRange 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public VisitPlanRateRange create(VisitPlanRateRange visitPlanRateRange) {
    this.createValidate(visitPlanRateRange);
    this.visitPlanRateRangeRepository.saveOrUpdate(visitPlanRateRange);
    //保存关联客户表
    Set<VisitPlanRateClientInfo> visitPlanOutletsClientInfos = visitPlanRateRange.getClientInfos();
    visitPlanOutletsClientInfos.stream().forEach(visitPlanOutletsClientInfo -> visitPlanOutletsClientInfo.setRangeId(visitPlanRateRange.getId()));
    this.visitPlanRateClientInfoService.createBatch(visitPlanOutletsClientInfos);
    return visitPlanRateRange;
  }

  @Transactional
  @Override
  public VisitPlanRateRange update(VisitPlanRateRange visitPlanRateRange) {
    Validate.notNull(visitPlanRateRange, "修改时，频率维度信息不能为空！");
    //删除数据
    VisitPlanRateRange planRateRange = this.visitPlanRateRangeRepository.findByVisitPlanCode(visitPlanRateRange.getVisitPlanCode());
    if (planRateRange != null) {
      try {
        String nowDateStr = DateFormatUtils.format(new Date(), "yyyy-MM-dd");
        Date nowDate = DateUtils.parseDate(nowDateStr, "yyyy-MM-dd");
        Validate.isTrue(visitPlanRateRange.getLoopsEndDate().after(nowDate), "修改时，循环结束时间小于当前时间不允许修改！");
      } catch (ParseException e) {
        e.printStackTrace();
      }
      this.visitPlanRateClientInfoService.deleteByRangeId(planRateRange.getId());
      this.visitPlanRateRangeRepository.deleteByVisitPlanCode(planRateRange.getVisitPlanCode());
    }
    //新增数据
    this.create(visitPlanRateRange);
    return visitPlanRateRange;
  }

  @Override
  public VisitPlanRateRouteRangeVo findByVisitPlanCode(String visitPlanCode) {
    if (StringUtils.isBlank(visitPlanCode)) {
      return null;
    }
    VisitPlanRateRange visitPlanRateRange = this.visitPlanRateRangeRepository.findByVisitPlanCode(visitPlanCode);
    if (visitPlanRateRange == null) {
      return null;
    }
    VisitPlanRateRouteRangeVo visitPlanRateRouteRangeVo = this.nebulaToolkitService.copyObjectByBlankList(visitPlanRateRange, VisitPlanRateRouteRangeVo.class, HashSet.class, ArrayList.class);
    List<VisitPlanRateClientInfo> clientInfos = this.visitPlanRateClientInfoService.findByRangeId(visitPlanRateRange.getId());
    List<VisitPlanRateClientInfoVo> visitPlanRateClientInfoVos = this.buildVisitPlanRateClientInfoVo(clientInfos);
    visitPlanRateRouteRangeVo.setClientInfos(visitPlanRateClientInfoVos);
    return visitPlanRateRouteRangeVo;
  }

  @Override
  public List<VisitPlanRateRange> findByVisitPlanCodes(List<String> visitPlanCodes) {
    if (CollectionUtils.isEmpty(visitPlanCodes)) {
      return new ArrayList<>(0);
    }
    return this.visitPlanRateRangeRepository.findByVisitPlanCodes(visitPlanCodes);
  }

  /**
   * 组装客户信息数据
   *
   * @param clientInfos
   * @return
   */
  private List<VisitPlanRateClientInfoVo> buildVisitPlanRateClientInfoVo(List<VisitPlanRateClientInfo> clientInfos) {
    if (CollectionUtils.isEmpty(clientInfos)) {
      return new ArrayList<>(0);
    }
    //Map<clientRoute, Map<clientType, List<clientCode>>>
    Map<String, Map<String, List<String>>> clientMap = clientInfos.stream()
        .collect(
            Collectors.groupingBy(VisitPlanRateClientInfo::getClientRoute,
                Collectors.groupingBy(VisitPlanRateClientInfo::getClientType, Collectors.mapping(VisitPlanRateClientInfo::getClientCode, Collectors.toList())
                )));
    //判断客户维度策略
    Map<String, List<ClientRouteInfoVo>> infoVoHashMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(visitPlanClientRouteStrategies)) {
      List<ClientRouteInfoVo> infoVos = new ArrayList<>();
      for (ClientRouteStrategy clientRouteStrategy : visitPlanClientRouteStrategies) {
        Map<String, List<String>> listMap = clientMap.get(clientRouteStrategy.getKey());
        if (listMap != null) {
          List<ClientRouteInfoVo> terminalVos = clientRouteStrategy.onFindByClientCodesAndType(listMap.get(ClientTypeEnum.TERMINAL.getDictCode()), ClientTypeEnum.TERMINAL.getDictCode());
          List<ClientRouteInfoVo> dealerVos = clientRouteStrategy.onFindByClientCodesAndType(listMap.get(ClientTypeEnum.DEALER.getDictCode()), ClientTypeEnum.DEALER.getDictCode());
          infoVos.addAll(terminalVos);
          infoVos.addAll(dealerVos);
        }
      }
      if (!CollectionUtils.isEmpty(infoVos)) {
        infoVoHashMap = infoVos.stream().collect(Collectors.groupingBy(o -> StringUtils.joinWith("_", o.getClientRoute(), o.getClientType(), o.getClientCode())));
      }
    }
    //循环赋值
    ArrayList<VisitPlanRateClientInfoVo> visitPlanRateClientInfoVos = new ArrayList<>();
    for (VisitPlanRateClientInfo visitPlanRateClientInfo : clientInfos) {
      VisitPlanRateClientInfoVo visitPlanRateClientInfoVo = this.nebulaToolkitService.copyObjectByBlankList(visitPlanRateClientInfo, VisitPlanRateClientInfoVo.class, HashSet.class, ArrayList.class);
      visitPlanRateClientInfoVos.add(visitPlanRateClientInfoVo);
      String key = StringUtils.joinWith("_", visitPlanRateClientInfo.getClientRoute(), visitPlanRateClientInfo.getClientType(), visitPlanRateClientInfo.getClientCode());
      List<ClientRouteInfoVo> infoVos = infoVoHashMap.get(key);
      if (CollectionUtils.isEmpty(infoVos)) {
        continue;
      }
      ClientRouteInfoVo vo = infoVos.get(0);
      BeanUtil.copyProperties(vo, visitPlanRateClientInfoVo);
      visitPlanRateClientInfoVo.setAddress(vo.getClientAddress());
    }
    return visitPlanRateClientInfoVos;
  }

  /**
   * 创建验证
   *
   * @param visitPlanRateRange
   */
  private void createValidate(VisitPlanRateRange visitPlanRateRange) {
    Validate.notNull(visitPlanRateRange, "新增时，对象信息不能为空！");
    visitPlanRateRange.setId(null);
    Date loopsEndDate = visitPlanRateRange.getLoopsEndDate();
    Date loopsStartDate = visitPlanRateRange.getLoopsStartDate();
    Set<VisitPlanRateClientInfo> clientInfos = visitPlanRateRange.getClientInfos();
    Validate.notNull(loopsEndDate, "新增数据时，循环结束日期不能为空！");
    Validate.notNull(loopsStartDate, "新增数据时，循环开始日期不能为空！");
    Validate.notBlank(visitPlanRateRange.getVisitPlanCode(), "新增数据时，拜访计划编码不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(clientInfos), "新增数据时，频率客户信息不能为空！");
    try {
      String nowDateStr = DateFormatUtils.format(new Date(), "yyyy-MM-dd");
      Date nowDate = DateUtils.parseDate(nowDateStr, "yyyy-MM-dd");
      Validate.isTrue(visitPlanRateRange.getLoopsStartDate().compareTo(nowDate) >= 0, "循环开始日期必须大于或者等于当前日期！");
      Validate.isTrue(visitPlanRateRange.getLoopsEndDate().compareTo(visitPlanRateRange.getLoopsStartDate()) >= 0, "循环开始日期必须小于或者等于循环结束日期！");
      clientInfos.stream().forEach(visitPlanRateClientInfo -> {
        Date firstVisitDate = visitPlanRateClientInfo.getFirstVisitDate();
        Validate.isTrue(firstVisitDate.compareTo(loopsStartDate) >= 0 && firstVisitDate.compareTo(loopsEndDate) <= 0, "首次拜访日期必须在循环日期内！");
      });
    } catch (ParseException e) {
      e.printStackTrace();
    }
  }

}

