package com.biz.crm.sfa.business.work.task.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.sfa.business.work.task.local.entity.WorkTaskReceve;
import com.biz.crm.sfa.business.work.task.local.repository.WorkTaskReceveRepository;
import com.biz.crm.sfa.business.work.task.local.service.WorkTaskPictureService;
import com.biz.crm.sfa.business.work.task.local.service.WorkTaskReceveService;
import com.biz.crm.sfa.business.work.task.local.service.WorkTaskService;
import com.biz.crm.sfa.business.work.task.sdk.dto.WorkTaskDto;
import com.biz.crm.sfa.business.work.task.sdk.dto.WorkTaskPictureDto;
import com.biz.crm.sfa.business.work.task.sdk.dto.WorkTaskReceveDto;
import com.biz.crm.sfa.business.work.task.sdk.enums.AttachmentBizTypeEnum;
import com.biz.crm.sfa.business.work.task.sdk.enums.SfaTaskReleaseEnum;
import com.biz.crm.sfa.business.work.task.sdk.enums.SfaTaskStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.Validate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import java.util.List;

/**
 * 工作计划接收表(WorkTaskReceve)serviceImpl
 *
 * @author rentao
 * @since 2022-06-13 14:05:00
 */
@Service
public class WorkTaskReceveServiceImpl implements WorkTaskReceveService {

  @Autowired private LoginUserService loginUserService;
  @Autowired private WorkTaskReceveRepository workTaskReceveRepository;
  @Autowired private WorkTaskPictureService workTaskPictureService;
  @Autowired private WorkTaskService workTaskService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;


  /**
   * 新增
   *
   * @param workTaskReceveDtos 新增参数
   *
   */
  @Override
  @Transactional
  public void createBatch(List<WorkTaskReceveDto> workTaskReceveDtos) {
    this.createValidate(workTaskReceveDtos);
    List<WorkTaskReceve> workTaskReceves =
        (List<WorkTaskReceve>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                workTaskReceveDtos,
                WorkTaskReceveDto.class,
                WorkTaskReceve.class,
                HashSet.class,
                ArrayList.class);
    this.workTaskReceveRepository.saveBatch(workTaskReceves);
  }

  /**
   * 编辑
   *
   * @param workTaskReceveDto 编辑参数
   *
   */
  @Override
  @Transactional
  public WorkTaskReceve update(WorkTaskReceveDto workTaskReceveDto) {
    this.updateValidate(workTaskReceveDto);
    WorkTaskReceve workTaskReceve = this.nebulaToolkitService.copyObjectByBlankList(workTaskReceveDto, WorkTaskReceve.class, HashSet.class, ArrayList.class);
    workTaskReceve.setReleaseDate(new Date());
    this.workTaskReceveRepository.updateById(workTaskReceve);
    // 更新主表的百分比和状态
    WorkTaskReceveDto workTaskReceveQueryDto = new WorkTaskReceveDto();
    workTaskReceveQueryDto.setTaskId(workTaskReceve.getTaskId());
    List<WorkTaskReceve> workTaskReceves = this.workTaskReceveRepository.findByConditions(workTaskReceveQueryDto);
    if (!CollectionUtils.isEmpty(workTaskReceves)) {
      String[] status = this.getStatus(workTaskReceves, workTaskReceve.getId(), workTaskReceve.getExecuteStatus());
      WorkTaskDto workTaskDto = new WorkTaskDto();
      workTaskDto.setId(workTaskReceveDto.getTaskId());
      workTaskDto.setProgress(status[0]);
      workTaskDto.setTaskStatus(status[1]);
      this.workTaskService.updateProgressAndTaskStatusById(workTaskDto);
    }
    List<WorkTaskPictureDto> pictures = workTaskReceveDto.getExecuteTaskPicList();
    if(!CollectionUtils.isEmpty(pictures)){
      pictures.forEach(workTaskPictureDto -> {
        workTaskPictureDto.setTaskId(workTaskReceve.getId());
        workTaskPictureDto.setTaskType(AttachmentBizTypeEnum.TASK_EXECUTE.getDictCode());
      });
      this.workTaskPictureService.createBatch(pictures);
    }
    return workTaskReceve;
  }

  /**
   * 任务定时过期策略
   */
  @Override
  @Transactional
  @DynamicTaskService(cornExpression = "0 0 0 * * ?", taskDesc = "任务定时过期策略")
  public void timeExpiredTask() {
    // 查询所有符合条件的数据
    WorkTaskReceveDto workTaskReceveQueryDto = new WorkTaskReceveDto();
    workTaskReceveQueryDto.setExecuteStatus(SfaTaskStatusEnum.WAIT_EXECUTE.getDictCode());
    workTaskReceveQueryDto.setEndDate(new Date());
    List<WorkTaskReceve> workTaskReceves = this.findByConditions(workTaskReceveQueryDto);
    if (!CollectionUtils.isEmpty(workTaskReceves)) {
      List<WorkTaskReceveDto> workTaskReceveDtos = (List<WorkTaskReceveDto>) this.nebulaToolkitService.copyCollectionByWhiteList(workTaskReceves, WorkTaskReceve.class
          , WorkTaskReceveDto.class, HashSet.class, ArrayList.class);
      workTaskReceveDtos.forEach(
          workTaskReceveDto -> {
            workTaskReceveDto.setExecuteStatus(SfaTaskStatusEnum.PAST_DUE.getDictCode());
            this.update(workTaskReceveDto);
          });
    }
  }

  /**
   * 条件查询部分也
   *
   * @param workTaskReceveDto
   *
   */
  @Override
  public List<WorkTaskReceve> findByConditions(WorkTaskReceveDto workTaskReceveDto) {
   return this.workTaskReceveRepository.findByConditions(workTaskReceveDto);
  }

  /**
   * 任务id查询 具体接收任务
   *
   * @param ids 新增参数
   *
   */
  @Override
  public List<WorkTaskReceve> findByTaskIds(Set<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return null;
    }
    return this.workTaskReceveRepository.findByTaskIds(ids);
  }

  /**
   * 新增验证
   *
   * @param workTaskReceveDtos
   */
  private void createValidate(List<WorkTaskReceveDto> workTaskReceveDtos) {
    Validate.isTrue(!CollectionUtils.isEmpty(workTaskReceveDtos), "新增时传入参数不能为空！");
    for (WorkTaskReceveDto sfaStaffMessageReqVo : workTaskReceveDtos) {
      Validate.notBlank(sfaStaffMessageReqVo.getReceverCode(), "任务执行人账号不能为空");
      Validate.notBlank(sfaStaffMessageReqVo.getReceverOrgCode(), "任务执行人组织不能为空");
      Validate.notBlank(sfaStaffMessageReqVo.getReceverOrgName(), "任务执行人组织名称不能为空");
      Validate.notBlank(sfaStaffMessageReqVo.getTaskId(), "任务执行任务id不能为空");
      Validate.notBlank(sfaStaffMessageReqVo.getReceverPosCode(), "任务执行人职位不能为空");
      Validate.notBlank(sfaStaffMessageReqVo.getReceverPosName(), "任务执行人职位名称不能为空");
      sfaStaffMessageReqVo.setTenantCode(TenantUtils.getTenantCode());
      sfaStaffMessageReqVo.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      sfaStaffMessageReqVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      sfaStaffMessageReqVo.setExecuteStatus(SfaTaskStatusEnum.WAIT_EXECUTE.getDictCode());
      sfaStaffMessageReqVo.setReceveDate(new Date());
    }
  }

  /**
   * 编辑验证
   *
   * @param workTaskReceveDto
   */
  private void updateValidate(WorkTaskReceveDto workTaskReceveDto) {
    Validate.notBlank(workTaskReceveDto.getId(), "编辑时，任务接收传入Id不能为空");
    Validate.notBlank(workTaskReceveDto.getTaskId(), "编辑时，任务接收发布Id不能为空");
    if (SfaTaskStatusEnum.EXECUTE_LODING
        .getDictCode()
        .equals(workTaskReceveDto.getExecuteStatus())) {
      Validate.notBlank(workTaskReceveDto.getSummary(), "编辑时，请输入执行总结");
    }
    Validate.notBlank(workTaskReceveDto.getExecuteStatus(), "编辑时，请输入执行状态结果");
  }

  /**
   * 接受任务 状态识别
   * @param receveRespVos 任务接收集合
   * @param id 本次修改id
   * @param exStatus 本次修改状态
   * @return
   */
  private  String[] getStatus(List<WorkTaskReceve> receveRespVos, String id, String exStatus) {
    //待执行
    int toPerformNum = 0;
    //已完成
    int offTheStocksNum = 0;
    //已拒绝
    int submittedNum = 0;
    //已过期
    int haveExpiredNum = 0;
    //[0]:已完成/总数 [1]:状态
    String[] list = new String[]{"", ""};
    //总任务数量
    int size = receveRespVos.size();
    Iterator<WorkTaskReceve> iterator = receveRespVos.iterator();
    while (iterator.hasNext()) {
      WorkTaskReceve next = iterator.next();
      if (id.equals(next.getId())) {
        next.setExecuteStatus(exStatus);
      }
      switch (next.getExecuteStatus()) {
        case "0":
          toPerformNum++;
          break;
        case "1":
          offTheStocksNum++;
          break;
        case "2":
          submittedNum++;
          break;
        case "3":
          haveExpiredNum++;
          break;
        default:
          break;
      }
    }
    //已完成
    boolean b = offTheStocksNum == size || (offTheStocksNum + submittedNum) == size;
    if (offTheStocksNum > 0 && b) {
      list[0] = size + "/" + size;
      list[1] = SfaTaskReleaseEnum.EXECUTE_LODING.getDictCode();
      return list;
    }
    //部分完成
    if (offTheStocksNum > 0 || submittedNum > 0) {
      list[0] = (offTheStocksNum+submittedNum) + "/" + size;
      list[1] = SfaTaskReleaseEnum.PARTLY_COMPLETED.getDictCode();
      return list;
    }
    //待执行
    if ((offTheStocksNum == 0 && toPerformNum > 0) && submittedNum == 0 ) {
      list[0] = submittedNum+"/" + size;
      list[1] = SfaTaskReleaseEnum.WAIT_EXECUTE.getDictCode();
      return list;
    }
    //已拒绝
    if (submittedNum == size) {
      list[0] = size+"/" + size;
      list[1] = SfaTaskReleaseEnum.ALREADY_FINISH.getDictCode();
      return list;
    }
    //已过期
    if (haveExpiredNum == size) {
      list[0] = "0/" + size;
      list[1] = SfaTaskReleaseEnum.PAST_DUE.getDictCode();
      return list;
    }
    //未完成
    if (offTheStocksNum == 0 && toPerformNum == 0) {
      list[0] = offTheStocksNum + "/" + size;
      list[1] = SfaTaskReleaseEnum.UNFINISHED.getDictCode();
      return list;
    }
    return list;
  }

}
