package com.biz.crm.tpm.business.activity.plan.table.local.controller;

import com.alibaba.excel.util.MapUtils;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.activity.plan.table.sdk.dto.ActivityPlanTableBatchSubmitApprovalDto;
import com.biz.crm.tpm.business.activity.plan.table.sdk.dto.ActivityPlanTableDto;
import com.biz.crm.tpm.business.activity.plan.table.sdk.service.ActivityPlanTableService;
import com.biz.crm.tpm.business.activity.plan.table.sdk.vo.*;
import com.biz.crm.tpm.business.subsidiary.activity.design.sdk.constant.SubComActivityDesignConstant;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * @author ：dengwei
 * @date ：Created in 2022/12/5 14:04
 * @description：活动计划套表
 */
@RestController
@RequestMapping("/v1/activityPlanTable")
@Slf4j
@Api(tags = "活动计划套表")
public class ActivityPlanTableController {

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private ActivityPlanTableService activityPlanTableService;

    @Autowired(required = false)
    private LoginUserService loginUserService;


    /**
     * 分页查询所有数据
     *
     * @param pageable     分页对象
     * @param dto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<ActivityPlanTableVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                              ActivityPlanTableDto dto) {
        try {
            Page<ActivityPlanTableVo> page = this.activityPlanTableService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("明细")
    @GetMapping("getById/{id}")
    public Result<ActivityPlanTableVo> getById(@PathVariable("id") String id){

        try {
            return Result.ok(this.activityPlanTableService.getById(id,null));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("创建")
    @PostMapping("save")
    public Result<?> save(@RequestBody ActivityPlanTableDto dto){
        try {
            this.activityPlanTableService.save(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "批量提交审批流")
    @PostMapping("batchSubmitApproval")
    public Result<?> batchSubmitApproval(@RequestBody ActivityPlanTableBatchSubmitApprovalDto dto){
        try {
            Assert.notNull(dto,"请选择数据!");
            Assert.notEmpty(dto.getIds(),"ID不能为空!");
            List<String> processNoList = this.activityPlanTableService.batchSubmitApproval(dto);
            activityPlanTableService.saveSubmitBatchByMq(processNoList);
            return Result.ok();
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 数据生成
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "数据生成")
    @PostMapping("generateData")
    public Result generateData(@ApiParam(name = "ids", value = "TPM-套表id") @RequestBody List<String> ids) {
        try {
            if (CollectionUtils.isEmpty(ids)) {
                throw new RuntimeException("请选择要操作的数据！");
            }
            ids.forEach(id -> {
                if (redisLockService.isLock(SubComActivityDesignConstant.LOCK_SUB_COM_ACTIVITY_DESIGN_APPROVE + id)) {
                    throw new RuntimeException("当前数据正在生成中，请勿重复操作");
                }
                this.activityPlanTableService.generateData(id);
            });
            return Result.ok("数据生成中，请稍后");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 下载活动规划套表
     * @param response
     * @throws IOException
     */
    @GetMapping("download")
    public void download(@RequestParam("id") String id,HttpServletResponse response) throws IOException {
        // 这里注意 有同学反应使用swagger 会导致各种问题，请直接用浏览器或者用postman
        try {

            ActivityPlanTableVo resultVo = this.activityPlanTableService.download(id,response);

        } catch (Exception e) {

            log.error("下载活动规划套表出错",e);
            // 重置response
            response.reset();
            response.setContentType("application/json");
            response.setCharacterEncoding("utf-8");
            Map<String, String> map = MapUtils.newHashMap();
            map.put("status", "failure");
            map.put("message", "下载文件失败" + e.getMessage());
            response.getWriter().println(JSON.toJSONString(map));
        }
    }


    @ApiOperation(value = "投入产出比分页查询")
    @GetMapping("findPutOutputRatioByConditions")
    public Result<Page<PutOutputRatioVo>> findPutOutputRatioByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                           @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<PutOutputRatioVo> page = this.activityPlanTableService.findPutOutputRatioByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation(value = "投入产出比客户分页查询")
    @GetMapping("findPutOutputRatioCustomerByConditions")
    public Result<Page<PutOutputRatioCustomerVo>> findPutOutputRatioCustomerByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                           @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<PutOutputRatioCustomerVo> page = this.activityPlanTableService.findPutOutputRatioCustomerByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "销售预测分页查询")
    @GetMapping("findSaleForecastByConditions")
    public Result<Page<SaleForecastVo>> findSaleForecastByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<SaleForecastVo> page = this.activityPlanTableService.findSaleForecastByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "费用汇总分页查询")
    @GetMapping("findCostCollectByConditions")
    public Result<Page<CostCollectVo>> findCostCollectByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                        @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<CostCollectVo> page = this.activityPlanTableService.findCostCollectByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "公投费用分页查询")
    @GetMapping("findReferendumCostByConditions")
    public Result<Page<ReferendumCostVo>> findReferendumCostByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<ReferendumCostVo> page = this.activityPlanTableService.findReferendumCostByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "固定支出分页查询")
    @GetMapping("findFinalFixedExpenseByConditions")
    public Result<Page<FinalFixedExpenseVo>> findFinalFixedExpenseByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<FinalFixedExpenseVo> page = this.activityPlanTableService.findFinalFixedExpenseByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "价格分页查询")
    @GetMapping("findPriceByConditions")
    public Result<Page<PriceVo>> findPriceByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Page<PriceVo> page = this.activityPlanTableService.findPriceByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "提交审批数据更新")
    @GetMapping("saveSubmitTimeData")
    public Result saveSubmitTimeData(@ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Assert.notNull(dto,"请选择数据!");
            Assert.hasLength(dto.getId(),"ID不能为空!");
            this.activityPlanTableService.saveSubmitTimeData(dto.getId());
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "提交审批数据更新")
    @GetMapping("saveSubmitTimeDataAsync")
    public Result saveSubmitTimeDataAsync(@ApiParam(name = "dto", value = "活动形式套表") ActivityPlanTableDto dto) {
        try {
            Assert.notNull(dto,"请选择数据!");
            Assert.hasLength(dto.getId(),"ID不能为空!");
            this.activityPlanTableService.saveSubmitBatchByMq(Collections.singletonList(dto.getProcessNo()));
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "查询自投预算预测剩余可用余额")
    @GetMapping("getAutoAmount")
    public Result<BigDecimal> getAutoAmount(@ApiParam(name = "yearAndMonth", value = "年月") @RequestParam("yearAndMonth") String yearAndMonth,
                                   @ApiParam(name = "orgCode", value = "组织编码") @RequestParam("orgCode")String orgCode) {
        try {
            BigDecimal autoAmount = this.activityPlanTableService.getAutoAmount(yearAndMonth, orgCode);
            return Result.ok(autoAmount);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

}
