package com.biz.crm.tpm.business.audit.formula.local.cache;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.page.cache.service.MnPageCacheHelper;
import com.biz.crm.tpm.business.variable.sdk.dto.ConVariableDetailDto;
import com.biz.crm.tpm.business.variable.sdk.service.ConVariableDetailSdkService;
import com.biz.crm.tpm.business.variable.sdk.vo.ConVariableDetailVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 可配置变量明细类型
 * @author wanghaojia
 * @date 2022/12/12 20:11
 */
@Slf4j
@Component
public class AuditFormulaItemCacheHelper extends MnPageCacheHelper<ConVariableDetailVo, ConVariableDetailDto> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ConVariableDetailSdkService conVariableDetailSdkService;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    /**
     * 缓存键前缀
     * @param
     * @return {@link String}
     */
    @Override
    public String getCacheKeyPrefix() {
        return "tpm_audit_formula_item_key:";
    }

    /**
     * Dto Class对象
     * @param
     * @return {@link Class}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public Class<ConVariableDetailDto> getDtoClass() {
        return ConVariableDetailDto.class;
    }

    /**
     * Vo Class对象
     * @param
     * @return {@link Class}<{@link BudgetCalConfigProductRatioVo}>
     */
    @Override
    public Class<ConVariableDetailVo> getVoClass() {
        return ConVariableDetailVo.class;
    }

    /**
     * 从数据库查询初始化数据
     * @param detailDto
     * @param cacheKey
     * @return {@link List}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public List<ConVariableDetailDto> findDtoListFromRepository(ConVariableDetailDto detailDto, String cacheKey) {
        try {
            Assert.notNull(detailDto, "参数不能为空");
            Assert.hasText(cacheKey, "缓存key不能为空");
            Assert.hasText(detailDto.getVariableCode(), "变量编码不能为空");
            if(!StringUtils.hasText(detailDto.getAuditFormulaCode())){
                return Lists.newArrayList();
            }
            Assert.hasText(detailDto.getItemType(), "类型不能为空");
            List<ConVariableDetailVo> detailVos = this.conVariableDetailSdkService.getItemDetail(detailDto);
            List<ConVariableDetailDto> detailDtos =
                (List<ConVariableDetailDto>) this.nebulaToolkitService
                    .copyCollectionByWhiteList(detailVos, ConVariableDetailVo.class,
                        ConVariableDetailDto.class, LinkedHashSet.class, ArrayList.class);
            if (!CollectionUtils.isEmpty(detailDtos)) {
                //获取缓存key
                String redisCacheIdKey = getRedisCacheIdKey(cacheKey);
                String redisCacheDataKey = getRedisCacheDataKey(cacheKey);
                //数据集合
                detailDtos.forEach(o -> {
                    o.setId(cacheKey + "-" + o.getItemCode());
                });
                Map<String, ConVariableDetailDto> detailDtoMap = detailDtos.stream()
                    .collect(Collectors.toMap(ConVariableDetailDto::getId, Function.identity()));
                //保存到缓存中
                redisService.lPushAll(redisCacheIdKey, this.getExpireTime(), detailDtoMap.keySet().toArray());
                redisTemplate.opsForHash().putAll(redisCacheDataKey, detailDtoMap);
                redisService.expire(redisCacheDataKey, this.getExpireTime());
            }
            return detailDtos;
        }catch (Exception e){
            String redisCacheInitKey = this.getRedisCacheInitKey(cacheKey);
            redisService.del(redisCacheInitKey);
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 是否初始化数据到缓存，默认为true
     * @param
     * @return {@link boolean}
     */
    @Override
    public boolean initToCacheFromRepository() {
        return false;
    }

    /**
     * 新增数据
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public List<ConVariableDetailDto> newItem(String cacheKey, List<ConVariableDetailDto> itemList) {
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        //获取已缓存的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        //过滤掉已经存在的数据
        List<ConVariableDetailDto> newList = itemList.stream()
            .filter(item -> !idList.contains(cacheKey + "-" + item.getItemCode())).collect(Collectors.toList());
        Assert.notEmpty(newList,"添加的数据全部重复");
        //获取缓存key
        newList.forEach(dto -> {
            dto.setId(cacheKey + "-" + dto.getItemCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setTenantCode(TenantUtils.getTenantCode());
        });
        return newList;
    }

    /**
     * 更新数据
     */
    public void updateItem(String cacheKey,List<ConVariableDetailDto> itemList){

    }

    /**
     * 复制数据
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public List<ConVariableDetailDto> copyItem(String cacheKey, List<ConVariableDetailDto> itemList) {
        return null;
    }

    /**
     * 获取Dto中的主键
     * @param detailDto
     * @return {@link Object}
     */
    @Override
    public Object getDtoKey(ConVariableDetailDto detailDto) {
        return detailDto.getId();
    }

    /**
     * 获取是否选中状态
     * @param detailDto
     * @return {@link String}
     */
    @Override
    public String getCheckedStatus(ConVariableDetailDto detailDto) {
        return detailDto.getChecked();
    }

    /**
     * 导入数据存入缓存
     * @param cacheKey
     * @param dto
     */
    public void importSave(String cacheKey, ConVariableDetailDto dto) {
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        dto.setId(cacheKey + "-" + dto.getItemCode());
        //获取已缓存的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        //判断是否已经存在
        if(!idList.contains(dto.getId())){
            //缓存ID
            Object[] newIdArr = {dto.getId()};
            redisService.lPushAll(redisCacheIdKey, this.getExpireTime(),newIdArr );
        }

        //导入新增数据
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());

        //缓存数据
        Map<Object, ConVariableDetailDto> ratioDtoMap = Maps.newHashMap();
        ratioDtoMap.put(dto.getId(), dto);

        redisTemplate.opsForHash().putAll(redisCacheDataKey, ratioDtoMap);
        redisService.expire(redisCacheDataKey, this.getExpireTime());
    }
}
