package com.biz.crm.tpm.business.audit.formula.local.imports;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.util.ExceptionStackMsgUtil;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.tpm.business.audit.formula.local.cache.AuditFormulaItemCacheHelper;
import com.biz.crm.tpm.business.variable.sdk.dto.ConVariableDetailDto;
import com.biz.crm.tpm.business.variable.sdk.enums.ConItemTypeEnum;
import com.biz.crm.tpm.business.variable.sdk.vo.ConVariableDetailImportVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @Description 配置明细导入
 * @Author YangWei
 * @Date 2023/2/2 下午4:17
 */
@Slf4j
@Component
public class AuditFormulaItemImportProcess implements ImportProcess<ConVariableDetailImportVo> {

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private DictDataVoService dictDataVoService;

    @Resource
    private ProductVoService productVoService;

    @Resource
    private CustomerVoService customerVoService;

    @Resource
    private AuditFormulaItemCacheHelper auditFormulaItemCacheHelper;

    private static final String TPM_CON_ITEM_TYPE = "tpm_con_item_type";

    /**
     * 数据字典
     */
    private Map<String, List<DictDataVo>> dictMap = Maps.newHashMap();

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<Integer, String> execute(LinkedHashMap<Integer, ConVariableDetailImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        final Optional<ConVariableDetailImportVo> first = data.values().stream().findFirst();
        if (!first.isPresent()) {
            return null;
        }
        ConVariableDetailImportVo vo = first.get();
        try {
            //初始数据字典
            //this.initDict();
            //保存
            String cacheKey = String.valueOf(params.get("cacheKey"));
            Assert.hasText(cacheKey,"cacheKey不能为空");
            String itemType = String.valueOf(params.get("itemType"));
            Assert.hasText(itemType,"类型[itemType]不能为空");
            //校验
            ConVariableDetailDto dto = this.validate(vo, itemType);
            this.auditFormulaItemCacheHelper.importSave(cacheKey, dto);
        } catch (IllegalArgumentException e) {
            throw e;
        } catch (Exception e) {
            String error = ExceptionStackMsgUtil.stackMsg(e);
            log.error("月度预算导入异常:{}", error);
            throw new IllegalArgumentException("导入异常[" + error + "]");
        }
        return null;
    }

    /**
     * 初始数据字典
     * @param
     */
    private void initDict() {
        this.dictMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(TPM_CON_ITEM_TYPE));
    }

    /**
     * 校验
     * @param vo
     * @return {@link MonthBudgetDto}
     */
    private ConVariableDetailDto validate(ConVariableDetailImportVo vo, String itemType) {
        //非空校验
        this.verifyEmpty(vo);
        //结果
        ConVariableDetailDto dto = new ConVariableDetailDto();
        dto.setItemCode(vo.getItemCode());
        //根据类型查询
        ConItemTypeEnum typeEnum = ConItemTypeEnum.findByCode(itemType);
        Assert.notNull(typeEnum, "类型错误");
        switch (typeEnum){
            case BRAND:
                break;
            case ITEM:
                break;
            case CATEGORY:
                break;
            case GOODS:
            case INCLUDE_GOODS:
                List<ProductVo> productList = this.productVoService.findByCodes(Lists.newArrayList(vo.getItemCode()));
                Assert.notEmpty(productList,"未查询到产品信息");
                ProductVo productVo = productList.get(0);
                dto.setItemName(productVo.getProductName());
                dto.setId(productVo.getId());
                dto.setPoint(vo.getPoint());
                break;
            case CUSTOMER:
                CustomerVo customerVo = this.customerVoService.findDetailsByIdOrCode(null, vo.getItemCode());
                Assert.notNull(customerVo, "未查询到客户信息");
                dto.setItemName(customerVo.getCustomerName());
                dto.setId(customerVo.getId());
                break;
            default:
                break;
        }

        return dto;
    }

    /**
     * 校验数据字典
     * @param vo
     */
    private void verifyDictCode(ConVariableDetailImportVo vo) {

    }

    /**
     * 非空校验
     * @param vo
     */
    private void verifyEmpty(ConVariableDetailImportVo vo) {
        Assert.hasText(vo.getItemCode(),"编码不能为空");
    }

    /**
     * 获取字典对应的value值
     *
     * @param dictValue
     * @param typeCode
     * @return
     */
    private String findDictValue(String dictValue, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictValue().equals(dictValue)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }

    /**
     * 获取字典对应的code值
     *
     * @param dictCode
     * @param typeCode
     * @return
     */
    private String findDictCode(String dictCode, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictCode().equals(dictCode)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }


    @Override
    public Integer getBatchCount() {
        //一次处理一条
        return 1;
    }

    @Override
    public Class<ConVariableDetailImportVo> findCrmExcelVoClass() {
        return ConVariableDetailImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_CON_VARIABLE_DETAIL_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-核销条件可配置变量明细类型导入模版";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_AUDIT_FORMULA_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-核销条件配置导入";
    }

}