package com.biz.crm.tpm.business.audit.formula.local.util;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.customer.sdk.enums.RtmModelEnum;
import com.biz.crm.tpm.business.audit.business.sdk.dto.AuditFormulaMainDto;
import com.biz.crm.tpm.business.audit.business.sdk.enums.RtmModelCodeEnum;
import com.biz.crm.tpm.business.audit.business.sdk.enums.TerminalChannelLevelEnum;
import com.biz.crm.tpm.business.audit.business.sdk.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.audit.business.sdk.vo.AuditFormulaInfoVo;
import com.biz.crm.tpm.business.audit.business.sdk.vo.AuditFormulaMainVo;
import com.biz.crm.tpm.business.audit.formula.local.entity.AuditFormulaChannel;
import com.biz.crm.tpm.business.audit.formula.local.entity.AuditFormulaInfo;
import com.biz.crm.tpm.business.audit.formula.local.entity.AuditFormulaMain;
import com.biz.crm.tpm.business.audit.formula.local.repository.AuditFormulaChannelRepository;
import com.biz.crm.tpm.business.audit.formula.local.repository.AuditFormulaInfoRepository;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-03-24
 */
@Slf4j
@Component
public class AuditFormulaUtil {

    @Autowired(required = false)
    private AuditFormulaChannelRepository auditFormulaChannelRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private AuditFormulaInfoRepository auditFormulaInfoRepository;

    /**
     * 2023-03-24 禅道：1022674
     * 活动细案预测查找制定核销公式逻辑：
     * 入参：业态、业务单元、核销类型为制度核销、销售组织（销售机构、销售部门、销售组）、客户类型、一级渠道、二级渠道、活动分类、活动形式；当活动分类是Z0001陈列时传【陈列批次】字段值，如果没有值默认传【常规】，当活动分类是Z0002人员时传【是否客户上账】字段值，如果没有值默认传【是】
     * 取核销条件：根据入参从核销条件查找状态为【启用】的核销条件
     * 1、优先按入参的全部参数精准匹配；
     * 2、全部入参没有匹配到条件时，优先级如下：
     * 0）去掉二级渠道入参匹配，没匹配到的执行1）
     * 1）去掉活动形式入参匹配，没匹配到的执行2）
     * 2）去掉活动形式和二级渠道入参进行匹配，没匹配到的执行2.2）
     * 2.2）去掉一级渠道入参进行匹配，没匹配到的执行3
     * 3）去掉活动形式、二级渠道、一级渠道入参进行匹配，没匹配到的执行4）
     * 4）去掉活动形式、二级渠道、一级渠道、销售组织入参进行匹配
     */
    /**
     * 主体匹配公式
     *
     * @param dto
     * @param list
     * @param orgParentMap
     * @return
     */
    public List<AuditFormulaMainVo> headQuarterFilter(AuditFormulaMainDto dto, List<AuditFormulaMain> list, Map<String, String> orgParentMap) {
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        //主体rtm模式过滤
        if (StringUtils.equals(RtmModelEnum.SON_COMPANY.getDictCode(), dto.getCustomerTypes())) {
            //分子公司模式只取分子公司模式的条件
            list = list.stream().filter(e -> StringUtils.isNotEmpty(e.getCustomerTypes()) && Arrays.asList(e.getCustomerTypes().split(",")).contains(RtmModelEnum.SON_COMPANY.getDictCode())).collect(Collectors.toList());
        } else {
            //其他的都取经销商的
            list = list.stream().filter(e -> StringUtils.isNotEmpty(e.getCustomerTypes()) && Arrays.asList(e.getCustomerTypes().split(",")).contains(RtmModelEnum.DEALER.getDictCode())).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        List<AuditFormulaMain> filterList = Lists.newArrayList();
        if (StringUtils.equals("Z0001", dto.getActivityTypeCode())) {
            //当活动分类是Z0001陈列时传【陈列批次】字段值，如果没有值默认传【常规】
            dto.setDisplayNumber(StringUtils.isNotEmpty(dto.getDisplayNumber()) ? dto.getDisplayNumber() : "1");
            dto.setCustomerAccount(null);
        } else if (StringUtils.equals("Z0002", dto.getActivityTypeCode())) {
            //当活动分类是Z0002人员时传【是否客户上账】字段值，如果没有值默认传【是】
            dto.setCustomerAccount(StringUtils.isNotEmpty(dto.getCustomerAccount()) ? dto.getCustomerAccount() : YesOrNoEnum.YES.getCode());
            dto.setDisplayNumber(null);
        } else {
            dto.setDisplayNumber(null);
            dto.setCustomerAccount(null);
        }
        if (StringUtils.isEmpty(dto.getFirstChannel())) {
            //一级渠道【如果一级渠道为空，则Q默认全渠道】
            dto.setFirstChannel("Q");
        }

        //渠道等级过滤
        List<String> auditFormulaCodes = list.stream().map(AuditFormulaMain::getAuditFormulaCode).collect(Collectors.toList());
        List<AuditFormulaChannel> channelList = this.auditFormulaChannelRepository.lambdaQuery()
                .eq(AuditFormulaChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(AuditFormulaChannel::getTenantCode, TenantUtils.getTenantCode())
                .in(AuditFormulaChannel::getAuditFormulaCode, auditFormulaCodes)
                .list();
        Map<String, Map<String, List<String>>> channelInfoMap = channelList.stream().collect(
                Collectors.groupingBy(AuditFormulaChannel::getAuditFormulaCode,
                        Collectors.groupingBy(AuditFormulaChannel::getChannelType,
                                Collectors.mapping(AuditFormulaChannel::getChannelCode, Collectors.toList()))));
        log.info("核销条件匹配=====》核销公式======》{}", JSONObject.toJSON(list));
        log.info("核销条件匹配=====》入参======》{}", JSONObject.toJSON(dto));
        log.info("核销条件匹配=====》渠道======》{}", JSONObject.toJSON(channelInfoMap));
        log.info("核销条件匹配=====》组织======》{}", JSONObject.toJSON(orgParentMap));
        //第一优先级，全条件：组织+陈列批次+是否客户上账+一级渠道+二级渠道+活动形式
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, true, true, true, true, true, true);
        log.info("核销条件匹配=====》第1优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        //第二优先级，部分条件：组织+陈列批次+是否客户上账+一级渠道+活动形式
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, true, true, true, true, false, true);
        log.info("核销条件匹配=====》第2优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        //第三优先级，部分条件：组织+陈列批次+是否客户上账+一级渠道+二级渠道
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, true, true, true, true, true, false);
        log.info("核销条件匹配=====》第3优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        //第四优先级，部分条件：组织+陈列批次+是否客户上账+一级渠道
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, true, true, true, true, false, false);
        log.info("核销条件匹配=====》第4优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        //第五优先级，部分条件：组织+陈列批次+是否客户上账+二级渠道+活动形式
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, true, true, true, false, true, true);
        log.info("核销条件匹配=====》第5优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        //第六优先级，部分条件：组织+陈列批次+是否客户上账
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, true, true, true, false, false, false);
        log.info("核销条件匹配=====》第6优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        //第七优先级，部分条件：陈列批次+是否客户上账
        filterList = this.filter(dto, list, filterList, channelInfoMap, orgParentMap, false, true, true, false, false, false);
        log.info("核销条件匹配=====》第7优先级=====>{}=====>{}", filterList.size(), JSONObject.toJSONString(filterList));
        return this.buildResult(filterList);
    }

    /**
     * 构建公式详细返回信息
     *
     * @param filterList
     * @return
     */
    public List<AuditFormulaMainVo> buildResult(List<AuditFormulaMain> filterList) {
        if (CollectionUtil.isNotEmpty(filterList)) {
            Collection<AuditFormulaMainVo> mainVos = this.nebulaToolkitService.copyCollectionByBlankList(filterList, AuditFormulaMain.class, AuditFormulaMainVo.class, LinkedHashSet.class, ArrayList.class);
            List<String> codes = mainVos.stream().map(AuditFormulaMainVo::getAuditFormulaCode).collect(Collectors.toList());
            List<AuditFormulaInfo> infos = this.auditFormulaInfoRepository.lambdaQuery().in(AuditFormulaInfo::getAuditFormulaCode, codes).list();
            Map<String, List<AuditFormulaInfo>> listMap = infos.stream().collect(Collectors.groupingBy(AuditFormulaInfo::getAuditFormulaCode));
            for (AuditFormulaMainVo mainVo : mainVos) {
                List<AuditFormulaInfo> infoList = listMap.getOrDefault(mainVo.getAuditFormulaCode(), Lists.newArrayList());
                Collection<AuditFormulaInfoVo> auditFormulaInfoVos = this.nebulaToolkitService.copyCollectionByBlankList(infoList, AuditFormulaInfo.class, AuditFormulaInfoVo.class, LinkedHashSet.class, ArrayList.class);
                mainVo.setAuditFormulaInfoVoList((List<AuditFormulaInfoVo>) auditFormulaInfoVos);
            }
            return (List<AuditFormulaMainVo>) mainVos;
        }
        return Lists.newArrayList();
    }

    /**
     * 根据传入的状态，逐级过滤
     * 销售组织是最高优先级，调整时需注意
     * filterList 为空时才进行过滤，不为空时直接返回
     *
     * @param dto
     * @param list
     * @param filterList
     * @param channelInfoMap
     * @param orgParentMap
     * @param saleOrg          是否包含组织过滤
     * @param displayNumber    是否包含陈列批次过滤
     * @param customerAccount  是否包含是否客户上账过滤
     * @param firstChannel     是否包含一级渠道过滤
     * @param secondChannel    是否包含二级渠道过滤
     * @param activityFormCode 是否包含活动形式过滤
     * @return
     */
    public List<AuditFormulaMain> filter(AuditFormulaMainDto dto, List<AuditFormulaMain> list, List<AuditFormulaMain> filterList,
                                         Map<String, Map<String, List<String>>> channelInfoMap, Map<String, String> orgParentMap,
                                         boolean saleOrg, boolean displayNumber, boolean customerAccount, boolean firstChannel,
                                         boolean secondChannel, boolean activityFormCode) {
        if (CollectionUtils.isEmpty(filterList)) {
            if (!saleOrg) {
                filterList = list.stream().filter(e ->
                        (!displayNumber || this.containsFilter(dto.getDisplayNumber(), e.getDisplayNumber()))
                                && (!customerAccount || this.containsFilter(dto.getCustomerAccount(), e.getCustomerAccount())))
                        .collect(Collectors.toList());
            }
            if (saleOrg) {
                //组织需要递归往上找
                AtomicReference<String> orgCode = new AtomicReference<>(dto.getSalesOrgCodes());
                int deep = 5;
                while (StringUtils.isNotEmpty(orgCode.get()) && CollectionUtils.isEmpty(filterList) && deep > 0) {
                    deep--;
                    filterList = list.stream().filter(e ->
                            StringUtils.isNotEmpty(e.getSalesOrgCodes())
                                    && Arrays.asList(e.getSalesOrgCodes().split(",")).contains(orgCode.get())
                                    && (!displayNumber || this.containsFilter(dto.getDisplayNumber(), e.getDisplayNumber()))
                                    && (!customerAccount || this.containsFilter(dto.getCustomerAccount(), e.getCustomerAccount()))
                                    && (!activityFormCode || this.containsFilter(dto.getActivityFormCode(), e.getActivityFormCode())))
                            .collect(Collectors.toList());
                    if (firstChannel) {
                        filterList = this.channelFilter(filterList, TerminalChannelLevelEnum.FIRST, channelInfoMap, true, dto.getFirstChannel());
                    }
                    if (secondChannel) {
                        filterList = this.channelFilter(filterList, TerminalChannelLevelEnum.SECOND, channelInfoMap, true, dto.getSecondChannel());
                    }
                    orgCode.set(orgParentMap.get(orgCode.get()));
                }
            }
        }
        return filterList;
    }

    /**
     * 渠道过滤
     *
     * @param filterList     数据集
     * @param levelEnum      渠道等级
     * @param channelInfoMap 核销条件关联渠道信息
     * @param matchFlag      是否需要匹配
     * @param value          渠道编码
     * @return
     */
    public List<AuditFormulaMain> channelFilter(List<AuditFormulaMain> filterList, TerminalChannelLevelEnum levelEnum,
                                                Map<String, Map<String, List<String>>> channelInfoMap, boolean matchFlag,
                                                String value) {
        if (CollectionUtils.isEmpty(filterList)) {
            return filterList;
        }
        return filterList.stream().filter(e -> {
            if (!channelInfoMap.containsKey(e.getAuditFormulaCode())) {
                return !matchFlag;
            }
            Map<String, List<String>> channelTypeMap = channelInfoMap.get(e.getAuditFormulaCode());
            if (!channelTypeMap.containsKey(levelEnum.getCode())) {
                return !matchFlag;
            }
            List<String> channelCodes = channelTypeMap.get(levelEnum.getCode());
            if (StringUtils.isEmpty(value)) {
                return CollectionUtils.isEmpty(channelCodes);
            }
            if (!channelCodes.contains(value)) {
                return !matchFlag;
            }
            return matchFlag;
        }).collect(Collectors.toList());
    }

    /**
     * 多选包含过滤
     *
     * @param param 入参
     * @param value 目标
     * @return
     */
    public boolean containsFilter(String param, String value) {
        if (StringUtils.isEmpty(param) && StringUtils.isEmpty(value)) {
            return true;
        } else if (StringUtils.isNotEmpty(param) && StringUtils.isNotEmpty(value)) {
            return Arrays.asList(value.split(",")).contains(param);
        }
        return false;
    }
}
