package com.biz.crm.tpm.business.audit.handle.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.audit.handle.sdk.constant.AuditHandleConstant;
import com.biz.crm.tpm.business.audit.handle.sdk.dto.AuditHandleDetailDto;
import com.biz.crm.tpm.business.audit.handle.sdk.dto.AuditHandleDto;
import com.biz.crm.tpm.business.audit.handle.sdk.dto.AuditHandleProcessSubmitDto;
import com.biz.crm.tpm.business.audit.handle.sdk.service.AuditHandleSdkService;
import com.biz.crm.tpm.business.audit.handle.sdk.vo.AuditHandleDetailVo;
import com.biz.crm.tpm.business.audit.handle.sdk.vo.AuditHandleVo;
import com.biz.crm.tpm.business.audit.handle.sdk.vo.AuditPrintVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @ClassName AuditHandleController
 * @Description
 * @AUTHOR WangJJ
 * @DATE 2023/5/30 17:10
 **/
@RequestMapping("/v1/audit/handle")
@RestController
@Slf4j
@Api(tags = "手动上账controller")
public class AuditHandleController {

    @Autowired(required = false)
    private AuditHandleSdkService auditHandleSdkService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 新增
     * @param auditHandleDto
     * @return
     */
    @PostMapping
    @ApiOperation("新增")
    public Result<AuditHandleVo> create(@RequestBody AuditHandleDto auditHandleDto){
        try {
            AuditHandleVo auditHandleVo = this.auditHandleSdkService.create(auditHandleDto);
            return Result.ok(auditHandleVo);
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 编辑
     * @param auditHandleDto
     * @return
     */
    @PatchMapping
    @ApiOperation("编辑")
    public Result<AuditHandleVo> update(@RequestBody AuditHandleDto auditHandleDto){
        boolean isLock = false;
        String key = StringUtils.EMPTY;
        try {
            Validate.notBlank(auditHandleDto.getId(),"主键ID不能为空");
            key = AuditHandleConstant.REDIS_UPDATE_LOCK_KEY_PREFIX + auditHandleDto.getId();
            isLock = redisLockService.tryLock(key, TimeUnit.SECONDS, 3600);
            if (!isLock){
                throw new RuntimeException("加锁异常");
            }
            AuditHandleVo auditHandleVo = this.auditHandleSdkService.update(auditHandleDto);
            return Result.ok(auditHandleVo);
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }finally {
            if (isLock){
                this.redisLockService.unlock(key);
            }
        }
    }

    /**
     * 根据ID获取详情
     * @param id
     * @return
     */
    @GetMapping("findById")
    @ApiOperation("根据ID获取详情")
    public Result<AuditHandleVo> findById(@ApiParam(name = "id", value = "主键ID") String id){
        try {
            AuditHandleVo handleVo = this.auditHandleSdkService.findById(id);
            return Result.ok(handleVo);
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据ID获取详情
     * @param auditHandleCode
     * @return
     */
    @GetMapping("findByAuditHandleCode")
    @ApiOperation("根据手动核销编码获取详情")
    public Result<AuditHandleVo> findByAuditHandleCode(@ApiParam(name = "auditHandleCode", value = "手动核销编码") String auditHandleCode){
        try {
            AuditHandleVo handleVo = this.auditHandleSdkService.findByAuditHandleCode(auditHandleCode);
            return Result.ok(handleVo);
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据核销单编码删除手动核销单
     * @param auditHandleCodes
     * @return
     */
    @DeleteMapping
    @ApiOperation("删除手动上账核销单")
    public Result<?> delete(@RequestParam("ids") List<String> ids){
        try {
            this.auditHandleSdkService.delete(ids);
            return Result.ok();
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交流程,多选批量提交
     */
    @ApiOperation(value = "提交流程,多选批量提交")
    @PostMapping(value = "submitApproval")
    public Result submitApproval(@ApiParam(value = "提交流程,多选批量提交")@RequestBody AuditHandleProcessSubmitDto dto){
        try {
            this.auditHandleSdkService.submitApprove(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 上账
     */
    @ApiOperation(value = "上账")
    @PostMapping(value = "upAccountBatch")
    public Result upAccountBatch(@RequestBody List<String> ids){
        boolean hasLock = false;
        try {
            hasLock = redisLockService.batchLock(AuditHandleConstant.AUDIT_HANDLE_UP_ACCOUNT_BATCH,ids,TimeUnit.MINUTES,30);
            if (!hasLock) {
                throw new RuntimeException("正在上账中,请勿重复推送");
            }
            this.auditHandleSdkService.upAccountBatch(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }finally {
            if(hasLock) {
                redisLockService.batchUnLock(AuditHandleConstant.AUDIT_HANDLE_UP_ACCOUNT_BATCH, ids);
            }
        }
    }

    @ApiOperation("手动上账报销冲销")
    @PostMapping("reimburseChargeAgainst")
    public Result<?> reimburseChargeAgainst(@RequestBody List<String> ids){

        boolean hasLock = false;
        try {

            hasLock = redisLockService.batchLock(AuditHandleConstant.AUDIT_REIMBURSE_CHARGE_AGAINST,ids,TimeUnit.MINUTES,30);
            Validate.isTrue(hasLock,"正在手动上账报销冲销中,请勿重复推送");
            this.auditHandleSdkService.reimburseChargeAgainst(ids);
            return Result.ok();
        }catch (Exception e){
            log.error("手动上账报销冲销失败：",e);
            return Result.error(e.getMessage());
        }finally {
            if(hasLock){
                redisLockService.batchUnLock(AuditHandleConstant.AUDIT_REIMBURSE_CHARGE_AGAINST, ids);
            }
        }
    }

    @ApiOperation("批量打印")
    @PostMapping("batchPrint")
    public Result<List<AuditPrintVo>> batchPrint(@RequestBody List<String> ids){
        return Result.ok(this.auditHandleSdkService.batchPrint(ids));
    }



    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<AuditHandleDetailVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                               AuditHandleDetailDto dto) {
        try {
            Page<AuditHandleDetailVo> page = this.auditHandleSdkService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


}
