package com.biz.crm.tpm.business.budget.controls.config.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.budget.controls.config.local.entity.BudgetProject;
import com.biz.crm.tpm.business.budget.controls.config.local.mapper.BudgetProjectMapper;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.BudgetProjectDto;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.BudgetProjectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;


/**
 * (BudgetItemRepository)表数据库访问层
 *
 * @author makejava
 * @since 2022-11-03 17:06:39
 */

@Component
public class BudgetProjectRepository extends ServiceImpl<BudgetProjectMapper, BudgetProject> {

    @Autowired
    private BudgetProjectMapper budgetProjectMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 通过controlsConfigCode查询选择预算项目
     */
    public List<BudgetProject> getcontrolsConfigCode(String delFlag) {
        return this.lambdaQuery().eq(BudgetProject::getDelFlag, delFlag).list();
    }


    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto)     查询实体
     * @return 所有数据
     */
    public Page<BudgetProjectVo> findByConditions(Pageable pageable, BudgetProjectDto dto) {
        if (pageable == null) {
            pageable = PageRequest.of(1, 50);
        }
        Page<BudgetProjectVo> page = new Page<BudgetProjectVo>(pageable.getPageNumber(), pageable.getPageSize());
        return budgetProjectMapper.findByConditions(page, dto);
    }

    /**
     * 获取选择预算项目集合
     */
    public List<BudgetProject> findBycontrolsConfigCode(String controlsConfigCode) {
        return this.lambdaQuery().eq(BudgetProject::getControlsConfigCode, controlsConfigCode).list();
    }


    /**
     * 通过基本信息查询预算项目接口
     *
     * @param controlsConfigCode 管控配置编码
     */
    public void deleteByControlsConfigCode(String controlsConfigCode) {
        if (StringUtils.isEmpty(controlsConfigCode)) {
            return;
        }
        List<BudgetProject> list = this.lambdaQuery().eq(BudgetProject::getControlsConfigCode, controlsConfigCode).list();
        if (!org.springframework.util.CollectionUtils.isEmpty(list)) {
            list.forEach(budgetProject -> {
                budgetProject.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            });
            this.updateBatchById(list);
        }
    }

    /**
     * 选择规则制定表详情
     */
    public List<BudgetProject> finByByControlsConfigCodeAndDelFlag(String delFlag, String controlsConfigCode) {
        if (StringUtils.isEmpty(delFlag)) {
            return null;
        }
        return this.lambdaQuery().eq(BudgetProject::getDelFlag, delFlag).eq(BudgetProject::getControlsConfigCode, controlsConfigCode).list();
    }

    /**
     * 批量查询
     *
     * @author huojia
     * @date 2022/12/27 16:29
     * @param controlsConfigCodeList
     * @return java.util.List<com.biz.crm.tpm.business.budget.controls.config.local.entity.BudgetProject>
     **/
    public List<BudgetProject> listByControlsConfigCodeList(List<String> controlsConfigCodeList) {
        if (CollectionUtils.isEmpty(controlsConfigCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(BudgetProject::getControlsConfigCode, controlsConfigCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据管控编码和预算项目编码查询数据
     *
     * @param controlsCodes 管控编码列表
     * @param codes         预算项目编码列表
     * @return List<BudgetProjectService>
     **/
    public List<BudgetProject> getByConfigCodesAndItemCodes(List<String> controlsCodes, List<String> codes) {
        Validate.notEmpty(controlsCodes, "预算管控编码列表为空");
        Validate.notEmpty(codes, "预算项目编码列表为空");
        return this.lambdaQuery()
                .eq(BudgetProject::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(BudgetProject::getControlsConfigCode, controlsCodes)
                .in(BudgetProject::getBudgetItemCode, codes)
                .list();
    }

    public List<BudgetProjectVo> findByBudgetItemNames(List<String> budgetItemNameList) {
        Validate.notEmpty(budgetItemNameList,"预算项目名称不能为空");
        List<BudgetProject> list = this.lambdaQuery()
                .eq(BudgetProject::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(BudgetProject::getBudgetItemName, budgetItemNameList)
                .list();
        if (CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        return (List<BudgetProjectVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list,BudgetProject.class,BudgetProjectVo.class, HashSet.class, ArrayList.class);

    }
    public List<BudgetProjectVo> findByBudgetItemCodes(List<String> budgetItemCodeList) {
        Validate.notEmpty(budgetItemCodeList,"预算项目编码不能为空");
        List<BudgetProject> list = this.lambdaQuery()
                .eq(BudgetProject::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(BudgetProject::getBudgetItemCode, budgetItemCodeList)
                .list();
        if (CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        return (List<BudgetProjectVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list,BudgetProject.class,BudgetProjectVo.class, HashSet.class, ArrayList.class);
    }
    public List<BudgetProjectVo> findByBudgetItemLevels(List<String> budgetItemLevelList) {
        Validate.notEmpty(budgetItemLevelList,"预算项目层级不能为空");
        List<BudgetProject> list = this.lambdaQuery()
                .eq(BudgetProject::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(BudgetProject::getBudgetItemHierarchy, budgetItemLevelList)
                .list();
        if (CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        return (List<BudgetProjectVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list,BudgetProject.class,BudgetProjectVo.class, HashSet.class, ArrayList.class);
    }


}
