package com.biz.crm.tpm.business.budget.controls.config.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.tpm.business.budget.controls.config.local.entity.BudgetProject;
import com.biz.crm.tpm.business.budget.controls.config.local.entity.DimensionControls;
import com.biz.crm.tpm.business.budget.controls.config.local.repository.BudgetProjectRepository;
import com.biz.crm.tpm.business.budget.controls.config.local.repository.DimensionControlsRepository;
import com.biz.crm.tpm.business.budget.controls.config.local.util.ListConverterUtils;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.BasicInformationQueryBudgetItemsDto;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.BudgetProjectDto;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.DimensionControlsDto;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.BudgetProjectService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.DimensionControlsService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.BudgetProjectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.stream.Collectors;


/**
 * (BudgetProjectServiceImpl)表服务实现类
 *
 * @author makejava
 * @since 2022-11-03 17:06:02
 */
@Slf4j
@Service("budgetProjectService")
public class BudgetProjectServiceImpl implements BudgetProjectService {

    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private BudgetProjectRepository budgetProjectRepository;


    @Autowired
    private DimensionControlsService dimensionControlsService;

    @Autowired(required = false)
    private DimensionControlsRepository dimensionControlsRepository;


    /**
     * 选择预算项目
     */
    @Override
    public void create(BudgetProjectDto budgetProjectDto, String promotionCode) {
        Validate.isTrue(StringUtils.isNotBlank(budgetProjectDto.getBudgetItemCode()), "预算项目编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(budgetProjectDto.getBudgetItemHierarchy()), "预算项目层级不能为空");
        Validate.isTrue(StringUtils.isNotBlank(budgetProjectDto.getBudgetItemName()), "预算项目名称不能为空");
        /**新增管控配置编码*/
        budgetProjectDto.setControlsConfigCode(promotionCode);
        BudgetProject budgetItem = this.nebulaToolkitService.copyObjectByWhiteList(budgetProjectDto, BudgetProject.class, LinkedHashSet.class, ArrayList.class);
        budgetProjectRepository.saveOrUpdate(budgetItem);
    }

    /**
     * 修改选择预算项目
     */
    @Override
    public BudgetProjectVo update(BudgetProjectDto dto) {
        List<BudgetProjectVo> formeResult = this.getcontrolsConfigCode(dto.getControlsConfigCode());
        if (ObjectUtils.isEmpty(formeResult)) {
            throw new RuntimeException("修改数据失败，原数据不存在！");
        }
        BudgetProject value = this.nebulaToolkitService.copyObjectByWhiteList(dto, BudgetProject.class, HashSet.class, ArrayList.class);
        this.budgetProjectRepository.saveOrUpdate(value);
        BudgetProjectVo budgetProjectVo = this.nebulaToolkitService.copyObjectByWhiteList(dto, BudgetProjectVo.class, HashSet.class, ArrayList.class);
        return budgetProjectVo;
    }


    /**
     * 通过controlsConfigCode查询选择预算项目
     */
    @Override
    public List<BudgetProjectVo> getcontrolsConfigCode(String delFlag) {
        List<BudgetProject> result = budgetProjectRepository.getcontrolsConfigCode(delFlag);
        ListConverterUtils converterUtils = new ListConverterUtils(BudgetProjectVo.class);
        List<BudgetProjectVo> arrayList = converterUtils.converterList(result);
        return arrayList;
    }


    /**
     * 通过基本信息查询预算项目接口
     *
     * @param dto 对象
     */
    @Override
    public List<BudgetProjectVo> basicInformationQueryBudgetItems(BasicInformationQueryBudgetItemsDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            throw new RuntimeException("查询数据不不存在！");
        }
        // 关键指标不能为空
        if (StringUtils.isEmpty(dto.getBusinessFormatCode())
                || StringUtils.isEmpty(dto.getBusinessUnitCode())
                || StringUtils.isEmpty(dto.getMarketOrganization())) {
            return Lists.newArrayList();
        }
        //通过dto查询预算管理对象
        DimensionControlsDto dimensionControlsDto = dimensionControlsService.findBudgetManagement(dto);
        //通过管控配置编码获取预算管理
        List<BudgetProjectVo> budgetProjectVos = this.getcontrolsConfigCode(dimensionControlsDto.getControlsConfigCode());
        if (ObjectUtils.isEmpty(budgetProjectVos)) {
            throw new RuntimeException("查询数据不存在");
        }
        return budgetProjectVos;
    }


    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto)     查询实体
     * @return 所有数据
     */
    @Override
    public Page<BudgetProjectVo> findByConditions(Pageable pageable, BudgetProjectDto dto) {

        if (pageable == null) {
            pageable = PageRequest.of(1, 50);
        }
        if (dto == null) {
            dto = new BudgetProjectDto();
        }

        return this.budgetProjectRepository.findByConditions(pageable, dto);
    }

    /**
     * 通过基本信息查询预算项目接口
     *
     * @param controlsConfigCode 管控配置编码
     */
    @Override
    public void deleteByControlsConfigCode(String controlsConfigCode) {
        this.budgetProjectRepository.deleteByControlsConfigCode(controlsConfigCode);
    }

    /**
     * 批量新增预算项目
     */
    @Override
    public void savaBatch(List<BudgetProjectDto> dto) {
        for (BudgetProjectDto d : dto) {
            d.setId(null);
        }

        if (dto != null && dto.size() > 0) {
            ArrayList<BudgetProject> productEntityList =
                    Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(dto, BudgetProjectDto.class, BudgetProject.class, HashSet.class, ArrayList.class));
            budgetProjectRepository.saveOrUpdateBatch(productEntityList);
        }
    }


    /**
     * 选择规则制定表详情
     */
    @Override
    public List<BudgetProjectVo> finByByControlsConfigCodeAndDelFlag(String delFlag, String controlsConfigCode) {
        List<BudgetProject> result = this.budgetProjectRepository.finByByControlsConfigCodeAndDelFlag(delFlag, controlsConfigCode);
        ListConverterUtils converterUtils = new ListConverterUtils(BudgetProjectVo.class);
        List<BudgetProjectVo> arrayList = converterUtils.converterList(result);
        return arrayList;
    }

    /**
     * 根据编码批量查询
     *
     * @param controlsConfigCodeList
     * @return java.util.List<com.biz.crm.tpm.business.budget.controls.config.sdk.vo.BudgetProjectVo>
     * @author huojia
     * @date 2022/12/27 16:27
     **/
    @Override
    public List<BudgetProjectVo> listByControlsConfigCodeList(List<String> controlsConfigCodeList) {
        if (CollectionUtils.isEmpty(controlsConfigCodeList)) {
            return null;
        }
        List<BudgetProject> budgetProjects = budgetProjectRepository.listByControlsConfigCodeList(controlsConfigCodeList);
        if (CollectionUtils.isEmpty(budgetProjects)) {
            return null;
        }
        return (List<BudgetProjectVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                budgetProjects, BudgetProject.class, BudgetProjectVo.class, LinkedHashSet.class, ArrayList.class
        );
    }


    /**
     * 根据维度判断预算项目是否重复
     */
    @Override
    public void checkRepeat(List<BudgetProjectDto> projectDtos, DimensionControlsDto controlsDto) {
        Validate.notEmpty(projectDtos, "预算项目列表为空");
        Validate.notNull(controlsDto, "预算管控信息为空");
        //获取预算项目编码集合
        List<String> codes = projectDtos.stream().map(BudgetProjectDto::getBudgetItemCode).collect(Collectors.toList());
        Validate.notEmpty(codes, "预算项目编码为空");
        Validate.notEmpty(controlsDto.getBusinessFormatCode(), "业态不能为空");
        Validate.notEmpty(controlsDto.getBusinessUnitCode(), "业务单元不能为空");
//        Validate.notEmpty(controlsDto.getOrgCode(), "部门不能为空");
        //先获取预算管控主信息
        List<DimensionControls> list = dimensionControlsRepository.getControlsByConditions(controlsDto);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        //根据预算管控主信息+预算项目获取预算项目信息
        List<String> controlsCodes = list.stream().map(DimensionControls::getControlsConfigCode).collect(Collectors.toList());
        List<BudgetProject> projects = this.budgetProjectRepository.getByConfigCodesAndItemCodes(controlsCodes, codes);
        if (!CollectionUtils.isEmpty(projects)) {
            Validate.isTrue(false, "预算项目[" + projects.get(0).getBudgetItemCode() + "]已存在其他冲突管控维度，请确认！");
        }
    }

    @Override
    public List<BudgetProjectVo> findByBudgetItemNames(List<String> budgetItemNameList) {
        if (CollectionUtils.isEmpty(budgetItemNameList)){
            return Lists.newArrayList();
        }
        return this.budgetProjectRepository.findByBudgetItemNames(budgetItemNameList);
    }
}
