package com.biz.crm.tpm.business.budget.controls.config.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.constant.DimensionControlsConstant;
import com.biz.crm.tpm.business.budget.controls.config.local.entity.DimensionControls;
import com.biz.crm.tpm.business.budget.controls.config.local.repository.DimensionControlsRepository;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.*;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.log.DimensionControlsLogEventDto;
import com.biz.crm.tpm.business.budget.controls.config.sdk.enums.DimensionControlsTypeEnum;
import com.biz.crm.tpm.business.budget.controls.config.sdk.event.log.DimensionControlsLogEventListener;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.BudgetProjectService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.DimensionControlsService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.RulesEnactService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.BudgetProjectVo;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.ControlTypeDataVo;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.DimensionControlsVo;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.RulesEnactVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * (DimensionControlsVo)表服务实现类
 *
 * @author dutaotao
 * @since 2022-11-03 17:06:02
 */
@Slf4j
@Service("dimensionControlsService")
public class DimensionControlsServiceImpl implements DimensionControlsService {

    @Autowired
    private DimensionControlsRepository dimensionControlsRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RulesEnactService rulesEnactService;

    @Autowired(required = false)
    private BudgetProjectService budgetProjectService;

    @Autowired
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired
    private GenerateCodeService generateCodeService;

    private final long REMARK_LENGTH = 400;

    private final int NAME_LENGTH = 64;


    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      实体对象
     * @return
     */
    @Override
    public Page<DimensionControlsVo> findByConditions(Pageable pageable, DimensionControlsDto dto) {
        if (pageable == null) {
            pageable = PageRequest.of(1, 50);
        }
        if (dto == null) {
            dto = new DimensionControlsDto();
        }

        /**获取租户编号*/
        if (StringUtils.isBlank(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }

        /**获取删除标志*/
        if (StringUtils.isBlank(dto.getDelFlag())) {
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }

        return this.dimensionControlsRepository.findByConditions(pageable, dto);
    }


    /**
     * 新增整体管控和管控维度数据
     *
     * @param dto 查询实体
     * @return 所有数据
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public DimensionControlsVo create(ControlTypeDataDto dto) {
        //选择预算项目
        DimensionControlsDto dimensionControlsDto = dto.getDimensionControlsDto();
        //预算管理
        List<RulesEnactDto> rulesEnactDto = dto.getRulesEnactDto();
        //规则制定表
        List<BudgetProjectDto> budgetProjectDto = dto.getBudgetProjectDto();
        //1、添加公共字段
        dimensionControlsDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        if (BooleanEnum.FALSE.getCapital().equals(dimensionControlsDto.getIfRolling())) {
            dimensionControlsDto.setRollingType(null);
        }
        //添加租房编号
        if (StringUtils.isBlank(dimensionControlsDto.getTenantCode())) {
            dimensionControlsDto.setTenantCode(TenantUtils.getTenantCode());
        }
        //部门编码重置
        if (StringUtils.isBlank(dimensionControlsDto.getOrgCode())) {
            dimensionControlsDto.setOrgCode(null);
            dimensionControlsDto.setOrgName(null);
        }
        //促销政策常量
        String ruleCode = DimensionControlsConstant.BUDGET_MANAGEMENT_TABLE_PREFIX;
        String promotionCode = generateCodeService.generateCode(ruleCode, 1).get(00000);
        Validate.notBlank(promotionCode, "促销编号生成失败");
        dimensionControlsDto.setControlsConfigCode(promotionCode);

        /**设置启用状态*/
        dimensionControlsDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        //新增整体管控
        DimensionControls dimensionControls = this.nebulaToolkitService.copyObjectByWhiteList(dimensionControlsDto, DimensionControls.class, LinkedHashSet.class, ArrayList.class);
        this.dimensionControlsRepository.saveOrUpdate(dimensionControls);
        if (dimensionControlsDto.getControlType().equals(DimensionControlsTypeEnum.OVERALL_CONTROL.getCode())) {
            rulesEnactDto.forEach(e -> {
                rulesEnactService.create(e, promotionCode);
            });
        }
        //新增管控维度
        if (dimensionControlsDto.getControlType().equals(DimensionControlsTypeEnum.DIMENSION_CONTROL.getCode())) {
            //判断预算项目是否存在重复数据
            budgetProjectService.checkRepeat(budgetProjectDto,dimensionControlsDto);
            budgetProjectDto.forEach(b -> {
                budgetProjectService.create(b, promotionCode);
            });
        }
        DimensionControlsVo dimensionControlsVo = this.nebulaToolkitService.copyObjectByWhiteList(dimensionControls, DimensionControlsVo.class, LinkedHashSet.class, ArrayList.class);
        //新增业务日志
        DimensionControlsLogEventDto logEventDto = new DimensionControlsLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dimensionControlsDto);
        SerializableBiConsumer<DimensionControlsLogEventListener, DimensionControlsLogEventDto> onCreate =
                DimensionControlsLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, DimensionControlsLogEventListener.class, onCreate);
        return dimensionControlsVo;
    }


    /**
     * 根据主键、是否编辑页面标记查询预算管理
     *
     * @param id
     * @return com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetVo
     * @author huojia
     * @date 2022/11/1 21:17
     **/
    @Override
    public ControlTypeDataVo findById(String id) {
        List<RulesEnactVo> arrayList1 = new ArrayList<>();
        List<BudgetProjectVo> arrayList2 = new ArrayList<>();
        if (StringUtils.isBlank(id)) {
            return null;
        }
        DimensionControls dimensionControls = this.dimensionControlsRepository.getById(id, DelFlagStatusEnum.NORMAL.getCode());
        if (dimensionControls.getControlType().equals(DimensionControlsTypeEnum.OVERALL_CONTROL.getCode())) {
            // 选择预算项目表详情
            arrayList1 = rulesEnactService.finByByControlsConfigCodeAndDelFlag(dimensionControls.getDelFlag(), dimensionControls.getControlsConfigCode());
        } else if (dimensionControls.getControlType().equals(DimensionControlsTypeEnum.DIMENSION_CONTROL.getCode())) {
            // 选择规则制定表详情
            arrayList2 = budgetProjectService.finByByControlsConfigCodeAndDelFlag(dimensionControls.getDelFlag(), dimensionControls.getControlsConfigCode());
        }
        DimensionControlsVo dimensionControlsVo = this.nebulaToolkitService.copyObjectByWhiteList(dimensionControls, DimensionControlsVo.class, LinkedHashSet.class, ArrayList.class);
        return new ControlTypeDataVo(
                dimensionControlsVo,
                arrayList1,
                arrayList2
        );
    }


    /**
     * 编辑预算管控预算
     *
     * @param dto
     * @return com.biz.crm.tpm.business.year.budget.sdk.vo.ControlTypeDataVo
     * @author huojia
     * @date 2022/11/1 21:15
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(ControlTypeDataDto dto) {
        //预算管理表
        DimensionControlsDto dimensionControlsDto = dto.getDimensionControlsDto();
        //规则制定表
        List<RulesEnactDto> rulesEnactDto = dto.getRulesEnactDto();
        //选择预算项目
        List<BudgetProjectDto> budgetProjectDto = dto.getBudgetProjectDto();

        DimensionControls byId = this.dimensionControlsRepository.getConfigCode(dimensionControlsDto.getControlsConfigCode());
        if (ObjectUtils.isEmpty(byId)) {
            throw new RuntimeException("修改数据失败，原数据不存在！");
        }
        if (!byId.getControlsConfigCode().equals(dimensionControlsDto.getControlsConfigCode())) {
            throw new RuntimeException("管控配置编码不可被修改");
        }
        if (dto.getDimensionControlsDto().getControlsConfigName().length() > NAME_LENGTH){
            throw new RuntimeException("管控配置名称过长，不能超过64");
        }
        if (StringUtils.isNotEmpty(dto.getDimensionControlsDto().getRemark())) {
            if (dto.getDimensionControlsDto().getRemark().length() > REMARK_LENGTH){
                throw new RuntimeException("规则描述过长，不能超过400");
            }
        }
        //设置创建时间
        if (dimensionControlsDto.getModifyTime() != null) {
            dimensionControlsDto.setModifyTime(new Date());
        }
        if (BooleanEnum.FALSE.getCapital().equals(dimensionControlsDto.getIfRolling())) {
            dimensionControlsDto.setRollingType(null);
        }
        // 保存预算管理表
        DimensionControls dimensionControls = this.nebulaToolkitService.copyObjectByWhiteList(dimensionControlsDto,
                DimensionControls.class,
                LinkedHashSet.class,
                ArrayList.class);
        this.dimensionControlsRepository.deleteById(dimensionControls);
        dimensionControls.setTenantCode(TenantUtils.getTenantCode());
        this.dimensionControlsRepository.saveOrUpdate(dimensionControls);
        if (dimensionControlsDto.getControlType().equals(DimensionControlsTypeEnum.OVERALL_CONTROL.getCode())) {
            //根据编码类型删除管控配置编码
            rulesEnactService.deleteByControlsConfigCode(dimensionControlsDto.getControlsConfigCode());
            //批量新增维度信息表
            rulesEnactService.savaBatch(rulesEnactDto);
        }
        if (dimensionControlsDto.getControlType().equals(DimensionControlsTypeEnum.DIMENSION_CONTROL.getCode())) {
            budgetProjectService.checkRepeat(budgetProjectDto,dimensionControlsDto);
            //根据编码类型删除选择预算项目
            budgetProjectService.deleteByControlsConfigCode(dimensionControlsDto.getControlsConfigCode());
            //批量新增预算项目
            budgetProjectService.savaBatch(budgetProjectDto);
        }
        // 业务日志更新
        DimensionControlsLogEventDto logEventDto = new DimensionControlsLogEventDto();
        DimensionControlsVo dimensionControlsVo = this.nebulaToolkitService.copyObjectByWhiteList(byId, DimensionControlsVo.class, HashSet.class, ArrayList.class);
        logEventDto.setOriginal(dimensionControlsVo);
        logEventDto.setNewest(dimensionControlsDto);
        SerializableBiConsumer<DimensionControlsLogEventListener, DimensionControlsLogEventDto> onUpdate =
                DimensionControlsLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, DimensionControlsLogEventListener.class, onUpdate);
    }

    /**
     * 批量查询
     */
    private List<DimensionControlsVo> findByIds(Set<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        List<DimensionControls> approvalCollects = dimensionControlsRepository.listByIds(ids);
        if (CollectionUtils.isEmpty(approvalCollects)) {
            return Lists.newArrayList();
        }
        return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(approvalCollects, DimensionControls.class, DimensionControlsVo.class, HashSet.class, ArrayList.class));
    }

    /**
     * 通过dto查询预算管理对象
     */
    @Override
    public DimensionControlsDto findBudgetManagement(BasicInformationQueryBudgetItemsDto dto) {
        return dimensionControlsRepository.findBudgetManagement(dto);
    }


    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(Set<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要删除的数据！");
        }
        List<DimensionControls> dimensionControls = this.dimensionControlsRepository.listByIds(ids);
        if (!CollectionUtils.isEmpty(dimensionControls)) {
            dimensionControls.forEach(dimensionBudgetVo -> {
                dimensionBudgetVo.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
                //删除业务日志
                DimensionControlsLogEventDto logEventDto = new DimensionControlsLogEventDto();
                DimensionControlsVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(dimensionControls, DimensionControlsVo.class, null, null);
                logEventDto.setOriginal(oldVo);
                DimensionControlsDto dimensionControlsDto = this.nebulaToolkitService.copyObjectByWhiteList(dimensionControls, DimensionControlsDto.class, null, null);
                dimensionControlsDto.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
                logEventDto.setNewest(dimensionControlsDto);
                final SerializableBiConsumer<DimensionControlsLogEventListener, DimensionControlsLogEventDto> onDelete =
                        DimensionControlsLogEventListener::onDelete;
                this.nebulaNetEventClient.publish(logEventDto, DimensionControlsLogEventListener.class, onDelete);
            });
            this.dimensionControlsRepository.updateBatchById(dimensionControls);
            ids.forEach(controlsById -> {
                //执行删除核销采集（主表逻辑删除，明细项信息不做变更）
                this.dimensionControlsRepository.deleteByYearBudgetId(controlsById);
            });
        }

    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要操作的数据");
        List<DimensionControls> dimensionControls = this.dimensionControlsRepository.listByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(dimensionControls), "不存在或已删除!");

        ArrayList<DimensionControls> dimensionControlsList = new ArrayList<>();
        dimensionControls.forEach(item -> {
            DimensionControls dimensionControls1 = new DimensionControls();
            dimensionControls1.setId(item.getId());
            dimensionControls1.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dimensionControlsList.add(dimensionControls1);

            //启用业务日志
            DimensionControlsLogEventDto logEventDto = new DimensionControlsLogEventDto();
            DimensionControlsVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DimensionControlsVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DimensionControlsDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DimensionControlsDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DimensionControlsLogEventListener, DimensionControlsLogEventDto> onEnable =
                    DimensionControlsLogEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, DimensionControlsLogEventListener.class, onEnable);
        });
        this.dimensionControlsRepository.updateBatchById(dimensionControlsList);
    }


    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要操作的数据");
        List<DimensionControls> dimensionControls = this.dimensionControlsRepository.listByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(dimensionControls), "不存在或已删除!");

        ArrayList<DimensionControls> dimensionControlsList = new ArrayList<>();
        dimensionControls.forEach(item -> {
            DimensionControls dimensionControls1 = new DimensionControls();
            dimensionControls1.setId(item.getId());
            dimensionControls1.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            dimensionControlsList.add(dimensionControls1);

            //禁用业务日志
            DimensionControlsLogEventDto logEventDto = new DimensionControlsLogEventDto();
            DimensionControlsVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DimensionControlsVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DimensionControlsDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DimensionControlsDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DimensionControlsLogEventListener, DimensionControlsLogEventDto> onDisable =
                    DimensionControlsLogEventListener::onDisable;
            this.nebulaNetEventClient.publish(logEventDto, DimensionControlsLogEventListener.class, onDisable);
        });
        this.dimensionControlsRepository.updateBatchById(dimensionControlsList);
    }

    /**
     * 条件查询
     *
     * @param dimensionControlsDto
     * @return java.util.List<com.biz.crm.tpm.business.budget.controls.config.sdk.vo.DimensionControlsVo>
     * @author huojia
     * @date 2022/12/27 16:17
     **/
    @Override
    public List<DimensionControlsVo> listByConditions(DimensionControlsDto dimensionControlsDto) {
        Validate.notNull(dimensionControlsDto, "请求参数不能为空！");
        List<DimensionControls> dimensionControlsList = this.dimensionControlsRepository.listByConditions(dimensionControlsDto);
        if (CollectionUtils.isEmpty(dimensionControlsList)) {
            return null;
        }
        List<DimensionControlsVo> dimensionControlsVos = (List<DimensionControlsVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                dimensionControlsList, DimensionControls.class, DimensionControlsVo.class, LinkedHashSet.class, ArrayList.class
        );
        List<String> controlsConfigCodeList = dimensionControlsVos.stream().map(DimensionControlsVo::getControlsConfigCode).collect(Collectors.toList());
        List<BudgetProjectVo> budgetProjectVos = budgetProjectService.listByControlsConfigCodeList(controlsConfigCodeList);
        if (CollectionUtils.isEmpty(budgetProjectVos)) {
            return dimensionControlsVos;
        }
        Map<String, List<BudgetProjectVo>> map = budgetProjectVos.stream().collect(Collectors.groupingBy(BudgetProjectVo::getControlsConfigCode));
        dimensionControlsVos.forEach(dimensionControlsVo -> {
            if (map.containsKey(dimensionControlsVo.getControlsConfigCode())) {
                List<String> collect = map.get(dimensionControlsVo.getControlsConfigCode()).stream().map(BudgetProjectVo::getBudgetItemCode).collect(Collectors.toList());
                dimensionControlsVo.setBudgetItemCodeList(collect);
            }
        });
        return dimensionControlsVos;
    }

    @Override
    public List<DimensionControlsVo> findByDimensionControlsDto(DimensionControlsDto dto) {
        return this.dimensionControlsRepository.findByDimensionControlsDto(dto);
    }
}
