package com.biz.crm.tpm.business.business.policy.local.imports;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.tpm.business.business.policy.local.imports.vo.PolicyProductImportVo;
import com.biz.crm.tpm.business.business.policy.local.service.BusinessPolicyProductService;
import com.biz.crm.tpm.business.business.policy.sdk.dto.BusinessPolicyProductDto;
import com.bizunited.nebula.common.util.JsonUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.beans.FeatureDescriptor;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description: 商务政策客户导入
 * @Author: dengwei
 * @Date: 2023/2/2 17:27
 * @Version: 1.0
 */
@Slf4j
@Component
public class PolicyProductImportsProcess implements ImportProcess<PolicyProductImportVo> {

    @Autowired(required = false)
    private BusinessPolicyProductService businessPolicyProductService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, PolicyProductImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        String cacheKey = (String) params.get("cacheKey");
        Validate.isTrue(!StringUtils.isEmpty(cacheKey), "缓存key不能空");

        log.info("商务政策导入数据{}", JsonUtils.obj2JsonString(data.values()));

        List<BusinessPolicyProductDto> buildData = buildData(data);

        log.info("商务政策导入构建后数据{}", JsonUtils.obj2JsonString(buildData));

        businessPolicyProductService.saveListCache(cacheKey,buildData);
        return null;
    }

    private List<BusinessPolicyProductDto> buildData(LinkedHashMap<Integer, PolicyProductImportVo> data) {

        List<BusinessPolicyProductDto> businessPolicyProductDtoList = new ArrayList<>();
        try {

            List<PolicyProductImportVo> policyProductImportVos = Lists.newArrayList(data.values());
            List<String> productList = policyProductImportVos.stream().map(PolicyProductImportVo::getProductCode).collect(Collectors.toList());
            List<ProductVo> productVoList = productVoService.findByProductCodes(productList);
            Map<String, ProductVo> productVoMap = productVoList.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity(), (o, n) -> n));
            policyProductImportVos.forEach(item -> {
                Validate.notBlank(item.getProductCode(), "产品编码不能为空");
                Validate.notNull(productVoMap.get(item.getProductCode()), "产品编码【%s】未找到", item.getProductCode());
            });
            Map<String, List<PolicyProductImportVo>> policyProductImportMap = policyProductImportVos.stream().collect(Collectors.groupingBy(PolicyProductImportVo::getProductCode));
            policyProductImportMap.forEach((key, value) -> {
                Validate.isTrue(value.size() <= 1, "产品编码【%s】录入重复", key);
            });
            for (PolicyProductImportVo item : policyProductImportVos) {
                ProductVo productVo = productVoMap.get(item.getProductCode());
                BusinessPolicyProductDto dto = new BusinessPolicyProductDto();
                BeanUtils.copyProperties(item, dto);
                amountTransform(item, dto);
                dto.setId(UuidCrmUtil.general());
                dto.setProductBrandCode(productVo.getProductBrandCode());
                dto.setProductBrandName(productVo.getProductBrandName());
                dto.setProductCategoryCode(productVo.getProductCategoryCode());
                dto.setProductCategoryName(productVo.getProductCategoryName());
                dto.setProductCode(productVo.getProductCode());
                dto.setProductName(productVo.getProductName());
                if (Objects.nonNull(dto.getOfferGoodsPrice()) && Objects.nonNull(dto.getPredictSaleQuantity())) {
                    BigDecimal amount = dto.getOfferGoodsPrice().multiply(dto.getPredictSaleQuantity());
                    dto.setPredictBeforeDiscountSaleAmount(dto.getOfferGoodsPrice().multiply(dto.getPredictSaleQuantity()));
                    if (Objects.nonNull(dto.getTenDayReturnPoint())) {
                        dto.setBaseReturnPointTenDayReturnFee(dto.getTenDayReturnPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getMonthReturnPoint())) {
                        dto.setBaseReturnPointMonthReturnFee(dto.getMonthReturnPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getTargetReachReturnPoint())) {
                        dto.setTargetReachReturnPointFee(dto.getTargetReachReturnPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getCombinationPoint())) {
                        dto.setCombinationFee(dto.getCombinationPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getBaseOperationPoint())) {
                        dto.setBaseOperationFee(dto.getBaseOperationPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getManPoint())) {
                        dto.setPeopleFee(dto.getManPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getPutPoint())) {
                        dto.setPutFee(dto.getPutPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getLogisticPoint())) {
                        dto.setLogisticSupportFee(dto.getLogisticPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getQuarterReturnPoint())) {
                        dto.setQuarterReturnFee(dto.getQuarterReturnPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getYearReturnPoint())) {
                        dto.setYearReturnFee(dto.getYearReturnPoint().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getOtherPolicy1Point())) {
                        dto.setOtherPolicy1Fee(dto.getOtherPolicy1Point().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getOtherPolicy2Point())) {
                        dto.setOtherPolicy2Fee(dto.getOtherPolicy2Point().multiply(amount));
                    }
                    if (Objects.nonNull(dto.getOtherPolicy3Point())) {
                        dto.setOtherPolicy3Fee(dto.getOtherPolicy3Point().multiply(amount));
                    }
                }
                if (Objects.nonNull(dto.getOtherPolicy4Price())&&Objects.nonNull(dto.getPredictSaleQuantity())) {
                    dto.setOtherPolicy4Price(dto.getOtherPolicy4Price().multiply(dto.getPredictSaleQuantity()));
                }
                dto.setFeeTotal(Optional.ofNullable(dto.getBaseReturnPointTenDayReturnFee()).orElse(BigDecimal.ZERO)
                        .multiply(Optional.ofNullable(dto.getBaseReturnPointMonthReturnFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getTargetReachReturnPointFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getCombinationFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getBaseOperationFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getPeopleFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getPutFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getLogisticSupportFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getQuarterReturnFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getYearReturnFee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getOtherPolicy1Fee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getOtherPolicy2Fee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getOtherPolicy3Fee()).orElse(BigDecimal.ZERO))
                        .multiply(Optional.ofNullable(dto.getOtherPolicy4Price()).orElse(BigDecimal.ZERO)));
                businessPolicyProductDtoList.add(dto);
            }

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            String errorMsg = e.getMessage();
            if (StringUtils.isEmpty(e.getMessage())) {
                errorMsg = "数据处理失败！";
            }
            throw new IllegalArgumentException(errorMsg);
        }
        return businessPolicyProductDtoList;
    }

    private static void amountTransform(PolicyProductImportVo item, BusinessPolicyProductDto dto) throws
            InvocationTargetException, IllegalAccessException {
        PropertyDescriptor[] ppivs = BeanUtils.getPropertyDescriptors(PolicyProductImportVo.class);
        PropertyDescriptor[] bppds = BeanUtils.getPropertyDescriptors(BusinessPolicyProductDto.class);
        Map<String, PropertyDescriptor> bppdsMap = Arrays.stream(bppds).collect(Collectors.toMap(FeatureDescriptor::getName, Function.identity()));
        for (PropertyDescriptor ppiv : ppivs) {
            Object ppivValue = ppiv.getReadMethod().invoke(item);
            if(Objects.isNull(ppivValue)){
                continue;
            }
            String propertyName = ppiv.getName();
            PropertyDescriptor bppd = bppdsMap.get(propertyName);
            if (!ppiv.getPropertyType().equals(bppd.getPropertyType())) {
                if (ppiv.getPropertyType().equals(String.class) && bppd.getPropertyType().equals(BigDecimal.class)) {
                    bppd.getWriteMethod().invoke(dto, new BigDecimal((String) ppivValue));
                }
            }
        }

    }

    public static void main(String[] args) throws InvocationTargetException, IllegalAccessException {
        PolicyProductImportVo policyProductImportVo = new PolicyProductImportVo();
        policyProductImportVo.setOfferGoodsPrice("10.01");
        BusinessPolicyProductDto businessPolicyProductDto = new BusinessPolicyProductDto();
        amountTransform(policyProductImportVo,businessPolicyProductDto);
        log.info(JsonUtils.obj2JsonString(businessPolicyProductDto));
    }

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    @Override
    public Class<PolicyProductImportVo> findCrmExcelVoClass() {
        return PolicyProductImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "POLICY_CUSTOMER_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "商务政策产品导入";
    }
}
