package com.biz.crm.tpm.business.account.reconciliation.rule.local.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.account.reconciliation.factor.sdk.service.TpmAccountReconciliationFactorService;
import com.biz.crm.tpm.business.account.reconciliation.factor.sdk.vo.TpmAccountReconciliationFactorRespVo;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.entity.TpmAccountReconciliationRuleDateConfigEntity;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.entity.TpmAccountReconciliationRuleEntity;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.entity.TpmAccountReconciliationRuleRFactorEntity;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.mapper.TpmAccountReconciliationRuleDateConfigMapper;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.mapper.TpmAccountReconciliationRuleMapper;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.mapper.TpmAccountReconciliationRuleRFactorMapper;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.repository.TpmAccountReconciliationRuleDateConfigRepository;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.repository.TpmAccountReconciliationRuleRFactorRepository;
import com.biz.crm.tpm.business.account.reconciliation.rule.local.repository.TpmAccountReconciliationRuleRepository;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.constant.GeneratePeriodEnum;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.constant.TpmAccountReconciliationRuleConstant;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.dto.TpmAccountReconciliationRuleDateConfigDto;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.dto.TpmAccountReconciliationRuleDto;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.dto.TpmAccountReconciliationRuleLogDto;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.dto.TpmAccountReconciliationRuleRFactorDto;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.event.TpmAccountReconciliationRuleLogEventListener;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.service.TpmAccountReconciliationRuleService;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleDateConfigRespVo;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRFactorRespVo;
import com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRespVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * (TpmAccountReconciliationRule)表服务实现类
 *
 * @author jiangyin
 * @since 2022-11-07 17:34:24
 */
@Service("tpmAccountReconciliationRuleService")
@Slf4j
public class TpmAccountReconciliationRuleServiceImpl implements TpmAccountReconciliationRuleService {

    @Autowired(required = false)
    private TpmAccountReconciliationRuleMapper tpmAccountReconciliationRuleMapper;

    @Autowired(required = false)
    private TpmAccountReconciliationRuleRepository tpmAccountReconciliationRuleRepository;

    @Autowired(required = false)
    private TpmAccountReconciliationRuleDateConfigMapper tpmAccountReconciliationRuleDateConfigMapper;

    @Autowired(required = false)
    private TpmAccountReconciliationRuleDateConfigRepository tpmAccountReconciliationRuleDateConfigRepository;

    @Autowired(required = false)
    private TpmAccountReconciliationRuleRFactorMapper tpmAccountReconciliationRuleRFactorMapper;

    @Autowired(required = false)
    private TpmAccountReconciliationRuleRFactorRepository tpmAccountReconciliationRuleRFactorRepository;

    @Autowired(required = false)
    private TpmAccountReconciliationFactorService tpmAccountReconciliationFactorService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private TpmAccountReconciliationRuleLogEventListener tpmAccountReconciliationRuleLogEventListener;


    /**
     * 根据主键查询对账规则详情
     *
     * @param id 主键
     * @return com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRespVo
     * @author jiangyin
     * @date 2022/11/9 15:54
     */
    @Override
    public TpmAccountReconciliationRuleRespVo queryById(String id) {
        TpmAccountReconciliationRuleRespVo ruleRespVo = tpmAccountReconciliationRuleMapper.findByIdAndTenantCode(id, TenantUtils.getTenantCode());
        if (ruleRespVo != null) {
            List<TpmAccountReconciliationRuleDateConfigRespVo> dateConfigRespVos = tpmAccountReconciliationRuleDateConfigRepository.findByRuleCode(ruleRespVo.getAccountReconciliationRuleCode());
            ruleRespVo.setDateConfigList(dateConfigRespVos);
            List<TpmAccountReconciliationRuleRFactorRespVo> factorRespVos = tpmAccountReconciliationRuleRFactorRepository.findByRuleCode(ruleRespVo.getAccountReconciliationRuleCode());
            //获取关联对账要素的详细配置
            List<String> factorCodeList = factorRespVos.stream().map(TpmAccountReconciliationRuleRFactorRespVo::getAccountReconciliationFactorCode).collect(Collectors.toList());
            List<TpmAccountReconciliationFactorRespVo> factors = tpmAccountReconciliationFactorService.findByFactorCodeList(factorCodeList);
            factorRespVos.forEach(o -> factors.forEach(p -> {
                if (p.getAccountReconciliationFactorCode().equals(o.getAccountReconciliationFactorCode())) {
                    o.setFactorConfig(p);
                }
            }));
            ruleRespVo.setFactorList(factorRespVos);
            return ruleRespVo;
        } else {
            throw new UnsupportedOperationException("未查询到对账规则详情");
        }
    }

    /**
     * 查询对账规则列表
     *
     * @param reqVo
     * @param pageable
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRespVo>
     * @author jiangyin
     * @date 2022/11/9 15:54
     */
    @Override
    public Page<TpmAccountReconciliationRuleRespVo> queryByPage(TpmAccountReconciliationRuleDto reqVo, Pageable pageable) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (reqVo == null) {
            reqVo = new TpmAccountReconciliationRuleDto();
        }
        reqVo.setTenantCode(TenantUtils.getTenantCode());
        Page<TpmAccountReconciliationFactorRespVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return tpmAccountReconciliationRuleMapper.findByConditions(page, reqVo);
    }

    /**
     * 创建对账规则
     *
     * @param reqVo 保存参数
     * @return com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRespVo
     * @author jiangyin
     * @date 2022/11/9 15:54
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public TpmAccountReconciliationRuleDto create(TpmAccountReconciliationRuleDto reqVo) {
        if (reqVo.validate()) {
            if (checkDateConfig(reqVo)) {
                reqVo.setTenantCode(TenantUtils.getTenantCode());
                //同一生效范围，同一对账类型，时间不能有重叠
                List<String> existList = tpmAccountReconciliationRuleMapper.findExist(reqVo);
                if (!CollectionUtils.isEmpty(existList)) {
                    throw new UnsupportedOperationException("该对账类型同生效范围内，存在生效期重叠的对账规则，请重新确认");
                }
                //校验对账要素
                List<String> factorCodeList = reqVo.getFactorList().stream().map(TpmAccountReconciliationRuleRFactorDto::getAccountReconciliationFactorCode).collect(Collectors.toList());
                if (tpmAccountReconciliationFactorService.checkFactorCodeList(factorCodeList)) {
                    //保存对账规则主数据
                    TpmAccountReconciliationRuleEntity ruleEntity = new TpmAccountReconciliationRuleEntity();
                    copyValue(ruleEntity, reqVo);
//                    String ruleCode = TpmAccountReconciliationRuleConstant.ACCOUNT_RECONCILIATION_RULE_CODE_PREFIX + DateFormatUtils.format(new Date(), "yyyyMM");
                    ruleEntity.setAccountReconciliationRuleCode(generateCodeService.generateCode(TpmAccountReconciliationRuleConstant.ACCOUNT_RECONCILIATION_RULE_CODE_PREFIX, 1, 4, 2, TimeUnit.DAYS).get(0));
                    ruleEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    ruleEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    tpmAccountReconciliationRuleMapper.insert(ruleEntity);
                    reqVo.setId(ruleEntity.getId());
                    //保存生成时间配置
                    for (TpmAccountReconciliationRuleDateConfigDto dateConfig : reqVo.getDateConfigList()) {
                        Validate.isTrue(StringUtils.isNotEmpty(dateConfig.getStartDate()), "对账规则-对账生成时间配置的开始日期不能为空");
                        Validate.isTrue(StringUtils.isNotEmpty(dateConfig.getEndDate()), "对账规则-对账生成时间配置的结束日期不能为空");
                        Validate.isTrue(StringUtils.isNotEmpty(dateConfig.getCalculateDate()), "对账规则-对账生成时间配置的计算时间不能为空");
                        dateConfig.setTenantCode(TenantUtils.getTenantCode());
                        TpmAccountReconciliationRuleDateConfigEntity dateConfigEntity = new TpmAccountReconciliationRuleDateConfigEntity();
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        try {
                            dateConfigEntity.setCalculateDate(sdf.parse(dateConfig.getCalculateDate()));
                            dateConfigEntity.setStartDate(sdf.parse(dateConfig.getStartDate()));
                            dateConfigEntity.setEndDate(sdf.parse(dateConfig.getEndDate()));
                            dateConfigEntity.setAccountReconciliationRuleCode(ruleEntity.getAccountReconciliationRuleCode());
                            dateConfigEntity.setTenantCode(TenantUtils.getTenantCode());
                            tpmAccountReconciliationRuleDateConfigMapper.insert(dateConfigEntity);
                        } catch (ParseException parseException) {
                            throw new UnsupportedOperationException("对账规则-生成时间设置日期格式异常");
                        }
                    }

                    //保存对账要素信息
                    for (TpmAccountReconciliationRuleRFactorDto factor : reqVo.getFactorList()) {
                        Validate.isTrue(StringUtils.isNotEmpty(factor.getAccountReconciliationFactorCode()), "对账规则-对账要素编码不能为空");
                        Validate.isTrue(factor.getOrderNum() != null, "对账规则-对账要素排序不能为空");
                        factor.setTenantCode(TenantUtils.getTenantCode());
                        TpmAccountReconciliationRuleRFactorEntity factorEntity = new TpmAccountReconciliationRuleRFactorEntity();
                        factorEntity.setAccountReconciliationFactorCode(factor.getAccountReconciliationFactorCode());
                        factorEntity.setOrderNum(factor.getOrderNum());
                        factorEntity.setAccountReconciliationRuleCode(ruleEntity.getAccountReconciliationRuleCode());
                        factorEntity.setTenantCode(TenantUtils.getTenantCode());
                        tpmAccountReconciliationRuleRFactorMapper.insert(factorEntity);
                    }
                    TpmAccountReconciliationRuleLogDto logDto = new TpmAccountReconciliationRuleLogDto();
                    logDto.setNewest(reqVo);
                    tpmAccountReconciliationRuleLogEventListener.onCreate(logDto);
                } else {
                    throw new UnsupportedOperationException("对账要素数据中有不可用数据");
                }
            }
        } else {
            throw new UnsupportedOperationException("缺少必填数据");
        }
        return reqVo;
    }

    /**
     * 转存参数
     *
     * @param ruleEntity 对账规则实体
     * @param reqVo      保存参数
     * @return void
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    private void copyValue(TpmAccountReconciliationRuleEntity ruleEntity, TpmAccountReconciliationRuleDto reqVo) {
        ruleEntity.setTenantCode(reqVo.getTenantCode());
        ruleEntity.setAccountReconciliationRuleName(reqVo.getAccountReconciliationRuleName());
        ruleEntity.setAccountReconciliationType(reqVo.getAccountReconciliationType());
        ruleEntity.setBusinessFormatCode(reqVo.getBusinessFormatCode());
        ruleEntity.setBusinessUnitCode(reqVo.getBusinessUnitCode());
        ruleEntity.setRemark(reqVo.getRemark());
        ruleEntity.setGeneratePeriod(reqVo.getGeneratePeriod());
        ruleEntity.setGenerateCostDay(reqVo.getGenerateCostDay());
        ruleEntity.setStartDate(DateUtil.getDateByFormat(reqVo.getStartDate(), "yyyy-MM-dd HH:mm:ss"));
        ruleEntity.setEndDate(DateUtil.getDateByFormat(reqVo.getEndDate(), "yyyy-MM-dd HH:mm:ss"));
    }

    /**
     * 修改数据
     *
     * @param reqVo 更新参数
     * @return com.biz.crm.tpm.business.account.reconciliation.rule.sdk.vo.TpmAccountReconciliationRuleRespVo
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public TpmAccountReconciliationRuleDto edit(TpmAccountReconciliationRuleDto reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getId()) && reqVo.validate()) {
            if (checkDateConfig(reqVo)) {
                //同一生效范围，同一对账类型，时间不能有重叠
                reqVo.setTenantCode(TenantUtils.getTenantCode());
                List<String> existList = tpmAccountReconciliationRuleMapper.findExist(reqVo);
                if (!CollectionUtils.isEmpty(existList)) {
                    throw new UnsupportedOperationException("该对账类型同生效范围内，存在生效期重叠的对账规则，请重新确认");
                }
                //校验对账要素
                List<String> factorCodeList = reqVo.getFactorList().stream().map(TpmAccountReconciliationRuleRFactorDto::getAccountReconciliationFactorCode).collect(Collectors.toList());
                if (tpmAccountReconciliationFactorService.checkFactorCodeList(factorCodeList)) {
                    //保存对账规则主数据
                    TpmAccountReconciliationRuleEntity ruleEntity = tpmAccountReconciliationRuleMapper.selectById(reqVo.getId());
                    TpmAccountReconciliationRuleEntity original = tpmAccountReconciliationRuleMapper.selectById(reqVo.getId());
                    copyValue(ruleEntity, reqVo);
                    tpmAccountReconciliationRuleMapper.updateById(ruleEntity);

                    //保存生成时间配置
                    //将之前设置的时间配置逻辑删除
                    tpmAccountReconciliationRuleDateConfigMapper.deleteByRuleIdAndTenantCode(reqVo.getId(), TenantUtils.getTenantCode());
                    for (TpmAccountReconciliationRuleDateConfigDto dateConfig : reqVo.getDateConfigList()) {
                        Validate.isTrue(StringUtils.isNotEmpty(dateConfig.getStartDate()), "对账规则-对账生成时间配置的开始日期不能为空");
                        Validate.isTrue(StringUtils.isNotEmpty(dateConfig.getEndDate()), "对账规则-对账生成时间配置的结束日期不能为空");
                        Validate.isTrue(StringUtils.isNotEmpty(dateConfig.getCalculateDate()), "对账规则-对账生成时间配置的计算时间不能为空");
                        dateConfig.setTenantCode(TenantUtils.getTenantCode());
                        TpmAccountReconciliationRuleDateConfigEntity dateConfigEntity = new TpmAccountReconciliationRuleDateConfigEntity();
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        try {
                            dateConfigEntity.setTenantCode(TenantUtils.getTenantCode());
                            dateConfigEntity.setCalculateDate(sdf.parse(dateConfig.getCalculateDate()));
                            dateConfigEntity.setStartDate(sdf.parse(dateConfig.getStartDate()));
                            dateConfigEntity.setEndDate(sdf.parse(dateConfig.getEndDate()));
                            dateConfigEntity.setAccountReconciliationRuleCode(ruleEntity.getAccountReconciliationRuleCode());
                            tpmAccountReconciliationRuleDateConfigMapper.insert(dateConfigEntity);
                        } catch (ParseException parseException) {
                            throw new UnsupportedOperationException("对账规则-生成时间设置日期格式异常");
                        }
                    }

                    //保存对账要素信息
                    //将之前设置的对账要素配置逻辑删除
                    tpmAccountReconciliationRuleRFactorMapper.deleteByRuleIdAndTenantCode(reqVo.getId(), TenantUtils.getTenantCode());
                    for (TpmAccountReconciliationRuleRFactorDto factor : reqVo.getFactorList()) {
                        Validate.isTrue(StringUtils.isNotEmpty(factor.getAccountReconciliationFactorCode()), "对账规则-对账要素编码不能为空");
                        Validate.isTrue(factor.getOrderNum() != null, "对账规则-对账要素排序不能为空");
                        factor.setTenantCode(TenantUtils.getTenantCode());
                        TpmAccountReconciliationRuleRFactorEntity factorEntity = new TpmAccountReconciliationRuleRFactorEntity();
                        factorEntity.setTenantCode(TenantUtils.getTenantCode());
                        factorEntity.setAccountReconciliationFactorCode(factor.getAccountReconciliationFactorCode());
                        factorEntity.setOrderNum(factor.getOrderNum());
                        factorEntity.setAccountReconciliationRuleCode(ruleEntity.getAccountReconciliationRuleCode());
                        tpmAccountReconciliationRuleRFactorMapper.insert(factorEntity);

                    }
                    //保存日志
                    TpmAccountReconciliationRuleDto originalDto = new TpmAccountReconciliationRuleDto();
                    translateEntityToDto(original, originalDto);
                    TpmAccountReconciliationRuleLogDto logDto = new TpmAccountReconciliationRuleLogDto();
                    logDto.setOriginal(originalDto);
                    logDto.setNewest(reqVo);
                    tpmAccountReconciliationRuleLogEventListener.onUpdate(logDto);
                } else {
                    throw new UnsupportedOperationException("对账要素数据中有不可用数据");
                }
            }
        } else {
            throw new UnsupportedOperationException("缺少必填数据");
        }
        return reqVo;
    }

    /**
     * 通过主键删除数据
     *
     * @param id 主键
     * @return boolean
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteById(String id) {
        return false;
    }

    /**
     * 通过主键列表删除数据
     *
     * @param ids 主键列表
     * @return void
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIdList(List<String> ids) {
        List<TpmAccountReconciliationRuleEntity> entityList = tpmAccountReconciliationRuleMapper.findByIdListAndTenantCode(ids, TenantUtils.getTenantCode());
        tpmAccountReconciliationRuleRFactorMapper.deleteByRuleIdListAndTenantCode(ids, TenantUtils.getTenantCode());
        tpmAccountReconciliationRuleDateConfigMapper.deleteByRuleIdListAndTenantCode(ids, TenantUtils.getTenantCode());
        tpmAccountReconciliationRuleMapper.logicDeleteByIdList(ids, TenantUtils.getTenantCode());
        //保存操作日志
        for (TpmAccountReconciliationRuleEntity entity : entityList) {
            TpmAccountReconciliationRuleDto original = new TpmAccountReconciliationRuleDto();
            translateEntityToDto(entity, original);
            TpmAccountReconciliationRuleDto newest = nebulaToolkitService.copyObjectByWhiteList(entity, TpmAccountReconciliationRuleDto.class, LinkedHashSet.class, ArrayList.class);
            newest.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            TpmAccountReconciliationRuleLogDto logDto = new TpmAccountReconciliationRuleLogDto();
            logDto.setNewest(newest);
            logDto.setOriginal(original);
            tpmAccountReconciliationRuleLogEventListener.onDelete(logDto);
        }
    }

    /**
     * 通过主键列表启用数据
     *
     * @param ids 主键列表
     * @return void
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enable(List<String> ids) {
        List<TpmAccountReconciliationRuleEntity> entityList = tpmAccountReconciliationRuleMapper.selectBatchIds(ids);
        tpmAccountReconciliationRuleRFactorMapper.enableByRuleIdList(ids);
        tpmAccountReconciliationRuleDateConfigMapper.enableByRuleIdList(ids);
        tpmAccountReconciliationRuleMapper.enableByIdList(ids);
        //保存操作日志
        for (TpmAccountReconciliationRuleEntity entity : entityList) {
            TpmAccountReconciliationRuleDto original = new TpmAccountReconciliationRuleDto();
            original.setId(entity.getId());
            original.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            TpmAccountReconciliationRuleDto newest = new TpmAccountReconciliationRuleDto();
            original.setId(entity.getId());
            original.setEnableStatus(entity.getEnableStatus());
            TpmAccountReconciliationRuleLogDto logDto = new TpmAccountReconciliationRuleLogDto();
            logDto.setNewest(newest);
            logDto.setOriginal(original);
            tpmAccountReconciliationRuleLogEventListener.onEnable(logDto);
        }
    }

    /**
     * 封装数据传输类
     *
     * @param entity 实体
     * @param dto    数据传输类
     * @return void
     * @author jiangyin
     * @date 2022/11/17 15:07
     */
    private void translateEntityToDto(TpmAccountReconciliationRuleEntity entity, TpmAccountReconciliationRuleDto dto) {
        String startDate = DateFormatUtils.format(entity.getEndDate(), "yyyy-MM-dd HH:mm:ss");
        String endDate = DateFormatUtils.format(entity.getStartDate(), "yyyy-MM-dd HH:mm:ss");
        entity.setEndDate(null);
        entity.setStartDate(null);
        dto = nebulaToolkitService.copyObjectByWhiteList(entity, TpmAccountReconciliationRuleDto.class, LinkedHashSet.class, ArrayList.class);
        dto.setStartDate(startDate);
        dto.setEndDate(endDate);
    }

    /**
     * 通过主键列表禁用数据
     *
     * @param ids 主键列表
     * @return void
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disable(List<String> ids) {
        List<TpmAccountReconciliationRuleEntity> entityList = tpmAccountReconciliationRuleMapper.selectBatchIds(ids);

        tpmAccountReconciliationRuleRFactorMapper.disableByRuleIdList(ids);
        tpmAccountReconciliationRuleDateConfigMapper.disableByRuleIdList(ids);
        tpmAccountReconciliationRuleMapper.disableByIdList(ids);
        //保存操作日志
        for (TpmAccountReconciliationRuleEntity entity : entityList) {
            TpmAccountReconciliationRuleDto original = new TpmAccountReconciliationRuleDto();
            original.setId(entity.getId());
            original.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            TpmAccountReconciliationRuleDto newest = new TpmAccountReconciliationRuleDto();
            original.setId(entity.getId());
            original.setEnableStatus(entity.getEnableStatus());
            TpmAccountReconciliationRuleLogDto logDto = new TpmAccountReconciliationRuleLogDto();
            logDto.setNewest(newest);
            logDto.setOriginal(original);
            tpmAccountReconciliationRuleLogEventListener.onDisable(logDto);
        }
    }

    @Override
    public Page<TpmAccountReconciliationRuleRespVo> findCurrentEffectiveRule(Pageable pageRequest) {
        Page<TpmAccountReconciliationRuleRespVo> rulePage = tpmAccountReconciliationRuleRepository.findCurrentEffectiveRule(pageRequest);
        if (CollectionUtils.isEmpty(rulePage.getRecords())) {
            return rulePage;
        }
        //封装时间配置和对账要素配置
        List<TpmAccountReconciliationRuleRespVo> respVos = rulePage.getRecords();
        List<String> codeList = respVos.stream()
                .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationRuleCode()))
                .map(TpmAccountReconciliationRuleRespVo::getAccountReconciliationRuleCode)
                .distinct().collect(Collectors.toList());
        List<TpmAccountReconciliationRuleDateConfigRespVo> dateConfigList = tpmAccountReconciliationRuleDateConfigRepository.findByRuleCodes(codeList);
        List<TpmAccountReconciliationRuleRFactorRespVo> factorList = tpmAccountReconciliationRuleRFactorRepository.findByRuleCodes(codeList);
        //获取关联对账要素的详细配置
        List<String> factorCodeList = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(factorList)) {
            factorCodeList.addAll(factorList.stream()
                    .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationFactorCode()))
                    .map(TpmAccountReconciliationRuleRFactorRespVo::getAccountReconciliationFactorCode)
                    .distinct()
                    .collect(Collectors.toList()));
        }
        List<TpmAccountReconciliationFactorRespVo> factors = tpmAccountReconciliationFactorService.findByFactorCodeList(factorCodeList);
        Map<String, List<TpmAccountReconciliationRuleDateConfigRespVo>> ruleDateConfigMap = dateConfigList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationRuleCode()))
                .collect(Collectors.groupingBy(TpmAccountReconciliationRuleDateConfigRespVo::getAccountReconciliationRuleCode));
        Map<String, List<TpmAccountReconciliationRuleRFactorRespVo>> ruleRFactorMap = factorList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationRuleCode()))
                .collect(Collectors.groupingBy(TpmAccountReconciliationRuleRFactorRespVo::getAccountReconciliationRuleCode));
        Map<String, List<TpmAccountReconciliationFactorRespVo>> ruleFactorMap = factors.stream()
                .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationFactorCode()))
                .collect(Collectors.groupingBy(TpmAccountReconciliationFactorRespVo::getAccountReconciliationFactorCode));
        String accountReconciliationRuleCode = null;
        for (TpmAccountReconciliationRuleRespVo respVo : respVos) {
            accountReconciliationRuleCode = respVo.getAccountReconciliationRuleCode();
            factorList = ruleRFactorMap.get(accountReconciliationRuleCode);
            if (CollectionUtil.isNotEmpty(factorList)) {
                factorList.stream()
                        .filter(k -> StringUtils.isNotBlank(k.getAccountReconciliationFactorCode()))
                        .filter(k -> ruleFactorMap.containsKey(k.getAccountReconciliationFactorCode()))
                        .forEach(o -> ruleFactorMap.get(o.getAccountReconciliationFactorCode())
                                .forEach(p -> {
                                    if (o.getAccountReconciliationFactorCode()
                                            .equals(p.getAccountReconciliationFactorCode())) {
                                        o.setFactorConfig(p);
                                    }
                                }));
            }
            respVo.setDateConfigList(ruleDateConfigMap.get(accountReconciliationRuleCode));
            respVo.setFactorList(factorList);
        }
        rulePage.setRecords(respVos);
        return rulePage;
    }

    @Override
    public TpmAccountReconciliationRuleRespVo findByCode(String code) {
        TpmAccountReconciliationRuleRespVo ruleRespVo = tpmAccountReconciliationRuleMapper.findByCodeAndTenantCode(code, TenantUtils.getTenantCode());
        List<TpmAccountReconciliationRuleDateConfigRespVo> dateConfigRespVos = tpmAccountReconciliationRuleDateConfigRepository.findByRuleCode(code);
        ruleRespVo.setDateConfigList(dateConfigRespVos);
        List<TpmAccountReconciliationRuleRFactorRespVo> factorRespVos = tpmAccountReconciliationRuleRFactorRepository.findByRuleCode(code);
        //获取关联对账要素的详细配置
        List<String> factorCodeList = factorRespVos.stream().map(TpmAccountReconciliationRuleRFactorRespVo::getAccountReconciliationFactorCode).collect(Collectors.toList());
        List<TpmAccountReconciliationFactorRespVo> factors = tpmAccountReconciliationFactorService.findByFactorCodeList(factorCodeList);
        factorRespVos.forEach(o -> factors.forEach(p -> {
            if (p.getAccountReconciliationFactorCode().equals(o.getAccountReconciliationFactorCode())) {
                o.setFactorConfig(p);
            }
        }));
        ruleRespVo.setFactorList(factorRespVos);
        return ruleRespVo;
    }

    /**
     * 校验日期配置
     *
     * @param reqVo 请求参数
     * @return boolean
     * @author jiangyin
     * @date 2022/11/9 15:55
     */
    private boolean checkDateConfig(TpmAccountReconciliationRuleDto reqVo) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        //根据生成日期类型判断时间
        try {
            Date startDate = sdf.parse(reqVo.getStartDate());
            Date endDate = sdf.parse(reqVo.getEndDate());
            Calendar startDateCalendar = Calendar.getInstance();
            startDateCalendar.setTime(startDate);
            Calendar endDateCalendar = Calendar.getInstance();
            endDateCalendar.setTime(endDate);
            //年度
            if (GeneratePeriodEnum.YEAR.getCode().equals(reqVo.getGeneratePeriod())) {

                Integer startYear = startDateCalendar.get(Calendar.YEAR);
                Integer endYear = endDateCalendar.get(Calendar.YEAR);

                if (reqVo.getDateConfigList().size() != endYear - startYear + 1) {
                    throw new UnsupportedOperationException("时间设置数据与有效时间跨度不匹配");
                }
            }
            //月度
            else if (GeneratePeriodEnum.MONTH.getCode().equals(reqVo.getGeneratePeriod())) {
                int monthNum = 0;

                startDateCalendar.setTime(startDate);
                endDateCalendar.setTime(endDate);

                //开始日期的日若大于结束日期的日，需要默认加1个月
                if (startDateCalendar.get(Calendar.DAY_OF_MONTH) > endDateCalendar.get(Calendar.DAY_OF_MONTH)) {
                    monthNum += 1;
                }

                while (!startDateCalendar.getTime().after(endDate)) {
                    monthNum += 1;
                    startDateCalendar.add(Calendar.MONTH, 1);
                }

                if (reqVo.getDateConfigList().size() != monthNum) {
                    throw new UnsupportedOperationException("时间设置数据与有效时间跨度不匹配");
                }
            }
            //季度
            else if (GeneratePeriodEnum.QUARTER.getCode().equals(reqVo.getGeneratePeriod())) {
                Set<String> quarterList = new HashSet<>();
                while (!startDateCalendar.getTime().after(endDate)) {
                    quarterList.add(String.valueOf(startDateCalendar.get(Calendar.YEAR)) + getQuarter(startDateCalendar.get(Calendar.MONTH)));
                    startDateCalendar.add(Calendar.MONTH, 1);
                }
                quarterList.add(String.valueOf(endDateCalendar.get(Calendar.YEAR)) + getQuarter(endDateCalendar.get(Calendar.MONTH)));

                if (reqVo.getDateConfigList().size() != quarterList.size()) {
                    throw new UnsupportedOperationException("时间设置数据与有效时间跨度不匹配");
                }
            }
        } catch (ParseException parseException) {
            throw new UnsupportedOperationException("日期格式异常");
        }
        return true;
    }

    /**
     * 根据月获得季度
     *
     * @param month 月
     * @return 季度
     */
    private int getQuarter(int month) {
        if (month == 1 || month == 2 || month == 3) {
            return 1;
        } else if (month == 4 || month == 5 || month == 6) {
            return 2;
        } else if (month == 7 || month == 8 || month == 9) {
            return 3;
        } else {
            return 4;
        }
    }
}
