package com.biz.crm.tpm.business.account.subject.local.imports;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.cost.center.sdk.service.CostCenterVoService;
import com.biz.crm.mdm.business.cost.center.sdk.vo.CostCenterVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.tpm.business.account.subject.local.entity.CustomerOwnership;
import com.biz.crm.tpm.business.account.subject.local.imports.vo.CustomerOwnershipImportVo;
import com.biz.crm.tpm.business.account.subject.sdk.dto.CustomerOwnershipDto;
import com.biz.crm.tpm.business.account.subject.sdk.service.CustomerOwnershipService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 客户归属关系导入
 * @author liyang
 * @date 2023/12/15
 */
@Slf4j
@Component
public class CustomerOwnershipImportProcess implements ImportProcess<CustomerOwnershipImportVo> {

  @Autowired
  private CustomerOwnershipService customerOwnershipService;
  @Autowired
  private SalesOrgVoService salesOrgVoService;
  @Autowired
  private CustomerChannelVoService customerChannelVoService;
  @Autowired
  private CustomerVoService customerVoService;
  @Autowired
  private CostCenterVoService costCenterVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Map<Integer, String> execute(LinkedHashMap<Integer, CustomerOwnershipImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
    try {
      Validate.notEmpty(data, "导入数据不能为空！");
      Map<Integer, String> errorMap = new HashMap<>();
      List<CustomerOwnershipDto> importList = this.validate(data, errorMap);
      if (errorMap.size() > 0) {
        return errorMap;
      }
      if (!CollectionUtils.isEmpty(importList)) {
        customerOwnershipService.importSave(importList);
      }
    } catch (Exception e) {
      e.printStackTrace();
      if (StringUtils.isEmpty(e.getMessage())){
        throw new IllegalArgumentException("空指针异常");
      }else{
        throw new IllegalArgumentException(e.getMessage());
      }
    }
    return null;
  }

  private List<CustomerOwnershipDto> validate(LinkedHashMap<Integer, CustomerOwnershipImportVo> data, Map<Integer, String> errorMap) {

    List<CustomerOwnershipImportVo> rowDataList = new ArrayList<>(data.values());

    Set<String> channelCodeSet = new HashSet<>(rowDataList.size());
    Set<String> salesInstitutionCodeSet = new HashSet<>(rowDataList.size());
    Set<String> customerCodeSet = new HashSet<>(rowDataList.size());
    Set<String> customerProfitLossCodeSet = new HashSet<>(rowDataList.size());
    Set<String> costCenterCodeSet = new HashSet<>(rowDataList.size());
    for (int i = 0; i < rowDataList.size(); i++) {
      CustomerOwnershipImportVo importVo = rowDataList.get(i);

      String businessFormatCode = importVo.getBusinessFormatCode();
      if (businessFormatCode == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("业态编码未传").concat(","));
      }

      String channelCode = importVo.getChannelCode();
      if (channelCode == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("渠道编码未传").concat(","));
      } else {
        channelCodeSet.add(channelCode);
      }

      String salesInstitutionErpCode = importVo.getSalesInstitutionErpCode();
      if (salesInstitutionErpCode == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("销售机构编码未传").concat(","));
      }

      String customerErpCode = importVo.getCustomerErpCode();
      if (customerErpCode == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("客户编码未传").concat(","));
      }

      String customerProfitLossErpCode = importVo.getCustomerProfitLossErpCode();
      if (customerProfitLossErpCode == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("预算所属客户编码未传").concat(","));
      }

      String costCenterCode = importVo.getCostCenterCode();
      if (costCenterCode == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("成本中心编码未传").concat(","));
      } else {
        costCenterCodeSet.add(costCenterCode);
      }

      String platform = importVo.getPlatform();
      if (platform == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("平台未传").concat(","));
      }

      // 销售组织长码
      String salesInstitutionCode = importVo.getChannelCode() + importVo.getBusinessFormatCode() + importVo.getSalesInstitutionErpCode();
      importVo.setSalesInstitutionCode(salesInstitutionCode);
      salesInstitutionCodeSet.add(importVo.getSalesInstitutionCode());

      // 客户长码
      String customerCode = importVo.getCustomerErpCode() + importVo.getSalesInstitutionErpCode() + importVo.getChannelCode() + importVo.getBusinessFormatCode();
      importVo.setCustomerCode(customerCode);
      customerCodeSet.add(importVo.getCustomerCode());

      // 预算所属客户长码
      String customerProfitLossCode = importVo.getCustomerProfitLossErpCode()+importVo.getSalesInstitutionErpCode()+importVo.getChannelCode()+importVo.getBusinessFormatCode();
      importVo.setCustomerProfitLossCode(customerProfitLossCode);
      customerProfitLossCodeSet.add(importVo.getCustomerProfitLossCode());
    }

    // 查询是否已有数据
    List<String> existCostCenterCodes = customerOwnershipService.checkExist(Lists.newArrayList(costCenterCodeSet));

    // 销售组织
    Map<String, String> salesOrgVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(salesInstitutionCodeSet)) {
      List<SalesOrgVo> salesOrgVos = salesOrgVoService.findBySalesOrgCodes(new ArrayList<>(salesInstitutionCodeSet));
      if (!CollectionUtils.isEmpty(salesOrgVos)) {
        salesOrgVoMap.putAll(salesOrgVos.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, SalesOrgVo::getSalesOrgName)));
      }
    }

    // 渠道
    Map<String, String> customerChannelVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(channelCodeSet)) {
      List<CustomerChannelVo> customerChannelVos = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
      if (!CollectionUtils.isEmpty(customerChannelVos)) {
        customerChannelVoMap.putAll(customerChannelVos.stream().collect(Collectors.toMap(CustomerChannelVo::getCustomerChannelCode, CustomerChannelVo::getCustomerChannelName)));
      }
    }

    // 客户
    Map<String, String> customerVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(customerCodeSet)) {
      List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
      if (!CollectionUtils.isEmpty(customerVos)) {
        customerVoMap.putAll(customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, CustomerVo::getCustomerName)));
      }
    }

    // 预算所属客户
    Map<String, String> customerProfitLossVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(customerProfitLossCodeSet)) {
      List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerProfitLossCodeSet));
      if (!CollectionUtils.isEmpty(customerVos)) {
        customerProfitLossVoMap.putAll(customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, CustomerVo::getCustomerName)));
      }
    }

    // 成本中心
    Map<String, String> costCenterCodeVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(costCenterCodeSet)) {
      List<CostCenterVo> costCenterVoList = costCenterVoService.findByCodeList(new ArrayList<>(costCenterCodeSet));
      if (!CollectionUtils.isEmpty(costCenterVoList)) {
        costCenterCodeVoMap.putAll(costCenterVoList.stream().collect(Collectors.toMap(CostCenterVo::getCostCenterCode, CostCenterVo::getCostCenterName)));
      }
    }

    for (int i = 0; i < rowDataList.size(); i++) {
      CustomerOwnershipImportVo importVo = rowDataList.get(i);

      String channelName = customerChannelVoMap.get(importVo.getChannelCode());
      if (channelName == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("渠道名称不存在").concat(","));
      } else importVo.setChannelName(channelName);

      String salesInstitutionName = salesOrgVoMap.get(importVo.getSalesInstitutionCode());
      if (salesInstitutionName == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("销售机构名称不存在").concat(","));
      } else importVo.setSalesInstitutionName(salesInstitutionName);

      String customerName = customerVoMap.get(importVo.getCustomerCode());
      if (customerName == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("客户名称不存在").concat(","));
      } else importVo.setCustomerName(customerName);

      String customerProfitLossName = customerProfitLossVoMap.get(importVo.getCustomerProfitLossCode());
      if (customerProfitLossName == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("预算所属客户名称不存在").concat(","));
      } else importVo.setCustomerProfitLossName(customerProfitLossName);

      String costCenterName = costCenterCodeVoMap.get(importVo.getCostCenterCode());
      if (costCenterName == null) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("成本中心名称不存在").concat(","));
      } else importVo.setCostCenterName(costCenterName);

      if (existCostCenterCodes.contains(importVo.getCostCenterCode())) {
        errorMap.put(i, this.initNullString(errorMap.get(i)).concat("该成本中心编码已存在客户归属关系不可重复").concat(","));
      }
    }

    return (List<CustomerOwnershipDto>) nebulaToolkitService.copyCollectionByWhiteList(
        rowDataList,
        CustomerOwnershipImportVo.class, CustomerOwnershipDto.class,
        HashSet.class, ArrayList.class);
  }

  private String initNullString(String s) {
    return s == null ? "" : s;
  }

  @Override
  public Class<CustomerOwnershipImportVo> findCrmExcelVoClass() {
    return CustomerOwnershipImportVo.class;
  }

  @Override
  public String getTemplateCode() {
    return "TPM_CUSTOMER_OWNERSHIP_IMPORT";
  }

  @Override
  public String getTemplateName() {
    return "TPM客户归属关系导入";
  }
}
