package com.biz.crm.tpm.business.account.subject.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.sap.fi.sdk.dto.SapFiAccountSubjectDto;
import com.biz.crm.mn.third.system.sap.fi.sdk.service.SapFiService;
import com.biz.crm.mn.third.system.sap.fi.sdk.vo.SapFiAccountSubjectVo;
import com.biz.crm.tpm.business.account.subject.local.entity.AccountSubjectEntity;
import com.biz.crm.tpm.business.account.subject.local.mapper.AccountSubjectMapper;
import com.biz.crm.tpm.business.account.subject.local.repository.AccountSubjectRepository;
import com.biz.crm.tpm.business.account.subject.sdk.constants.AccountSubjectConstant;
import com.biz.crm.tpm.business.account.subject.sdk.dto.AccountSubjectDto;
import com.biz.crm.tpm.business.account.subject.sdk.dto.AccountSubjectSelectDto;
import com.biz.crm.tpm.business.account.subject.sdk.service.AccountSubjectService;
import com.biz.crm.tpm.business.account.subject.sdk.vo.AccountSubjectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月27日 16:43
 */
@Slf4j
@Service
public class AccountSubjectServiceImpl implements AccountSubjectService {

    @Autowired(required = false)
    private AccountSubjectMapper accountSubjectMapper;

    @Autowired(required = false)
    private AccountSubjectRepository accountSubjectRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private SapFiService sapFiService;

    /**
     * 分页查询会计科目
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.account.subject.sdk.vo.AccountSubjectVo>
     * @author huojia
     * @date 2022/11/1 21:02
     **/
    @Override
    public Page<AccountSubjectVo> findByConditions(Pageable pageable, AccountSubjectDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new AccountSubjectDto();
        }
        Page<AccountSubjectVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.accountSubjectMapper.findByConditions(page, dto);
    }

    @Override
    public List<AccountSubjectVo> findAllList(AccountSubjectSelectDto dto) {

        return this.accountSubjectRepository.findSelectList(dto);
    }

    @Override
    public List<AccountSubjectVo> findListByCode(List<String> codeList) {

        List<AccountSubjectEntity> accountSubjectList = this.accountSubjectRepository.findBySupplierCodes(codeList);

        if (accountSubjectList.isEmpty() || accountSubjectList.size() <= 0){
            return null;
        }

        return (List<AccountSubjectVo>) this.nebulaToolkitService.copyCollectionByWhiteList(accountSubjectList
                , AccountSubjectEntity.class
                , AccountSubjectVo.class
                , HashSet.class
                , ArrayList.class);
    }

    /**
     * 批量拉取会计科目
     *
     * @author huojia
     * @date 2022/12/7 22:03
     **/
    @Override
    public void pullAccountSubjectList() {
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            List<SapFiAccountSubjectVo> sapFiAccountSubjectVos = sapFiService.pullAccountSubjectList(new SapFiAccountSubjectDto());
            if (CollectionUtils.isEmpty(sapFiAccountSubjectVos)) {
                return;
            }
            // 编码不会重复，如果存在重复情况，需要到对方系统检查
            Map<String, String> codeMap = new HashMap<>(8);
            List<AccountSubjectEntity> pullList = new ArrayList<>();
            sapFiAccountSubjectVos.forEach(sapFiAccountSubjectVo -> {
                if (StringUtils.isEmpty(sapFiAccountSubjectVo.getSAKNR())) {
                    log.info("本次拉取数据[{}]会计科目编码不能为空，请检查！", sapFiAccountSubjectVos);
                    return;
                }
                if (codeMap.containsKey(sapFiAccountSubjectVo.getSAKNR())) {
                    log.info("本次拉取数据[{}]重复拉取，请检查！", sapFiAccountSubjectVos);
                    return;
                }
                codeMap.put(sapFiAccountSubjectVo.getSAKNR(), "");
                AccountSubjectEntity accountSubjectEntity = new AccountSubjectEntity();
                accountSubjectEntity.setAccountSubjectCode(sapFiAccountSubjectVo.getSAKNR());
                accountSubjectEntity.setAccountSubjectName(sapFiAccountSubjectVo.getTXT50());
                accountSubjectEntity.setTenantCode(TenantUtils.getTenantCode());
                accountSubjectEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                accountSubjectEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                pullList.add(accountSubjectEntity);
            });
            List<String> accountSubjectCodeList = pullList.stream()
                    .map(AccountSubjectEntity::getAccountSubjectCode)
                    .collect(Collectors.toList());
            List<AccountSubjectEntity> byCodes = accountSubjectRepository.findBySupplierCodes(accountSubjectCodeList);
            if (CollectionUtils.isEmpty(byCodes)) {
                this.saveOrUpdateSapBatch(pullList, null);
                return;
            }
            // 区分更新、新增
            List<AccountSubjectEntity> saveList = new ArrayList<>();
            List<AccountSubjectEntity> updateList = new ArrayList<>();
            Map<String, AccountSubjectEntity> map = byCodes.stream()
                    .collect(Collectors.toMap(AccountSubjectEntity::getAccountSubjectCode, Function.identity()));
            pullList.forEach(pull -> {
                if (map.containsKey(pull.getAccountSubjectCode())) {
                    AccountSubjectEntity accountSubjectEntity = map.get(pull.getAccountSubjectCode());
                    accountSubjectEntity.setAccountSubjectCode(pull.getAccountSubjectCode());
                    accountSubjectEntity.setAccountSubjectName(pull.getAccountSubjectName());
                    updateList.add(accountSubjectEntity);
                } else {
                    saveList.add(pull);
                }
            });
            this.saveOrUpdateSapBatch(saveList, updateList);
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    @Override
    public List<AccountSubjectVo> findAccountSubjectByCode(List<String> codes) {

        List<AccountSubjectEntity> list = this.accountSubjectRepository.findBySupplierCodes(codes);

        return (List<AccountSubjectVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list
                , AccountSubjectEntity.class
                , AccountSubjectVo.class
                , HashSet.class
                , ArrayList.class);

    }

    /**
     * 批量保存SAP会计科目数据
     *
     * @param saveList
     * @param updateList
     * @author huojia
     * @date 2023/1/3 17:54
     **/
    @Transactional(rollbackFor = Exception.class)
    void saveOrUpdateSapBatch(List<AccountSubjectEntity> saveList, List<AccountSubjectEntity> updateList) {
        if (!CollectionUtils.isEmpty(saveList)) {
            this.accountSubjectRepository.saveBatch(saveList);
        }
        if (!CollectionUtils.isEmpty(updateList)) {
            this.accountSubjectRepository.updateBatchById(updateList);
        }
    }

    /**
     * 按年月日加锁
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    public boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取会计科目加锁失败，日期不能为空！");
        }
        return this.redisLockService.tryLock(AccountSubjectConstant.ACCOUNT_SUBJECT_LOCK + yearMonthDay, TimeUnit.HOURS, 12);
    }

    /**
     * 按年月日解锁
     *
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    public void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取会计科目解锁失败，日期不能为空！");
        }
        redisLockService.unlock(AccountSubjectConstant.ACCOUNT_SUBJECT_LOCK + yearMonthDay);
    }
}
