package com.biz.crm.tpm.business.activity.apply.rules.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.tpm.business.activity.apply.rules.local.entity.ActivityApplyRules;
import com.biz.crm.tpm.business.activity.apply.rules.local.repository.ActivityApplyRulesRepository;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.constant.ActivityApplyRulesConstant;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.*;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.log.ActivityApplyRulesLogEventDto;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.enums.PromotionPlanActFormEnum;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.event.log.ActivityApplyRulesEventListener;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.service.ActivityApplyRulesService;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.service.VariableService;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.variable.register.FormulaVariableRegister;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.vo.ActivityApplyRulesFormulaVariableVo;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.vo.ActivityApplyRulesPromotionPlanAmountVo;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.vo.ActivityApplyRulesVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.GeneralExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PurchaseSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2022/11/8 15:38
 * @description: 活动申请规则表(ActivityApplyRules)表服务实现类
 */
@Slf4j
@Service("activityApplyRulesService")
public class ActivityApplyRulesServiceImpl implements ActivityApplyRulesService {

    @Autowired(required = false)
    private ActivityApplyRulesRepository activityApplyRulesRepository;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private List<FormulaVariableRegister> formulaVariableRegisterList;
    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private VariableService variableService;
    @Autowired(required = false)
    private CurrentMonthSaleService currentMonthSaleService;
    @Autowired(required = false)
    private CurrentMonthDeliveryService currentMonthDeliveryService;
    @Autowired(required = false)
    private GeneralExpensesService generalExpensesService;
    @Autowired(required = false)
    private PurchaseSaleService purchaseSaleService;
    @Autowired(required = false)
    private PromotionPlanService promotionPlanService;

    @Autowired(required = false)
    private LoginUserService loginUserService;


    /**
     * 分页查询数据
     *
     * @param pageable              分页对象
     * @param activityApplyRulesDto 查询Dto
     * @return Page<ActivityApplyRulesVo>
     */
    @Override
    public Page<ActivityApplyRulesVo> findByConditions(Pageable pageable, ActivityApplyRulesDto activityApplyRulesDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(activityApplyRulesDto)) {
            activityApplyRulesDto = new ActivityApplyRulesDto();
        }
        return this.activityApplyRulesRepository.findByConditions(pageable, activityApplyRulesDto);
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public ActivityApplyRulesVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        ActivityApplyRules applyRules = this.activityApplyRulesRepository.getById(id);
        if (null == applyRules) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(applyRules, ActivityApplyRulesVo.class, null, null);
    }

    /**
     * 新增数据
     *
     * @param activityApplyRulesDto 实体对象
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(ActivityApplyRulesDto activityApplyRulesDto) {
        this.createValidate(activityApplyRulesDto);
        activityApplyRulesDto.setCreateAccount(loginUserService.getLoginAccountName());
        //判断数据重复
//        int count = this.activityApplyRulesRepository.checkDuplicate(activityApplyRulesDto);
        int count = this.activityApplyRulesRepository.lambdaQuery()
                .eq(ActivityApplyRules::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityApplyRules::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityApplyRules::getOnlyKey, activityApplyRulesDto.getOnlyKey())
                .count();
        // Validate.isTrue(count == 0, "已存在相同维度的申请规则，请勿重复添加");

        ActivityApplyRules applyRules = nebulaToolkitService.copyObjectByWhiteList(activityApplyRulesDto, ActivityApplyRules.class, null, null);

//        String ruleCode = StringUtils.join(ActivityApplyRulesConstant.ACTIVITY_APPLY_RULES_PREFIX, DateFormatUtils.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH));
        String code = this.generateCodeService.generateCode(ActivityApplyRulesConstant.ACTIVITY_APPLY_RULES_PREFIX, 1, 5, 2, TimeUnit.DAYS).get(0);
        applyRules.setApplyRulesCode(code);
        applyRules.setTenantCode(TenantUtils.getTenantCode());
        applyRules.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        applyRules.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        this.activityApplyRulesRepository.save(applyRules);

        //新增业务日志
        ActivityApplyRulesLogEventDto logEventDto = new ActivityApplyRulesLogEventDto();
        logEventDto.setOriginal(null);
        activityApplyRulesDto.setId(applyRules.getId());
        logEventDto.setNewest(activityApplyRulesDto);
        SerializableBiConsumer<ActivityApplyRulesEventListener, ActivityApplyRulesLogEventDto> onCreate =
                ActivityApplyRulesEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, ActivityApplyRulesEventListener.class, onCreate);
    }

    /**
     * 创建验证
     *
     * @param activityApplyRulesDto dto对象
     */
    private void createValidate(ActivityApplyRulesDto activityApplyRulesDto) {
        Validate.notNull(activityApplyRulesDto, "新增时，对象信息不能为空！");
        activityApplyRulesDto.setId(null);
        Validate.notBlank(activityApplyRulesDto.getApplyRulesName(), "新增时，申请规则名称不能为空！");
        Validate.notBlank(activityApplyRulesDto.getActivityFormCode(), "新增时，活动形式编码不能为空！");
        Validate.notBlank(activityApplyRulesDto.getActivityFormName(), "新增时，活动形式名称不能为空！");
        Validate.notBlank(activityApplyRulesDto.getSalesOrgCode(), "新增时，销售部门编码不能为空！");
        Validate.notBlank(activityApplyRulesDto.getSalesOrgName(), "新增时，销售部门名称不能为空！");
//        Validate.notBlank(activityApplyRulesDto.getPlatformCode(), "新增时，平台不能为空！");
        Validate.notBlank(activityApplyRulesDto.getAmountFormula(), "新增时，申请金额计算公式不能为空！");
        Validate.notNull(activityApplyRulesDto.getAmountFormulaName(), "新增时，申请金额计算公式（展示用）不能为空！");
        String onlyKey = activityApplyRulesDto.getActivityFormCode() + activityApplyRulesDto.getSalesOrgCode();

        if (StringUtils.isNotBlank(activityApplyRulesDto.getPlatformCode())) {
            onlyKey = onlyKey + activityApplyRulesDto.getPlatformCode();
        }
        if (StringUtils.isNotBlank(activityApplyRulesDto.getCustomerCode())) {
            onlyKey = onlyKey + activityApplyRulesDto.getCustomerCode();
        }
        activityApplyRulesDto.setOnlyKey(onlyKey);
    }

    /**
     * 修改新据
     *
     * @param dto 实体对象
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(ActivityApplyRulesDto dto) {
        this.updateValidate(dto);
        //判断数据重复
//        int count = this.activityApplyRulesRepository.checkDuplicate(activityApplyRulesDto);
        int count = this.activityApplyRulesRepository.lambdaQuery()
                .eq(ActivityApplyRules::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityApplyRules::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityApplyRules::getOnlyKey, dto.getOnlyKey())
                .ne(ActivityApplyRules::getId, dto.getId())
                .count();
        Validate.isTrue(count == 0, "已存在相同维度的申请规则，请勿重复添加");
        //获取历史vo
        ActivityApplyRulesVo oldVo = this.findById(dto.getId());
        dto.setModifyAccount(loginUserService.getLoginAccountName());
        ActivityApplyRules applyRules = nebulaToolkitService.copyObjectByBlankList(dto, ActivityApplyRules.class, null, null);
        this.activityApplyRulesRepository.updateById(applyRules);
        //修改业务日志
        ActivityApplyRulesLogEventDto logEventDto = new ActivityApplyRulesLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<ActivityApplyRulesEventListener, ActivityApplyRulesLogEventDto> onUpdate =
                ActivityApplyRulesEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, ActivityApplyRulesEventListener.class, onUpdate);
    }

    /**
     * 编辑验证
     *
     * @param activityApplyRulesDto dto对象
     */
    private void updateValidate(ActivityApplyRulesDto activityApplyRulesDto) {
        Validate.notNull(activityApplyRulesDto, "编辑时，对象信息不能为空！");
        Validate.notBlank(activityApplyRulesDto.getId(), "编辑数据时，id不能为空！");
        Validate.notBlank(activityApplyRulesDto.getApplyRulesName(), "编辑数据时，申请规则名称不能为空！");
        Validate.notBlank(activityApplyRulesDto.getActivityFormCode(), "编辑数据时，活动形式编码不能为空！");
        Validate.notBlank(activityApplyRulesDto.getActivityFormName(), "编辑数据时，活动形式名称不能为空！");
        Validate.notBlank(activityApplyRulesDto.getSalesOrgCode(), "编辑数据时，销售部门编码不能为空！");
        Validate.notBlank(activityApplyRulesDto.getSalesOrgName(), "编辑数据时，销售名称不能为空！");
        Validate.notBlank(activityApplyRulesDto.getAmountFormula(), "编辑数据时，申请金额计算公式不能为空！");
        Validate.notNull(activityApplyRulesDto.getAmountFormulaName(), "编辑数据时，申请金额计算公式（展示用）不能为空！");
        Validate.notBlank(activityApplyRulesDto.getAmountFormula(), "新增时，申请金额计算公式不能为空！");
        Validate.notNull(activityApplyRulesDto.getAmountFormulaName(), "新增时，申请金额计算公式（展示用）不能为空！");
        String onlyKey = activityApplyRulesDto.getActivityFormCode() + activityApplyRulesDto.getSalesOrgCode();
        if (StringUtils.isNotBlank(activityApplyRulesDto.getPlatformCode())) {
            onlyKey = onlyKey + activityApplyRulesDto.getPlatformCode();
        }
        if (StringUtils.isNotBlank(activityApplyRulesDto.getCustomerCode())) {
            onlyKey = onlyKey + activityApplyRulesDto.getCustomerCode();
        }
        activityApplyRulesDto.setOnlyKey(onlyKey);
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<ActivityApplyRules> rulesList = this.activityApplyRulesRepository.listByIds(idList);
        Validate.notEmpty(rulesList,"要删除的数据不存在！");
        rulesList.forEach(item -> {

            //删除业务日志
            ActivityApplyRulesLogEventDto logEventDto = new ActivityApplyRulesLogEventDto();
            ActivityApplyRulesVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, ActivityApplyRulesVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            ActivityApplyRulesDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, ActivityApplyRulesDto.class, null, null);
            newDto.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<ActivityApplyRulesEventListener, ActivityApplyRulesLogEventDto> onDelete =
                    ActivityApplyRulesEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, ActivityApplyRulesEventListener.class, onDelete);
        });
        List<String> ids = rulesList.stream().map(ActivityApplyRules::getId).filter(Objects::nonNull).collect(Collectors.toList());
        this.activityApplyRulesRepository.deleteByIds(ids);
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要操作的数据");
        List<ActivityApplyRules> activityApplyRuless = this.activityApplyRulesRepository.listByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(activityApplyRuless), "不存在或已删除!");

        ArrayList<ActivityApplyRules> activityApplyRuleList = new ArrayList<>();
        activityApplyRuless.forEach(item -> {
            ActivityApplyRules activityApplyRules = new ActivityApplyRules();
            activityApplyRules.setId(item.getId());
            activityApplyRules.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            activityApplyRuleList.add(activityApplyRules);

            //启用业务日志
            ActivityApplyRulesLogEventDto logEventDto = new ActivityApplyRulesLogEventDto();
            ActivityApplyRulesVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, ActivityApplyRulesVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            ActivityApplyRulesDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, ActivityApplyRulesDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<ActivityApplyRulesEventListener, ActivityApplyRulesLogEventDto> onEnable =
                    ActivityApplyRulesEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, ActivityApplyRulesEventListener.class, onEnable);
        });
        this.activityApplyRulesRepository.updateBatchById(activityApplyRuleList);
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要操作的数据");
        List<ActivityApplyRules> activityApplyRuless = this.activityApplyRulesRepository.listByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(activityApplyRuless), "不存在或已删除!");

        ArrayList<ActivityApplyRules> activityApplyRuleList = new ArrayList<>();
        activityApplyRuless.forEach(item -> {
            ActivityApplyRules activityApplyRules = new ActivityApplyRules();
            activityApplyRules.setId(item.getId());
            activityApplyRules.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            activityApplyRuleList.add(activityApplyRules);

            //禁用业务日志
            ActivityApplyRulesLogEventDto logEventDto = new ActivityApplyRulesLogEventDto();
            ActivityApplyRulesVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, ActivityApplyRulesVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            ActivityApplyRulesDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, ActivityApplyRulesDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<ActivityApplyRulesEventListener, ActivityApplyRulesLogEventDto> onDisable =
                    ActivityApplyRulesEventListener::onDisable;
            this.nebulaNetEventClient.publish(logEventDto, ActivityApplyRulesEventListener.class, onDisable);
        });
        this.activityApplyRulesRepository.updateBatchById(activityApplyRuleList);
    }

    /**
     * 获取申请规则公式变量
     *
     * @return 列表数据
     */
    @Override
    public List<ActivityApplyRulesFormulaVariableVo> getFormulaVariable() {
        List<ActivityApplyRulesFormulaVariableVo> voList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(formulaVariableRegisterList)) {
            formulaVariableRegisterList.forEach(register -> {
                ActivityApplyRulesFormulaVariableVo vo = new ActivityApplyRulesFormulaVariableVo();
                vo.setCode(register.getVariableCode());
                vo.setName(register.getVariableName());
                vo.setSort(register.getSort());
                voList.add(vo);
            });
        }
        if (!CollectionUtils.isEmpty(voList)) {
            return voList.stream().sorted(Comparator.comparing(ActivityApplyRulesFormulaVariableVo::getSort)).collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    /**
     * 当月销售计算规则金额
     * 注：若促销规划里的活动形式更改字段，则枚举(PromotionPlanActFormEnum)中也需要更改，字典(promotion_plan_act_form)也需要更改
     *
     * @param dto 规则匹配参数
     * @return ActivityApplyRulesPromotionPlanAmountVo
     */
    @Override
    public ActivityApplyRulesPromotionPlanAmountVo calAmountMonthSale(MonthSalePromotionPlanAmountDto dto) {
        this.calMonthSaleValidate(dto);
        //获取字典中的活动形式编码
        //获取促销规划活动形式编码配置字典
        Map<String, String> formMap = dictToolkitService.findMapByDictTypeCode(ActivityApplyRulesConstant.PROMOTION_PLAN_ACT_FORM2);
        if (formMap.isEmpty()) {
            return new ActivityApplyRulesPromotionPlanAmountVo();
        }
        //获取活动形式编码
        List<String> formCodes = new ArrayList<>();
        //产品促销（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.PRODUCT_PROMOTION, formCodes);
        //大日期处理（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.LARGE_DATE_PROCESS, formCodes);
        //新客专享（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.NEW_CUSTOMER, formCodes);
        //老客留存（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.OLD_CUSTOMER, formCodes);
        //达人佣金（抖音）（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.TIKTOK_MASTER, formCodes);
        //平台毛保（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.PLATFORM_GROSS_PROTECTION, formCodes);
        //平台服务费（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.PLATFORM_SERVICE, formCodes);
        //平台返点费用（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.PLATFORM_REBATE, formCodes);
        //平台佣金扣点（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.PLATFORM_COMMISSION, formCodes);
        //销售返点（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.SALE_COMMISSION, formCodes);
        //物流费用（元）
//        this.getFormCode(formMap, PromotionPlanActFormEnum.LOGISTICS, formCodes);
        //渠道推广费（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.CHANNEL_PROMOTION_FEE, formCodes);
        //仓间不均（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.UNEVEN, formCodes);
        //高周转（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.HIGH_TURNOVER, formCodes);

        if (CollectionUtils.isEmpty(formCodes)) {
            return new ActivityApplyRulesPromotionPlanAmountVo();
        }
        //查询申请规则
        ActivityApplyRulesPromotionPlanAmountDto amountDto = new ActivityApplyRulesPromotionPlanAmountDto();
        amountDto.setTenantCode(TenantUtils.getTenantCode());
        amountDto.setSalesOrgCode(dto.getSalesOrgCode());
        amountDto.setPlatformCode(dto.getPlatformCode());
        amountDto.setCustomerCode(dto.getCustomerCode());
        amountDto.setEstoreChannel(dto.getChannelCode());
        Map<String, Map<String, ActivityApplyRules>> applyRulesMap = this.getActivityApplyRuleMap(amountDto, formCodes);

        CalculateDto calculateDto = new CalculateDto();
        calculateDto.setTaxRate(dto.getTaxRate());
        calculateDto.setRatio(dto.getRatio());
        calculateDto.setStandardRetailPrice(dto.getStandardRetailPrice());
        calculateDto.setPlatformSupplyPrice(dto.getPlatformSupplyPrice());
        calculateDto.setActivityBasePrice(dto.getActivityBasePrice());
        calculateDto.setCombinationQuantity(dto.getCombinationQuantity());
        calculateDto.setEstimatedSalesBox(dto.getEstimatedSalesBox());
        calculateDto.setEstimatedSalesTon(dto.getEstimatedSalesTon());
        calculateDto.setVat(dto.getVat());
        calculateDto.setEstimatedAmountBefore(dto.getEstimatedAmountBefore());
        calculateDto.setEstimatedAmountAfter(dto.getEstimatedAmountAfter());
        calculateDto.setCostPrice(dto.getCostPrice());
        calculateDto.setGmv(dto.getGmv());
        calculateDto.setFormMap(formMap);
        //当月销售
        if (!CollectionUtils.isEmpty(dto.getCurrentMonthSaleDtoList())) {
            calculateDto.setMonthSaleServiceCacheList(dto.getCurrentMonthSaleDtoList());
        } else {
            if (StringUtils.isNotBlank(dto.getMonthSaleCacheKey())) {
                List<CurrentMonthSaleDto> salesList = currentMonthSaleService.findCacheList(dto.getMonthSaleCacheKey());
                if (CollectionUtils.isEmpty(salesList)) {
                    if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                        calculateDto.setMonthSaleServiceCacheList(currentMonthSaleService.findList(dto.getPromotionPlanCode()));
                    }
                } else {
                    calculateDto.setMonthSaleServiceCacheList(salesList);
                }
            }
        }
        //费用申请
        if (!CollectionUtils.isEmpty(dto.getExpensesDtoList())) {
            calculateDto.setExpensesServiceCacheList(dto.getExpensesDtoList());
        } else {
            if (StringUtils.isNotBlank(dto.getGeneralExpensesCacheKey())) {
                List<GeneralExpensesDto> expensesDtoList = generalExpensesService.findCacheList(dto.getGeneralExpensesCacheKey());
                if (CollectionUtils.isEmpty(expensesDtoList)) {
                    if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                        calculateDto.setExpensesServiceCacheList(generalExpensesService.findList(dto.getPromotionPlanCode()));
                    }
                } else {
                    calculateDto.setExpensesServiceCacheList(expensesDtoList);
                }
            }
        }
        ActivityApplyRulesPromotionPlanAmountVo amountVo = new ActivityApplyRulesPromotionPlanAmountVo();
        //产品促销（元）
        amountVo.setProductPromotion(this.packageFormValue(formMap, PromotionPlanActFormEnum.PRODUCT_PROMOTION, applyRulesMap, calculateDto));
        //大日期处理（元）
        amountVo.setLargeDateProcess(this.packageFormValue(formMap, PromotionPlanActFormEnum.LARGE_DATE_PROCESS, applyRulesMap, calculateDto));
        //新客专享（元）
        amountVo.setNewCustomer(this.packageFormValue(formMap, PromotionPlanActFormEnum.NEW_CUSTOMER, applyRulesMap, calculateDto));
        //老客留存（元）
        amountVo.setOldCustomer(this.packageFormValue(formMap, PromotionPlanActFormEnum.OLD_CUSTOMER, applyRulesMap, calculateDto));
        //达人佣金（抖音）（元）
        amountVo.setTiktokMaster(this.packageFormValue(formMap, PromotionPlanActFormEnum.TIKTOK_MASTER, applyRulesMap, calculateDto));
        //平台毛保（元）
        amountVo.setPlatformGrossProtection(this.packageFormValue(formMap, PromotionPlanActFormEnum.PLATFORM_GROSS_PROTECTION, applyRulesMap, calculateDto));
        //平台服务费（元）
        amountVo.setPlatformService(this.packageFormValue(formMap, PromotionPlanActFormEnum.PLATFORM_SERVICE, applyRulesMap, calculateDto));
        //平台返点费用（元）
        amountVo.setPlatformRebate(this.packageFormValue(formMap, PromotionPlanActFormEnum.PLATFORM_REBATE, applyRulesMap, calculateDto));
        //平台佣金扣点（元）
        amountVo.setPlatformCommission(this.packageFormValue(formMap, PromotionPlanActFormEnum.PLATFORM_COMMISSION, applyRulesMap, calculateDto));
        //销售返点（元）
        amountVo.setSaleCommission(this.packageFormValue(formMap, PromotionPlanActFormEnum.SALE_COMMISSION, applyRulesMap, calculateDto));

        //渠道推广费（元）
        amountVo.setChannelPromotionFee(this.packageFormValue(formMap, PromotionPlanActFormEnum.CHANNEL_PROMOTION_FEE, applyRulesMap, calculateDto));
        //仓间不均（元）
        amountVo.setUneven(this.packageFormValue(formMap, PromotionPlanActFormEnum.UNEVEN, applyRulesMap, calculateDto));
        //高周转（元）
        amountVo.setHighTurnover(this.packageFormValue(formMap, PromotionPlanActFormEnum.HIGH_TURNOVER, applyRulesMap, calculateDto));
        return amountVo;
    }

    /**
     * 匹配规则
     *
     * @param amountDto
     * @param formCodes
     * @return
     */
    private Map<String, Map<String, ActivityApplyRules>> getActivityApplyRuleMap(ActivityApplyRulesPromotionPlanAmountDto amountDto, List<String> formCodes) {
        if (CollectionUtil.isEmpty(formCodes)) {
            return Collections.emptyMap();
        }
        Map<String, Map<String, ActivityApplyRules>> resultMap = Maps.newLinkedHashMap();
        List<ActivityApplyRules> rulesList = this.activityApplyRulesRepository.getPromotionPlanApplyRulesAll(amountDto, formCodes);
        if (CollectionUtil.isNotEmpty(rulesList)) {
            resultMap.put("1", rulesList.stream().collect(Collectors.toMap(ActivityApplyRules::getActivityFormCode, v -> v, (newValue, oldValue) -> newValue)));
        }

        rulesList = this.activityApplyRulesRepository.getPromotionPlanNoChannelApplyRules(amountDto, formCodes);
        if (CollectionUtil.isNotEmpty(rulesList)) {
            resultMap.put("2", rulesList.stream().collect(Collectors.toMap(ActivityApplyRules::getActivityFormCode, v -> v, (newValue, oldValue) -> newValue)));
        }

        rulesList = this.activityApplyRulesRepository.getPromotionPlanNoCusChannelApplyRules(amountDto, formCodes);
        if (CollectionUtil.isNotEmpty(rulesList)) {
            resultMap.put("3", rulesList.stream().collect(Collectors.toMap(ActivityApplyRules::getActivityFormCode, v -> v, (newValue, oldValue) -> newValue)));
        }

        rulesList = this.activityApplyRulesRepository.getPromotionPlanNoCusPlatformChannelApplyRules(amountDto, formCodes);
        if (CollectionUtil.isNotEmpty(rulesList)) {
            resultMap.put("4", rulesList.stream().collect(Collectors.toMap(ActivityApplyRules::getActivityFormCode, v -> v, (newValue, oldValue) -> newValue)));
        }
        resultMap.putIfAbsent("1", Maps.newHashMap());
        resultMap.putIfAbsent("2", Maps.newHashMap());
        resultMap.putIfAbsent("3", Maps.newHashMap());
        resultMap.putIfAbsent("4", Maps.newHashMap());

        return resultMap;
    }

    /**
     * 获取活动形式编码
     *
     * @param formMap  字典map
     * @param formEnum 促销规划活动形式枚举
     */
    private void getFormCode(Map<String, String> formMap, PromotionPlanActFormEnum formEnum, List<String> formCodes) {
        String formCode = formMap.getOrDefault(formEnum.getCode(), null);
        if (StringUtils.isNotBlank(formCode)) {
            formCodes.add(formCode);
        }
    }

    /**
     * 获取活动形式编码
     *
     * @param formMap  字典map
     * @param formEnum 促销规划活动形式枚举
     */
    private BigDecimal packageFormValue(Map<String, String> formMap, PromotionPlanActFormEnum formEnum,
                                        Map<String, Map<String, ActivityApplyRules>> applyRulesMap,
                                        CalculateDto calculateDto) {

        String formCode = formMap.getOrDefault(formEnum.getCode(), null);
        if (StringUtils.isBlank(formCode)) {
            return BigDecimal.ZERO;
        }
        ActivityApplyRules rules = this.getActivityApplyRules(applyRulesMap, formCode);
        if (Objects.isNull(rules)) {
            return BigDecimal.ZERO;
        }
        calculateDto.setFormula(rules.getAmountFormula());
        return variableService.singleCalculateExpression(calculateDto);
    }

    /**
     * 匹配规则
     * @param applyRulesMap
     * @param formCode
     * @return
     */
    private ActivityApplyRules getActivityApplyRules(Map<String, Map<String, ActivityApplyRules>> applyRulesMap,
                                                     String formCode) {
        if (CollectionUtil.isEmpty(applyRulesMap)
                || StringUtils.isEmpty(formCode)) {
            return null;
        }
        Map<String, ActivityApplyRules> map = applyRulesMap.getOrDefault("1", Maps.newHashMap());
        ActivityApplyRules rules = map.getOrDefault(formCode, null);
        if (Objects.nonNull(rules)) {
            return rules;
        }

        map = applyRulesMap.getOrDefault("2", Maps.newHashMap());
        rules = map.getOrDefault(formCode, null);
        if (Objects.nonNull(rules)) {
            return rules;
        }

        map = applyRulesMap.getOrDefault("3", Maps.newHashMap());
        rules = map.getOrDefault(formCode, null);
        if (Objects.nonNull(rules)) {
            return rules;
        }

        map = applyRulesMap.getOrDefault("4", Maps.newHashMap());
        rules = map.getOrDefault(formCode, null);
        if (Objects.nonNull(rules)) {
            return rules;
        }

        return null;

    }

    /**
     * 当月送货计算规则金额
     *
     * @param dto 规则匹配参数
     * @return ActivityApplyRulesPromotionPlanAmountVo
     */
    @Override
    public ActivityApplyRulesPromotionPlanAmountVo calAmountMonthDelivery(MonthDeliveryPromotionPlanAmountDto dto) {
        this.calMonthDeliveryValidate(dto);
        //获取字典中的活动形式编码
        //获取促销规划活动形式编码配置字典
        Map<String, String> formMap = dictToolkitService.findMapByDictTypeCode(ActivityApplyRulesConstant.PROMOTION_PLAN_ACT_FORM2);
        Validate.isTrue(!formMap.isEmpty(), "未查找到促销规划活动形式编码配置字典");
        //获取活动形式编码
        List<String> formCodes = new ArrayList<>();
        //渠道推广费（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.CHANNEL_PROMOTION_FEE, formCodes);
        //采购返点（元）
        this.getFormCode(formMap, PromotionPlanActFormEnum.PURCHASE_REBATE, formCodes);
        //查询申请规则
        ActivityApplyRulesPromotionPlanAmountDto amountDto = new ActivityApplyRulesPromotionPlanAmountDto();
        amountDto.setTenantCode(TenantUtils.getTenantCode());
        amountDto.setSalesOrgCode(dto.getSalesOrgCode());
        amountDto.setCustomerCode(dto.getCustomerCode());
        amountDto.setPlatformCode(dto.getPlatformCode());
        amountDto.setEstoreChannel(dto.getChannelCode());


        Map<String, Map<String, ActivityApplyRules>> applyRulesMap = this.getActivityApplyRuleMap(amountDto, formCodes);

        CalculateDto calculateDto = new CalculateDto();
        calculateDto.setTaxRate(dto.getTaxRate());
        calculateDto.setRatio(dto.getRatio());
        calculateDto.setStandardRetailPrice(dto.getStandardRetailPrice());
        calculateDto.setPlatformSupplyPrice(dto.getPlatformSupplyPrice());
        calculateDto.setActivityBasePrice(dto.getActivityBasePrice());
        calculateDto.setCombinationQuantity(dto.getCombinationQuantity());
        calculateDto.setEstimatedSalesBox(dto.getEstimatedSalesBox());
        calculateDto.setEstimatedSalesTon(dto.getEstimatedSalesTon());
        calculateDto.setVat(dto.getVat());
        calculateDto.setEstimatedAmount(dto.getEstimatedAmount());
        calculateDto.setCostPrice(dto.getCostPrice());
        calculateDto.setFormMap(formMap);

        ActivityApplyRulesPromotionPlanAmountVo amountVo = new ActivityApplyRulesPromotionPlanAmountVo();
        //渠道推广费（元）
        amountVo.setChannelPromotionFee(this.packageFormValue(formMap, PromotionPlanActFormEnum.CHANNEL_PROMOTION_FEE, applyRulesMap, calculateDto));
        //采购返点（元）
        amountVo.setPurchaseRebate(this.packageFormValue(formMap, PromotionPlanActFormEnum.PURCHASE_REBATE, applyRulesMap, calculateDto));
        return amountVo;
    }

    /**
     * 采销库存费用计算规则金额
     *
     * @param dto 规则匹配参数
     * @return ActivityApplyRulesPromotionPlanAmountVo
     */
    @Override
    public ActivityApplyRulesPromotionPlanAmountVo calAmountPurchaseSale(PurchaseSalePromotionPlanAmountDto dto) {
        this.calPurchaseSaleValidate(dto);
        //获取字典中的活动形式编码
        //获取促销规划活动形式编码配置字典
        Map<String, String> formMap = dictToolkitService.findMapByDictTypeCode(ActivityApplyRulesConstant.PROMOTION_PLAN_ACT_FORM2);
        Validate.isTrue(!formMap.isEmpty(), "未查找到促销规划活动形式编码配置字典");
        //获取活动形式编码
        List<String> formCodes = new ArrayList<>();
        //费用池-补差
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_DIFFERENCE, formCodes);
        //费用池-销售返
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_SALE_COMMISSION, formCodes);
        //费用池-采购返
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_PURCHASE_REBATE, formCodes);
        //费用池-投放
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_PUT, formCodes);
        //费用池-毛保
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_GROSS_PROTECTION, formCodes);
        //查询申请规则
        ActivityApplyRulesPromotionPlanAmountDto amountDto = new ActivityApplyRulesPromotionPlanAmountDto();
        amountDto.setTenantCode(TenantUtils.getTenantCode());
        amountDto.setSalesOrgCode(dto.getSalesOrgCode());
        amountDto.setCustomerCode(dto.getCustomerCode());
        amountDto.setPlatformCode(dto.getPlatformCode());
        amountDto.setEstoreChannel(dto.getChannelCode());


        Map<String, Map<String, ActivityApplyRules>> applyRulesMap = this.getActivityApplyRuleMap(amountDto, formCodes);

        CalculateDto calculateDto = new CalculateDto();
        calculateDto.setSupplyPrice(dto.getSupplyPrice());
        calculateDto.setPromotionPrice(dto.getPromotionPrice());
        calculateDto.setMonthInventoryEnd(dto.getMonthInventoryEnd());
        calculateDto.setNextMonthEndInventory(dto.getNextMonthEndInventory());
        calculateDto.setNextMonthInventoryAmount(dto.getNextMonthInventoryAmount());
        calculateDto.setStandardRetailPrice(dto.getStandardRetailPrice());
        calculateDto.setCostPrice(dto.getCostPrice());
        calculateDto.setPurchaseId(dto.getId());
//        calculateDto.setPutCostDto(dto.getPutCostDto());
        calculateDto.setFormMap(formMap);

        //取缓存,若缓存里没有则通过编码查详情，若无详情则为空
        //当月销售
        if (StringUtils.isNotBlank(dto.getMonthSaleCacheKey())) {
            List<CurrentMonthSaleDto> salesList = currentMonthSaleService.findCacheList(dto.getMonthSaleCacheKey());
            if (CollectionUtils.isEmpty(salesList)) {
                if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                    calculateDto.setMonthSaleServiceCacheList(currentMonthSaleService.findList(dto.getPromotionPlanCode()));
                }
            } else {
                calculateDto.setMonthSaleServiceCacheList(salesList);
            }
        }
        //费用申请
        if (StringUtils.isNotBlank(dto.getGeneralExpensesCacheKey())) {
            List<GeneralExpensesDto> expensesDtoList = generalExpensesService.findCacheList(dto.getGeneralExpensesCacheKey());
            if (CollectionUtils.isEmpty(expensesDtoList)) {
                if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                    calculateDto.setExpensesServiceCacheList(generalExpensesService.findList(dto.getPromotionPlanCode()));
                }
            } else {
                calculateDto.setExpensesServiceCacheList(expensesDtoList);
            }
        }
        //采销库存
        if (StringUtils.isNotBlank(dto.getCacheKey())) {
            List<PurchaseSaleDto> purchaseSaleDtoList = purchaseSaleService.findCacheList(dto.getCacheKey());
            if (CollectionUtils.isEmpty(purchaseSaleDtoList)) {
                if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                    calculateDto.setPurchaseSaleServiceCacheList(purchaseSaleService.findList(dto.getPromotionPlanCode()));
                }
            } else {
                calculateDto.setPurchaseSaleServiceCacheList(purchaseSaleDtoList);
            }
        }


        ActivityApplyRulesPromotionPlanAmountVo amountVo = new ActivityApplyRulesPromotionPlanAmountVo();
        //费用池-补差
        amountVo.setFeePoolDifference(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_DIFFERENCE, applyRulesMap, calculateDto));
        //费用池-销售返
        amountVo.setFeePoolSaleCommission(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_SALE_COMMISSION, applyRulesMap, calculateDto));
        //费用池-采购返
        amountVo.setFeePoolPurchaseRebate(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_PURCHASE_REBATE, applyRulesMap, calculateDto));
        //费用池-投放
        amountVo.setFeePoolPut(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_PUT, applyRulesMap, calculateDto));
        //费用池-毛保
        amountVo.setFeePoolGrossProtection(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_GROSS_PROTECTION, applyRulesMap, calculateDto));
        return amountVo;
    }

    /**
     * 采销库存费用重新计算
     *
     * @param dto 规则匹配参数
     */
    @Override
    public void reCalPurchaseSale(PurchaseSaleReCalDto dto) {
        Validate.notBlank(dto.getCustomerCode(), "计算金额时，客户编码不能为空！");
        Validate.notBlank(dto.getPlatformCode(), "计算金额时，平台不能为空！");
        Validate.notBlank(dto.getSalesOrgCode(), "计算金额时，销售部门编码不能为空！");
        Validate.isTrue(StringUtils.isNotBlank(dto.getCacheKey()), "采销库存缓存key不能为空");
        //获取字典中的活动形式编码
        //获取促销规划活动形式编码配置字典
        Map<String, String> formMap = dictToolkitService.findMapByDictTypeCode(ActivityApplyRulesConstant.PROMOTION_PLAN_ACT_FORM2);
        Validate.isTrue(!formMap.isEmpty(), "未查找到促销规划活动形式编码配置字典");
        //获取活动形式编码
        List<String> formCodes = new ArrayList<>();
        //费用池-补差
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_DIFFERENCE, formCodes);
        //费用池-销售返
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_SALE_COMMISSION, formCodes);
        //费用池-采购返
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_PURCHASE_REBATE, formCodes);
        //费用池-投放
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_PUT, formCodes);
        //费用池-毛保
        this.getFormCode(formMap, PromotionPlanActFormEnum.FEE_POOL_GROSS_PROTECTION, formCodes);
        //查询申请规则
        ActivityApplyRulesPromotionPlanAmountDto amountDto = new ActivityApplyRulesPromotionPlanAmountDto();
        amountDto.setTenantCode(TenantUtils.getTenantCode());
        amountDto.setSalesOrgCode(dto.getSalesOrgCode());
        amountDto.setCustomerCode(dto.getCustomerCode());
        amountDto.setPlatformCode(dto.getPlatformCode());
        amountDto.setEstoreChannel(dto.getChannelCode());

        Map<String, Map<String, ActivityApplyRules>> applyRulesMap = this.getActivityApplyRuleMap(amountDto, formCodes);

        //采销库存
        List<PurchaseSaleDto> purchaseSaleDtoList = purchaseSaleService.findCacheList(dto.getCacheKey());
        if (CollectionUtils.isEmpty(purchaseSaleDtoList)) {
            if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                purchaseSaleDtoList = purchaseSaleService.findList(dto.getPromotionPlanCode());
            } else {
                return;
            }
        }
        CalculateDto calculateDto = new CalculateDto();
        calculateDto.setFormMap(formMap);
        calculateDto.setPurchaseSaleServiceCacheList(purchaseSaleDtoList);

        //取缓存,若缓存里没有则通过编码查详情，若无详情则为空
        //当月销售
        if (StringUtils.isNotBlank(dto.getMonthSaleCacheKey())) {
            List<CurrentMonthSaleDto> salesList = currentMonthSaleService.findCacheList(dto.getMonthSaleCacheKey());
            if (CollectionUtils.isEmpty(salesList)) {
                if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                    calculateDto.setMonthSaleServiceCacheList(currentMonthSaleService.findList(dto.getPromotionPlanCode()));
                }
            } else {
                calculateDto.setMonthSaleServiceCacheList(salesList);
            }
        }
        //费用申请
        if (StringUtils.isNotBlank(dto.getGeneralExpensesCacheKey())) {
            List<GeneralExpensesDto> expensesDtoList = generalExpensesService.findCacheList(dto.getGeneralExpensesCacheKey());
            if (CollectionUtils.isEmpty(expensesDtoList)) {
                if (StringUtils.isNotBlank(dto.getPromotionPlanCode())) {
                    calculateDto.setExpensesServiceCacheList(generalExpensesService.findList(dto.getPromotionPlanCode()));
                }
            } else {
                calculateDto.setExpensesServiceCacheList(expensesDtoList);
            }
        }
        for (PurchaseSaleDto saleDto : purchaseSaleDtoList) {
            calculateDto.setSupplyPrice(saleDto.getSupplyPrice());
            calculateDto.setPromotionPrice(saleDto.getPromotionPrice());
            calculateDto.setMonthInventoryEnd(saleDto.getMonthInventoryEnd());
            calculateDto.setNextMonthEndInventory(saleDto.getNextMonthEndInventory());
            calculateDto.setNextMonthInventoryAmount(saleDto.getNextMonthInventoryAmount());
            calculateDto.setStandardRetailPrice(saleDto.getStandardRetailPrice());
            calculateDto.setCostPrice(saleDto.getCostPrice());
            calculateDto.setPurchaseId(saleDto.getId());
            //费用池-补差
            saleDto.setFeePoolDifference(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_DIFFERENCE, applyRulesMap, calculateDto));
            //费用池-销售返
            saleDto.setFeePoolSaleCommission(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_SALE_COMMISSION, applyRulesMap, calculateDto));
            //费用池-采购返
            saleDto.setFeePoolPurchaseRebate(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_PURCHASE_REBATE, applyRulesMap, calculateDto));
            //费用池-投放
            saleDto.setFeePoolPut(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_PUT, applyRulesMap, calculateDto));
            //费用池-毛保
            saleDto.setFeePoolGrossProtection(this.packageFormValue(formMap, PromotionPlanActFormEnum.FEE_POOL_GROSS_PROTECTION, applyRulesMap, calculateDto));
        }
        purchaseSaleService.updateListCache(dto.getCacheKey(), purchaseSaleDtoList);
    }

    /**
     * 计算金额参数校验
     *
     * @param dto dto对象
     */
    private void calMonthSaleValidate(MonthSalePromotionPlanAmountDto dto) {
        Validate.notNull(dto, "计算金额时，对象信息不能为空！");
        Validate.notBlank(dto.getCustomerCode(), "计算金额时，客户编码不能为空！");
        Validate.notBlank(dto.getPlatformCode(), "计算金额时，平台不能为空！");
        Validate.notBlank(dto.getSalesOrgCode(), "计算金额时，销售部门编码不能为空！");
        Validate.notNull(dto.getStartDate(), "计算金额时，开始时间不能为空！");
    }

    /**
     * 计算金额参数校验
     *
     * @param dto dto对象
     */
    private void calMonthDeliveryValidate(MonthDeliveryPromotionPlanAmountDto dto) {
        Validate.notNull(dto, "计算金额时，对象信息不能为空！");
        Validate.notBlank(dto.getCustomerCode(), "计算金额时，客户编码不能为空！");
        Validate.notBlank(dto.getPlatformCode(), "计算金额时，平台不能为空！");
        Validate.notBlank(dto.getSalesOrgCode(), "计算金额时，销售部门编码不能为空！");
    }

    /**
     * 计算金额参数校验
     *
     * @param dto dto对象
     */
    private void calPurchaseSaleValidate(PurchaseSalePromotionPlanAmountDto dto) {
        Validate.notNull(dto, "计算金额时，对象信息不能为空！");
        Validate.notBlank(dto.getCustomerCode(), "计算金额时，客户编码不能为空！");
        Validate.notBlank(dto.getPlatformCode(), "计算金额时，平台不能为空！");
        Validate.notBlank(dto.getSalesOrgCode(), "计算金额时，销售部门编码不能为空！");
    }

}
