package com.biz.crm.tpm.business.activity.apply.rules.local.variable.utils;

import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.constant.ActivityApplyRulesConstant;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.enums.PromotionPlanActFormEnum;
import com.biz.crm.tpm.business.activity.apply.rules.sdk.enums.PromotionPlanPutCostEnum;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.GeneralExpensesDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.PurchaseSaleDto;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author duyiran
 * @create 2023-01-17 16:05
 */
@Component
public class CalculateDtoUtil {

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    /**
     * 计算当月销售中【活动底价*组合数量*预计销量】之和
     *
     * @param dto dto
     */
    public void calCurrentMonthSaleBasePriceCombinationQuantityEstimatedSalesBoxSum(CalculateDto dto) {
        if (Objects.nonNull(dto.getCurrentMonthSaleBasePriceCombinationQuantityEstimatedSalesBoxSum())) {
            return;
        }
        List<CurrentMonthSaleDto> monthSaleServiceCacheList = dto.getMonthSaleServiceCacheList();
        if (CollectionUtils.isEmpty(monthSaleServiceCacheList)) {
            dto.setCurrentMonthSaleBasePriceCombinationQuantityEstimatedSalesBoxSum(BigDecimal.ZERO);
            return;
        }
        BigDecimal currentMonthSaleBasePriceCombinationQuantityEstimatedSalesBoxSum =
                monthSaleServiceCacheList.stream().filter(Objects::nonNull).map(e -> {
                    if (Objects.isNull(e.getActivityBasePrice()) || Objects.isNull(e.getCombinationQuantity()) || Objects.isNull(e.getEstimatedSalesBox())) {
                        return BigDecimal.ZERO;
                    }
                    return e.getActivityBasePrice().multiply(e.getCombinationQuantity()).multiply(e.getEstimatedSalesBox());
                }).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        dto.setCurrentMonthSaleBasePriceCombinationQuantityEstimatedSalesBoxSum(currentMonthSaleBasePriceCombinationQuantityEstimatedSalesBoxSum);
    }

    /**
     * 计算费用申请中活动形式为猫超卡的申请金额之和
     *
     * @param dto dto
     */
    public void calCatCardApplicationFee(CalculateDto dto) {
        if (Objects.nonNull(dto.getCatCardApplicationFee())) {
            return;
        }
        List<GeneralExpensesDto> expensesServiceCacheList = dto.getExpensesServiceCacheList();
        if (CollectionUtils.isEmpty(expensesServiceCacheList)) {
            dto.setCatCardApplicationFee(BigDecimal.ZERO);
            return;
        }
        String fromCode = dto.getFormMap().getOrDefault(PromotionPlanActFormEnum.CAT_CARD.getCode(), null);
        if (StringUtils.isBlank(fromCode)) {
            dto.setCatCardApplicationFee(BigDecimal.ZERO);
            return;
        }
        BigDecimal catCardApplicationFee = expensesServiceCacheList.stream().filter(e -> fromCode.equals(e.getActivityFormCode()))
                .map(GeneralExpensesDto::getApplyAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        dto.setCatCardApplicationFee(catCardApplicationFee);
    }

    /**
     * 计算费用申请中活动形式为优惠券的申请金额之和
     *
     * @param dto dto
     */
    public void calDiscountCouponApplicationFee(CalculateDto dto) {
        if (Objects.nonNull(dto.getDiscountCouponApplicationFee())) {
            return;
        }
        List<GeneralExpensesDto> expensesServiceCacheList = dto.getExpensesServiceCacheList();
        if (CollectionUtils.isEmpty(expensesServiceCacheList)) {
            dto.setDiscountCouponApplicationFee(BigDecimal.ZERO);
            return;
        }
        String fromCode = dto.getFormMap().getOrDefault(PromotionPlanActFormEnum.DISCOUNT_COUPON.getCode(), null);
        if (StringUtils.isBlank(fromCode)) {
            dto.setDiscountCouponApplicationFee(BigDecimal.ZERO);
            return;
        }
        BigDecimal discountCouponApplicationFee = expensesServiceCacheList.stream().filter(e -> fromCode.equals(e.getActivityFormCode()))
                .map(GeneralExpensesDto::getApplyAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        dto.setDiscountCouponApplicationFee(discountCouponApplicationFee);
    }

    /**
     * 计算费用申请中活动形式为“双计-日常”和“双计-合同款”的明细行的申请金额合计
     *
     * @param dto dto
     */
    public void calDoubleCountApplicationFee(CalculateDto dto) {
        if (Objects.nonNull(dto.getDoubleCountApplicationFee())) {
            return;
        }
        List<GeneralExpensesDto> expensesServiceCacheList = dto.getExpensesServiceCacheList();
        if (CollectionUtils.isEmpty(expensesServiceCacheList)) {
            dto.setDoubleCountApplicationFee(BigDecimal.ZERO);
            return;
        }
        //双计-日常
        BigDecimal b1 = BigDecimal.ZERO;
        String fromCode1 = dto.getFormMap().getOrDefault(PromotionPlanActFormEnum.DOUBLE_COUNT_DAILY.getCode(), null);
        if (StringUtils.isNotBlank(fromCode1)) {
            b1 = expensesServiceCacheList.stream()
                    .filter(e -> fromCode1.equals(e.getActivityFormCode()))
                    .map(GeneralExpensesDto::getApplyAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        }
        //双计-合同款
        BigDecimal b2 = BigDecimal.ZERO;
        String fromCode2 = dto.getFormMap().getOrDefault(PromotionPlanActFormEnum.DOUBLE_COUNT_CONTRACT.getCode(), null);
        if (StringUtils.isNotBlank(fromCode2)) {
            b2 = expensesServiceCacheList.stream()
                    .filter(e -> fromCode2.equals(e.getActivityFormCode()))
                    .map(GeneralExpensesDto::getApplyAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        }
        dto.setDoubleCountApplicationFee(b1.add(b2));
    }

    /**
     * 计算 费用申请页面的活动形式排除 ‘费用池-投放’的其他活动形式对应的一级预算项目为投放费用的明细行的申请金额合计
     *
     * @param dto dto
     */
    public void calPutInFee(CalculateDto dto) {
        if (Objects.nonNull(dto.getPutInFee())) {
            return;
        }
        List<GeneralExpensesDto> expensesServiceCacheList = dto.getExpensesServiceCacheList();
        if (CollectionUtils.isEmpty(expensesServiceCacheList)) {
            dto.setPutInFee(BigDecimal.ZERO);
            return;
        }
        //获取费用池-投放的活动形式
        String feeOut = dto.getFormMap().getOrDefault(PromotionPlanActFormEnum.FEE_POOL_PUT.getCode(), "");
        //获取促销规划活动形式编码配置字典
        Map<String, String> itemMap = dictToolkitService.findMapByDictTypeCode(ActivityApplyRulesConstant.PROMOTION_PLAN_BUDGET_ITEM_CODE1);
        String itemCode = itemMap.getOrDefault(PromotionPlanPutCostEnum.PUT_IN_FEE.getCode(), null);
        if (StringUtils.isBlank(itemCode)) {
            dto.setPutInFee(BigDecimal.ZERO);
            return;
        }
        BigDecimal putInFee = expensesServiceCacheList.stream()
                .filter(e -> !feeOut.equals(e.getActivityFormCode()))
                .filter(e -> itemCode.equals(e.getBudgetItemCode1()))
                .map(GeneralExpensesDto::getApplyAmount).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        dto.setPutInFee(putInFee);
    }

    /**
     * 计算（当月库存数量*当月库存预估售价）汇总
     *
     * @param dto dto
     */
    public void calMonthInventoryTotal(CalculateDto dto) {
        if (Objects.nonNull(dto.getMonthInventoryTotal())) {
            return;
        }
        List<PurchaseSaleDto> purchaseSaleDtoList = dto.getPurchaseSaleServiceCacheList();
        if (CollectionUtils.isEmpty(purchaseSaleDtoList)) {
            dto.setMonthInventoryTotal(BigDecimal.ZERO);
            return;
        }
        BigDecimal total = BigDecimal.ZERO;
        for (PurchaseSaleDto saleDto : purchaseSaleDtoList) {
            //当前行取编辑的数据
            if (StringUtils.isNotBlank(dto.getPurchaseId()) && saleDto.getId().equals(dto.getPurchaseId())) {
                if (null == dto.getMonthInventoryEnd() || null == dto.getPromotionPrice()) {
                    continue;
                }
                total = total.add(dto.getMonthInventoryEnd().multiply(dto.getPromotionPrice()));
                continue;
            }
            if (null == saleDto.getMonthInventoryEnd() || null == saleDto.getPromotionPrice()) {
                continue;
            }
            total = total.add(saleDto.getMonthInventoryEnd().multiply(saleDto.getPromotionPrice()));

        }
        dto.setMonthInventoryTotal(total);
    }

    /**
     * 计算（（次月库存数量*次月库存预估售价）汇总
     *
     * @param dto dto
     */
    public void calNextMonthInventoryTotal(CalculateDto dto) {
        if (Objects.nonNull(dto.getNextMonthInventoryTotal())) {
            return;
        }
        List<PurchaseSaleDto> purchaseSaleDtoList = dto.getPurchaseSaleServiceCacheList();
        if (CollectionUtils.isEmpty(purchaseSaleDtoList)) {
            dto.setNextMonthInventoryTotal(BigDecimal.ZERO);
            return;
        }
        BigDecimal total = BigDecimal.ZERO;
        for (PurchaseSaleDto saleDto : purchaseSaleDtoList) {
            //当前行取编辑的数据
            if (StringUtils.isNotBlank(dto.getPurchaseId()) && saleDto.getId().equals(dto.getPurchaseId())) {
                if (null == dto.getNextMonthEndInventory() || null == dto.getNextMonthInventoryAmount()) {
                    continue;
                }
                total = total.add(dto.getNextMonthEndInventory().multiply(dto.getNextMonthInventoryAmount()));
                continue;
            }
            if (null == saleDto.getNextMonthEndInventory() || null == saleDto.getNextMonthInventoryAmount()) {
                continue;
            }
            total = total.add(saleDto.getNextMonthEndInventory().multiply(saleDto.getNextMonthInventoryAmount()));

        }
        dto.setNextMonthInventoryTotal(total);
    }

    /**
     * 计算当月销售中的GMV汇总
     *
     * @param dto dto
     */
    public void calMonthSaleGmvTotal(CalculateDto dto) {
        if (Objects.nonNull(dto.getMonthSaleGmvTotal())) {
            return;
        }
        List<CurrentMonthSaleDto> monthSaleServiceCacheList = dto.getMonthSaleServiceCacheList();
        if (CollectionUtils.isEmpty(monthSaleServiceCacheList)) {
            dto.setMonthSaleGmvTotal(BigDecimal.ZERO);
            return;
        }
        BigDecimal total = BigDecimal.ZERO;

        for (CurrentMonthSaleDto saleDto : monthSaleServiceCacheList) {
            if (null == saleDto.getGmv()) {
                continue;
            }
            total = total.add(saleDto.getGmv());
        }
        dto.setMonthSaleGmvTotal(total);
    }

    /**
     * 计算 【（当月库存数量*标准零售价）汇总】
     *
     * @param dto dto
     */
    public void calMonthInventoryStandTotal(CalculateDto dto) {
        if (Objects.nonNull(dto.getMonthInventoryStandTotal())) {
            return;
        }
        List<PurchaseSaleDto> purchaseSaleDtoList = dto.getPurchaseSaleServiceCacheList();
        if (CollectionUtils.isEmpty(purchaseSaleDtoList)) {
            dto.setMonthInventoryStandTotal(BigDecimal.ZERO);
            return;
        }
        BigDecimal total = BigDecimal.ZERO;
        for (PurchaseSaleDto saleDto : purchaseSaleDtoList) {
            //当前行取编辑的数据
            if (StringUtils.isNotBlank(dto.getPurchaseId()) && saleDto.getId().equals(dto.getPurchaseId())) {
                if (null == dto.getMonthInventoryEnd() || null == dto.getStandardRetailPrice()) {
                    continue;
                }
                total = total.add(dto.getMonthInventoryEnd().multiply(dto.getStandardRetailPrice()));
                continue;
            }
            if (null == saleDto.getMonthInventoryEnd() || null == saleDto.getStandardRetailPrice()) {
                continue;
            }
            total = total.add(saleDto.getMonthInventoryEnd().multiply(saleDto.getStandardRetailPrice()));
        }

        dto.setMonthInventoryStandTotal(total);
    }

    /**
     * 计算 【（次月库存数量*标准零售价）汇总】
     *
     * @param dto dto
     */
    public void calNextMonthInventoryStandTotal(CalculateDto dto) {
        if (Objects.nonNull(dto.getNextMonthInventoryStandTotal())) {
            return;
        }
        List<PurchaseSaleDto> purchaseSaleDtoList = dto.getPurchaseSaleServiceCacheList();
        if (CollectionUtils.isEmpty(purchaseSaleDtoList)) {
            dto.setMonthInventoryStandTotal(BigDecimal.ZERO);
            return;
        }
        BigDecimal total = BigDecimal.ZERO;
        for (PurchaseSaleDto saleDto : purchaseSaleDtoList) {
            //当前行取编辑的数据
            if (StringUtils.isNotBlank(dto.getPurchaseId()) && saleDto.getId().equals(dto.getPurchaseId())) {
                if (null == dto.getNextMonthEndInventory() || null == dto.getStandardRetailPrice()) {
                    continue;
                }
                total = total.add(dto.getNextMonthEndInventory().multiply(dto.getStandardRetailPrice()));
                continue;
            }
            if (null == saleDto.getNextMonthEndInventory() || null == saleDto.getStandardRetailPrice()) {
                continue;
            }
            total = total.add(saleDto.getNextMonthEndInventory().multiply(saleDto.getStandardRetailPrice()));
        }

        dto.setNextMonthInventoryStandTotal(total);
    }


    /**
     * 当前行的预计折前销售额（当月销售）/预计折前销售额汇总（当月销售）
     *
     * @param dto
     * @return
     */
    public void discountBeforeRatio(CalculateDto dto) {
        List<CurrentMonthSaleDto> monthSaleServiceCacheList = dto.getMonthSaleServiceCacheList();
        if (CollectionUtils.isEmpty(monthSaleServiceCacheList)) {
            dto.setDiscountBeforeRatio(BigDecimal.ZERO);
            return;
        }

        //分摊比例
        BigDecimal discountBeforeTotalCustomer = monthSaleServiceCacheList.stream().map(e -> bdNull(e.getEstimatedAmountBefore()))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        dto.setDiscountBeforeRatio(dto.getEstimatedAmountBefore().divide(discountBeforeTotalCustomer, 6, BigDecimal.ROUND_HALF_UP));
    }


    private BigDecimal bdNull(BigDecimal b) {
        return b == null ? BigDecimal.ZERO : b;
    }
}
