package com.biz.crm.tpm.business.activity.detail.plan.local.repository;


import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.dto.audit.AutoAuditParamsDto;
import com.biz.crm.tpm.business.activity.detail.plan.local.entity.ActivityDetailPlan;
import com.biz.crm.tpm.business.activity.detail.plan.local.mapper.ActivityDetailPlanMapper;
import com.biz.crm.tpm.business.activity.detail.plan.local.vo.ActivityPlanMessageVo;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.dto.ActivityDetailPlanDto;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.dto.ActivityDetailPlanItemDto;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.dto.ActivityDetailPlanReportDto;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.vo.ActivityDetailPlanItemVo;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.vo.ActivityDetailPlanReportVo;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.vo.ActivityDetailPlanVo;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.ActivityPlanStatusEnum;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityDetailPlanPlanVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;


/**
 * 活动细案(ActivityDetailPlan)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-11-04 16:52:04
 */
@Slf4j
@Component
public class ActivityDetailPlanRepository extends ServiceImpl<ActivityDetailPlanMapper, ActivityDetailPlan> {

    @Autowired(required = false)
    private ActivityDetailPlanMapper activityDetailPlanMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    private static final String AUTO_AUDIT_KEY = "Y";

    private static final String WHOLE_AUDIT_KEY = "Y";

    /**
     * 分页查询数据
     * @param pageable 分页对象
     * @param activityDetailPlan 实体对象
     * @return
     */
    public Page<ActivityDetailPlanVo> findByConditions(Pageable pageable, ActivityDetailPlanDto activityDetailPlan) {
        Page<ActivityDetailPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<ActivityDetailPlanVo> pageList = this.activityDetailPlanMapper.findByConditions(page, activityDetailPlan);
        return pageList;
    }


    public Page<ActivityDetailPlanReportVo> findDetailByConditions(Pageable pageable, ActivityDetailPlanReportDto dto) {
        Page<ActivityDetailPlanReportVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.activityDetailPlanMapper.findDetailByConditions(page, dto);
    }

    public Page<ActivityDetailPlanItemVo> queryActivityDetailPlan(Pageable pageable, ActivityDetailPlanItemDto dto) {
        Page<ActivityDetailPlanItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<ActivityDetailPlanItemVo> pageList = this.activityDetailPlanMapper.queryActivityDetailPlan(page,dto);
        return pageList;
    }

    /**
     * 删除活动方案
     */
    public void deleteByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        LambdaUpdateWrapper<ActivityDetailPlan> updateWrapper = new UpdateWrapper<ActivityDetailPlan>().lambda();
        updateWrapper.set(ActivityDetailPlan::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(ActivityDetailPlan::getId, ids);
        this.update(updateWrapper);
    }

    /**
     * 更新业务数据流程状态
     * @param codeList 活动细案编码集合
     * @param processStatus 流程状态
     */
    public void updateProcessStatus(List<String> codeList, String processStatus) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        this.update(Wrappers.lambdaUpdate(ActivityDetailPlan.class)
                .set(ActivityDetailPlan::getProcessStatus, processStatus)
                .in(ActivityDetailPlan::getDetailPlanCode, codeList)
                .eq(ActivityDetailPlan::getTenantCode, TenantUtils.getTenantCode())
        );
    }

    /**
     * 更新业务数据流程状态和流程编号
     * @param codeList
     * @param processStatus
     * @param processNo
     */
    public void updateProcessStatusAndProcessNo(List<String> codeList, String processStatus, String processNo) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        this.update(Wrappers.lambdaUpdate(ActivityDetailPlan.class)
                .set(ActivityDetailPlan::getProcessStatus, processStatus)
                .set(ActivityDetailPlan::getProcessNo, processNo)
                .in(ActivityDetailPlan::getDetailPlanCode, codeList)
                .eq(ActivityDetailPlan::getTenantCode, TenantUtils.getTenantCode())
        );
    }

    public List<ActivityDetailPlan> findByPlanCode(String planCode) {

        if (StringUtils.isEmpty(planCode)) {
            return null;
        }
        List<ActivityDetailPlan> list = this.lambdaQuery().eq(ActivityDetailPlan::getPlanCode, planCode)
                .eq(ActivityDetailPlan::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityDetailPlan::getTenantCode, TenantUtils.getTenantCode())
                .list();
        if (!CollectionUtils.isEmpty(list)) {
            return list;
        }
        return null;
    }

    public List<ActivityDetailPlanItemVo> getCanAutoAuditDto() {

        Date date = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.HOUR_OF_DAY,0);
        cal.set(Calendar.MINUTE,0);
        cal.set(Calendar.SECOND,0);
        System.out.println(cal.getTime());

        return this.getBaseMapper().getCanAutoAuditDto(cal.getTime());
    }

    /**
     * 查询可自动核销数据
     * @param page 分页
     * @param auditDate 核销时间
     * @return Page
     */
    public List<ActivityDetailPlanVo> findCanAutoAudit(Page<ActivityDetailPlanVo> page, String auditDate){

        Page<ActivityDetailPlanVo> resultPage = this.getBaseMapper()
                .listDetailPlanCanAudoAudit(page
                , auditDate
                , AUTO_AUDIT_KEY
                , WHOLE_AUDIT_KEY);

        if (resultPage != null){
            return resultPage.getRecords();
        }

        return null;
    }

    public List<ActivityDetailPlanVo> findCanAutoAudit(Page<ActivityDetailPlanVo> page, AutoAuditParamsDto autoAuditParams){

        Page<ActivityDetailPlanVo> resultPage = this.baseMapper.findCanAutoAudit(page, autoAuditParams);

        return resultPage.getRecords();

    }

    public int getCanAutoAuditNumber(String autoDate, String businessUnitCode){


        Integer canAutoAuditTotalNumber = this.getBaseMapper()
                .getCanAutoAuditTotalNumber(autoDate, AUTO_AUDIT_KEY, WHOLE_AUDIT_KEY, businessUnitCode);

        return (canAutoAuditTotalNumber == null || canAutoAuditTotalNumber < 0)
                ? 0
                : canAutoAuditTotalNumber;

    }

    public int getActivityTotalNumber(AutoAuditParamsDto autoAuditParams){

        Integer resultNumber = this.baseMapper.getActivityTotalNumber(autoAuditParams);

        return resultNumber == null ? 0 : resultNumber;

    }

    public void updateActivityDetailPlanAutoAudit(String detailPlanCode){
        this.getBaseMapper().updateActivityDetailPlanAutoAudit(detailPlanCode, AUTO_AUDIT_KEY);
    }

    public List<ActivityDetailPlanVo> findByPlanCodes(Set<String> detailPlanCodes) {

        if(CollectionUtils.isEmpty(detailPlanCodes)){
            return Lists.newArrayList();
        }
        List<ActivityDetailPlan> list = this.lambdaQuery().in(ActivityDetailPlan::getDetailPlanCode, detailPlanCodes).list();
        if(CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        Collection<ActivityDetailPlanVo> activityDetailPlanVos = this.nebulaToolkitService.copyCollectionByWhiteList(list, ActivityDetailPlan.class, ActivityDetailPlanVo.class, LinkedHashSet.class, ArrayList.class);
        return (List<ActivityDetailPlanVo>) activityDetailPlanVos;
    }

    /**
     * 根据活动细案编码获取活动细案信息
     *
     * @param detailPlanCodeList
     * @return
     */
    public List<ActivityDetailPlan> findByDetailPlanCodeList(List<String> detailPlanCodeList) {

        if (CollectionUtils.isEmpty(detailPlanCodeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(ActivityDetailPlan::getDetailPlanCode, detailPlanCodeList)
                .eq(ActivityDetailPlan::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityDetailPlan::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }


    /**
     * 根据活动细案编码获取活动细案信息
     * @param foreignPlanCodeList 外部系统活动编码
     */
    public List<ActivityDetailPlan> findByForeignPlanCodeList(List<String> foreignPlanCodeList) {
        if (CollectionUtils.isEmpty(foreignPlanCodeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(ActivityDetailPlan::getForeignPlanCode, foreignPlanCodeList)
                .eq(ActivityDetailPlan::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityDetailPlan::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    public List<ActivityDetailPlanVo> findPlanCodes(Set<String> codeSet) {
        if (CollectionUtils.isEmpty(codeSet)) {
            return Collections.emptyList();
        }
        return this.activityDetailPlanMapper.findPlanCodes(codeSet,TenantUtils.getTenantCode(),ActivityPlanStatusEnum.allClose.getCode());
    }

    public List<ActivityPlanMessageVo> findPlanMessageByDetailPlanCodeSet(Set<String> detailPlanCodeSet) {
        if (CollectionUtils.isEmpty(detailPlanCodeSet)) {
            return Collections.emptyList();
        }
        return this.activityDetailPlanMapper.findPlanMessageByDetailPlanCodeSet(detailPlanCodeSet);
    }

    /**
     * 修改自动核销标识
     * @param codeList 活动编码
     * @param autoAudit 标识
     */
    public void updateAutoAuditFlag(List<String> codeList, String autoAudit){
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        List<List<String>> partition = Lists.partition(codeList, 800);
        partition.forEach(list -> this.baseMapper.updateAutoAuditFlag(list, autoAudit));

    }

    public ActivityDetailPlan findById(String id) {
        return this.lambdaQuery()
                .eq(ActivityDetailPlan::getId, id)
                .eq(ActivityDetailPlan::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    /**
     * 查找没有被核销的细案
     * @param dto 查询条件
     * @return 实体集合
     */
    public List<ActivityDetailPlan> findHaveNotAuditDetailPlanEntity(ActivityDetailPlanDto dto) {
        return this.activityDetailPlanMapper.findHaveNotAuditDetailPlanEntity(dto);
    }

    public Page<ActivityDetailPlanVo> findPageForOut(Pageable pageable, ActivityDetailPlanDto dto) {
        Page<ActivityDetailPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.activityDetailPlanMapper.findPageForOut(page, dto);
    }

    /**
     * 更新细案活动状态
     *
     * @param detailPlanCodeList
     * @param statusEnum
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-08-26 02:29
     */
    public void updatePlanStatusByDetailPlanCodeList(List<String> detailPlanCodeList, ActivityPlanStatusEnum statusEnum) {
        if (CollectionUtil.isEmpty(detailPlanCodeList)
                || Objects.isNull(statusEnum)) {
            return;
        }
        log.info("活动细案活动状态更新repository {} {}", detailPlanCodeList, statusEnum);
        this.activityDetailPlanMapper.updatePlanStatusByDetailPlanCodeList(detailPlanCodeList, statusEnum.getCode(), TenantUtils.getTenantCode());

    }

    /**
     * 根据方案明细编码获取细案信息
     *
     * @param planItemCodeList
     * @return java.util.List<com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityDetailPlanPlanVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-11-29 15:08
     */
    public List<ActivityDetailPlanPlanVo> getDetailPlanInfoByPlanItemCodeList(List<String> planItemCodeList) {
        if (CollectionUtil.isEmpty(planItemCodeList)) {
            return Collections.emptyList();
        }
        return this.activityDetailPlanMapper.getDetailPlanInfoByPlanItemCodeList(planItemCodeList, TenantUtils.getTenantCode());

    }

    /**
     * 细案编码查询方案信息
     * @param detailPlanCodeList
     * @return
     */
    public List<ActivityDetailPlanPlanVo> findPlanByRelateDetailPlanCode(List<String> detailPlanCodeList) {
        if (CollectionUtil.isEmpty(detailPlanCodeList)) {
            return Collections.emptyList();
        }
        return this.activityDetailPlanMapper.findPlanByRelateDetailPlanCode(detailPlanCodeList, TenantUtils.getTenantCode());
    }
}

