package com.biz.crm.tpm.business.activity.detail.plan.local.service.third;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.*;
import com.biz.crm.mn.third.system.cow.manager.sdk.service.CowManagerService;
import com.biz.crm.mn.third.system.cow.manager.sdk.vo.CowManagerVo;
import com.biz.crm.tpm.business.activity.detail.plan.local.service.ActivityDetailPlanItemService;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.InterfacePushStateEnum;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.VerticalActivityTypeEnum;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.vo.ActivityDetailPlanItemVo;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.ActivityPlanWhereFrom;
import com.biz.crm.tpm.business.activity.plan.sdk.service.ActivityPlanSdkService;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description: 促销活动推送牛人管家
 * @Author qiancheng
 * @Date 2023/1/16
 */
@Slf4j
@Component
public class ActivityPushCowManager {

    @Autowired(required = false)
    private CowManagerService cowManagerService;

    @Autowired(required = false)
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private ActivityCowManagerInterData activityCowManagerInterData;

    @Autowired(required = false)
    private ActivityDetailPlanItemService activityDetailPlanItemService;

    @Autowired(required = false)
    private ActivityPlanSdkService activityPlanSdkService;


    /**
     *活动执行查询牛人管家
     *
     * @param type 费用类型
     * @param activeNumber 明细编码
     * @return 查询结果
     */
    public List<CowManagerExecutionDto> queryActivityExecution(String type, String activeNumber) {
        Validate.notEmpty(type,"费用类型不能为空！");
        CowManagerQueryActivityDto dto = new CowManagerQueryActivityDto();
        dto.setType(type);
        dto.setActiveNumber(activeNumber);
        return cowManagerService.queryActivityExecution(dto);
    }

    /**
     * 促销活动关闭推送牛人管家
     * @param itemCodes 明细编码集合
     */
    public void closeActivityDetailPlan(List<String> itemCodes) {
        Validate.notEmpty(itemCodes,"传入编码不能为空！");
        List<ActivityDetailPlanItemVo> itemList = activityDetailPlanItemService.findByItemCodes(itemCodes);
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        this.buildActivityDetailPlan(itemList, true);

    }


    /**
     * 促销活动创建、变更推送牛人管家
     * @param planCodes 活动细案编码集合
     */
    public void buildAndPushActivityDetailPlan(List<String> planCodes) {
        Validate.notEmpty(planCodes,"传入编码不能为空！");
        //获取审批通过的活动细案的明细数据
        List<ActivityDetailPlanItemVo> detailPlanItemVos = activityDetailPlanItemService.findByDetailPlanCode(planCodes);
        if (CollectionUtils.isEmpty(detailPlanItemVos)) {
            return;
        }
        this.buildActivityDetailPlan(detailPlanItemVos, false);
    }

    /**
     * 过滤不需要推送牛人管家的明细。
     * @param detailPlanItemVoList 明细集合
     * @return
     */
    private void filterNoPushDetailPlanItem(List<ActivityDetailPlanItemVo> detailPlanItemVoList,Set<String> formCodeSet) {
        if (CollectionUtils.isEmpty(detailPlanItemVoList)) {
            return;
        }
        if(CollectionUtils.isEmpty(formCodeSet)) {
            detailPlanItemVoList.clear();
            return;
        }
        Iterator<ActivityDetailPlanItemVo> iterator = detailPlanItemVoList.iterator();
        while (iterator.hasNext()) {
            ActivityDetailPlanItemVo next = iterator.next();
            if(!formCodeSet.contains(next.getActivityFormCode())){
                iterator.remove();
            }
        }
        //垂直的查询关联的方案，如果是未匹配费用单、未匹配结算单，不推送SAP
        List<String> planCodeList = detailPlanItemVoList.stream().map(ActivityDetailPlanItemVo::getRelatePlanCode).distinct().filter(Objects::nonNull).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(planCodeList)){
            List<ActivityPlanVo> planList = activityPlanSdkService.findByPlanCodeList(Sets.newHashSet(planCodeList));
            if (CollectionUtils.isEmpty(planList)){
                throw new RuntimeException("关联活动方案查询失败！");
            }
            Map<String, ActivityPlanVo> planMap = planList.stream().collect(Collectors.toMap(ActivityPlanVo::getPlanCode, Function.identity()));
            Iterator<ActivityDetailPlanItemVo> iterator1 = detailPlanItemVoList.iterator();
            while (iterator1.hasNext()){
                ActivityDetailPlanItemVo detailPlanItemVo = iterator1.next();
                if (StringUtils.isEmpty(detailPlanItemVo.getRelatePlanCode())){
                    continue;
                }
                ActivityPlanVo planVo = planMap.get(detailPlanItemVo.getRelatePlanCode());
                if (!ActivityPlanWhereFrom.MANUAL.getCode().equals(planVo.getWhereFrom())){
                    iterator1.remove();
                }
            }
        }
    }

    /**
     * 组装活动细案明细数据 并推送牛人管家
     * @param detailPlanItemVoList 活动细案明细集合
     * @param close 是否关闭 true：关闭
     */
    private void buildActivityDetailPlan(List<ActivityDetailPlanItemVo> detailPlanItemVoList,boolean close) {

        Map<String, String> formMap = activityFormService.findPushCowManager();
        if(CollectionUtils.isEmpty(formMap)){
            return;
        }
        //过滤不需要推送牛人管家的明细
        this.filterNoPushDetailPlanItem(detailPlanItemVoList,formMap.keySet());

        //区分推送的活动类型
        Map<String, List<ActivityDetailPlanItemVo>> planItemMap = this.categorizeDetailPlanItem(formMap, detailPlanItemVoList);
        //组装数据
        List<CowManagerCellActivityDto> marketingDataList = activityCowManagerInterData.buildCellActivityInterfaceData(planItemMap.get(VerticalActivityTypeEnum.MARKETING.getCode()),close);
        List<CowManagerDisplayActivityDto> displayDataList = activityCowManagerInterData.buildDisplayActivityInterfaceData(planItemMap.get(VerticalActivityTypeEnum.DISPLAY.getCode()),close);
        List<CowManagerPromotionActivityDto> promotionDataList = activityCowManagerInterData.buildPromotionActivityInterfaceData(planItemMap.get(VerticalActivityTypeEnum.PROMOTION.getCode()),close);
        List<CowManagerTemporaryPersonnelActivityDto> temporaryDataList = activityCowManagerInterData.buildTemporaryPersonnelActivity(planItemMap.get(VerticalActivityTypeEnum.TEMPORARY.getCode()),close);
        List<CowManagerLongTermPersonnelActivityDto> longPersonDataList = activityCowManagerInterData.buildLongTermPersonnelActivity(planItemMap.get(VerticalActivityTypeEnum.LONG_PROMOTION.getCode()),close);
        //推送
        this.pushData(marketingDataList,displayDataList,promotionDataList,temporaryDataList,longPersonDataList);
    }

    /**
     * 将明细按垂直活动类型分类
     * @param fromMap 活动形式编码对应的垂直活动类型map
     * @param detailPlanItemVoList 明细集合
     */
    private Map<String,List<ActivityDetailPlanItemVo>> categorizeDetailPlanItem(Map<String, String> fromMap, List<ActivityDetailPlanItemVo> detailPlanItemVoList) {
        if (CollectionUtils.isEmpty(detailPlanItemVoList) || CollectionUtils.isEmpty(fromMap)){
            return Maps.newHashMap();
        }
        Map<String,List<ActivityDetailPlanItemVo>> map = new HashMap<>(5);
        //推送行销活动接口的明细集合
        map.put(VerticalActivityTypeEnum.MARKETING.getCode(),new ArrayList<>());
        //推送陈列接口的明细集合
        map.put(VerticalActivityTypeEnum.DISPLAY.getCode(),new ArrayList<>());
        //推送促销接口的明细集合
        map.put(VerticalActivityTypeEnum.PROMOTION.getCode(),new ArrayList<>());
        //推送临促人员活动的明细集合
        map.put(VerticalActivityTypeEnum.TEMPORARY.getCode(),new ArrayList<>());
        //推送长促人员活动的明细集合
        map.put(VerticalActivityTypeEnum.LONG_PROMOTION.getCode(),new ArrayList<>());
        detailPlanItemVoList.forEach(detailPlanItem -> {
            String activityType = fromMap.get(detailPlanItem.getActivityFormCode());
            if(StringUtil.isNotBlank(activityType)) {
                if(VerticalActivityTypeEnum.MARKETING.getCode().equals(activityType)){
                    map.get(VerticalActivityTypeEnum.MARKETING.getCode()).add(detailPlanItem);

                }else if(VerticalActivityTypeEnum.DISPLAY.getCode().equals(activityType)){
                    map.get(VerticalActivityTypeEnum.DISPLAY.getCode()).add(detailPlanItem);

                }else if(VerticalActivityTypeEnum.PROMOTION.getCode().equals(activityType)){
                    map.get(VerticalActivityTypeEnum.PROMOTION.getCode()).add(detailPlanItem);

                }else if(VerticalActivityTypeEnum.TEMPORARY.getCode().equals(activityType)){
                    map.get(VerticalActivityTypeEnum.TEMPORARY.getCode()).add(detailPlanItem);

                }else if(VerticalActivityTypeEnum.LONG_PROMOTION.getCode().equals(activityType)){
                    map.get(VerticalActivityTypeEnum.LONG_PROMOTION.getCode()).add(detailPlanItem);
                }
            }
        });
        return map;
    }

    /**
     * 推送活动数据至牛人管家
     * @param marketingDataList 行销活动集合
     * @param displayDataList 陈列活动集合
     * @param promotionDataList 促销活动集合
     * @param temporaryDataList 临促活动集合
     * @param longPersonDataList 长促活动集合
     */
    private void pushData(List<CowManagerCellActivityDto> marketingDataList,
                            List<CowManagerDisplayActivityDto> displayDataList,
                            List<CowManagerPromotionActivityDto> promotionDataList,
                            List<CowManagerTemporaryPersonnelActivityDto> temporaryDataList,
                            List<CowManagerLongTermPersonnelActivityDto> longPersonDataList) {

        List<String> successCodes = new ArrayList<>();
        //行销活动
        if(!CollectionUtils.isEmpty(marketingDataList)){
            CowManagerVo mar = this.pushCellActivity(marketingDataList);
            if(mar.isSuccess()){
                try {
                    // 牛人管家接口响应信息，data字段，成功时为 true(boolean)，失败时为 原因(String)
                    if((boolean)mar.getData()){
                        List<String> itemCodes = marketingDataList.stream().map(CowManagerCellActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
                        successCodes.addAll(itemCodes);
                    }
                } catch (Exception e) {
                    log.warn("行销活动推送失败! cause by：",mar.getData());
                }
            }else {
                log.warn("行销活动推送失败! cause by：",mar.getData());
            }
        }
        //陈列活动
        if(!CollectionUtils.isEmpty(displayDataList)){
            CowManagerVo dis = this.pushDisplayActivity(displayDataList);
            if(dis.isSuccess()){
                try {
                    if((boolean)dis.getData()) {
                        List<String> itemCodes = displayDataList.stream().map(CowManagerDisplayActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
                        successCodes.addAll(itemCodes);
                    }
                } catch (Exception e) {
                    log.warn("陈列活动推送失败! cause by：",dis.getData());
                }
            }else {
                log.warn("陈列活动推送失败! cause by：",dis.getData());
            }
        }
        //促销活动
        if(!CollectionUtils.isEmpty(promotionDataList)){
            CowManagerVo pro = this.pushPromotionActivity(promotionDataList);
            if(pro.isSuccess()){
                try {
                    if((boolean)pro.getData()){
                        List<String> itemCodes = promotionDataList.stream().map(CowManagerPromotionActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
                        successCodes.addAll(itemCodes);
                    }
                } catch (Exception e) {
                    log.warn("促销活动推送失败! cause by：",pro.getData());
                }
            }else {
                log.warn("促销活动推送失败! cause by：",pro.getData());
            }
        }
        //临促活动
        if(!CollectionUtils.isEmpty(temporaryDataList)){
            CowManagerVo tem = this.pushTemporaryPersonnelActivity(temporaryDataList);
            if(tem.isSuccess()){
                try {
                    if((boolean)tem.getData()){
                        List<String> itemCodes = temporaryDataList.stream().map(CowManagerTemporaryPersonnelActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
                        successCodes.addAll(itemCodes);
                    }
                } catch (Exception e) {
                    log.warn("临促活动推送失败! cause by：",tem.getData());
                }

            }else {
                log.warn("临促活动推送失败! cause by：",tem.getData());
            }
        }
        //长促活动
        if(!CollectionUtils.isEmpty(longPersonDataList)){
            CowManagerVo lon = this.pushLongTermPersonnelActivity(longPersonDataList);
            if(lon.isSuccess()){
                try {
                    if((boolean)lon.getData()){
                        List<String> itemCodes = longPersonDataList.stream().map(CowManagerLongTermPersonnelActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
                        successCodes.addAll(itemCodes);
                    }
                } catch (Exception e) {
                    log.warn("长促活动推送失败! cause by：",lon.getData());
                }
            }else {
                log.warn("长促活动推送失败! cause by：",lon.getData());
            }
        }
        this.updateDetailPlanInterfaceState(successCodes);
    }

    /**
     * 推送行销活动
     * @param marketingDataList 行销活动集合
     */
    private CowManagerVo pushCellActivity(List<CowManagerCellActivityDto> marketingDataList) {
        try {
            CowManagerVo result = cowManagerService.pushCellActivity(marketingDataList);
            return result;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new CowManagerVo("2001",false,"接口调用异常","失败");
    }

    /**
     * 推送陈列活动
     * @param displayDataList 陈列活动集合
     */
    private CowManagerVo pushDisplayActivity(List<CowManagerDisplayActivityDto> displayDataList) {
        try {
            CowManagerVo result = cowManagerService.pushDisplayActivity(displayDataList);
            return result;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new CowManagerVo("2001",false,"接口调用异常","失败");
    }

    /**
     * 推送促销活动
     * @param promotionDataList 促销活动集合
     */
    private CowManagerVo pushPromotionActivity(List<CowManagerPromotionActivityDto> promotionDataList) {
        try {
            CowManagerVo result = cowManagerService.pushPromotionActivity(promotionDataList);
            return result;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new CowManagerVo("2001",false,"接口调用异常","失败");
    }

    /**
     * 推送临促活动
     * @param temporaryDataList 临促活动集合
     */
    private CowManagerVo pushTemporaryPersonnelActivity(List<CowManagerTemporaryPersonnelActivityDto> temporaryDataList) {
        try {
            CowManagerVo result = cowManagerService.pushTemporaryPersonnelActivity(temporaryDataList);
            return result;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new CowManagerVo("2001",false,"接口调用异常","失败");
    }

    /**
     * 推送长促活动
     * @param longPersonDataList 长促活动集合
     */
    private CowManagerVo pushLongTermPersonnelActivity(List<CowManagerLongTermPersonnelActivityDto> longPersonDataList) {
        try {
            CowManagerVo result = cowManagerService.pushLongTermPersonnelActivity(longPersonDataList);
            return result;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new CowManagerVo("2001",false,"接口调用异常","失败");
    }

    /**
     * 更新推送成功的明细状态
     * @param successCodes 明细编码集合
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateDetailPlanInterfaceState(List<String> successCodes) {
        if(CollectionUtils.isEmpty(successCodes)){
            return;
        }
        activityDetailPlanItemService.updateDetailPlanCowManagerState(successCodes);
    }
}
