package com.biz.crm.tpm.business.activity.plan.local.controller;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DataTooLongGenerateRespUtil;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlan;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanAsyncService;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanBudgetService;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanService;
import com.biz.crm.tpm.business.activity.plan.local.service.CorrectionDiscountRateQuantumService;
import com.biz.crm.tpm.business.activity.plan.local.vo.ActivityPlanApproveSubmitDto;
import com.biz.crm.tpm.business.activity.plan.sdk.constant.ActivityPlanConstant;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanBudgetDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanOutDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.sfa.ActivityPlanItemPushSfaDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.sfa.ActivityPlanItemPushSfaListDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.sfa.SfaActivityPlanItemStatusUpdateDto;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.ActivityPlanTypeEnum;
import com.biz.crm.tpm.business.activity.plan.sdk.event.ActivityPlanPushSfaEventListener;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyItemDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategyItemSdkService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategySdkService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyItemVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyVo;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 活动方案表(ActivityPlan)表相关的http接口
 *
 * @author wanghaojia
 * @since 2022-10-31 15:55:41
 */
@RestController
@RequestMapping("/v1/activityPlan")
@Slf4j
@Api(tags = "活动方案表")
public class ActivityPlanController {
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private ActivityPlanService activityPlanService;

    @Autowired(required = false)
    private ActivityPlanAsyncService activityPlanAsyncService;
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private ActivityPlanBudgetService activityPlanBudgetService;
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private MarketingStrategySdkService marketingStrategySdkService;
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private MarketingStrategyItemSdkService marketingStrategyItemSdkService;

    /**
     * redis服务
     */
    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private CorrectionDiscountRateQuantumService correctionDiscountRateQuantumService;


    /**
     * 分页查询所有数据
     *
     * @param pageable        分页对象
     * @param activityPlanDto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<ActivityPlanVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "activityDetailPlanDto", value = "查询实体") ActivityPlanDto activityPlanDto) {
        try {
            Page<ActivityPlanVo> page = this.activityPlanService.findByConditions(pageable, activityPlanDto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 活动方案新增编辑
     *
     * @return
     */
    @ApiOperation(value = "活动方案新增编辑-能力中心")
    @PostMapping(value = "saveActivityPlanNoCache")
    public Result<ActivityPlanOutDto> saveActivityPlanNoCache(@ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanOutDto dto) {
        try {
            loginUserService.refreshAuthentication(null);
            return Result.ok(this.activityPlanService.saveActivityPlanNoCache(dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, ActivityPlan.class));
        }
    }

    @ApiOperation("免费货物推送sap")
    @PostMapping("pushActivityToFreeGoods")
    public Result<?> pushActivityToFreeGoods(@RequestBody List<String> planCodes) {
        try {
            this.activityPlanService.pushActivityToFreeGoods(planCodes);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, ActivityPlan.class));
        }
    }

    /**
     * 活动方案新增编辑
     *
     * @param cacheKey 明细缓存key
     * @return
     */
    @ApiOperation(value = "活动方案新增编辑")
    @PostMapping(value = "saveActivityPlan")
    public Result saveActivityPlan(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                   @ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanDto dto) {
        try {
            UserIdentity loginUser = loginUserService.getLoginUser();
            //异步保存数据，通过websocket推送保存结果
            this.activityPlanAsyncService.saveActivityPlanAsync(dto, cacheKey, loginUser);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 活动方案新增原子接口
     *
     * @param dto 提交入参
     * @return
     */
    @ApiOperation(value = "活动方案新增原子接口")
    @PostMapping(value = "atomSaveActivityPlan")
    public Result atomSaveActivityPlan(@ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanDto dto) {
        try {
            return Result.ok(this.activityPlanService.atomSaveActivityPlan(dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, ActivityPlan.class));
        }
    }


    /**
     * 通过方案编码查询推送sfa数据
     */
    @ApiOperation(value = "通过方案编码查询推送sfa数据")
    @PostMapping(value = "findSfaDataByPlanCodes")
    public Result findSfaDataByPlanCodes(@RequestBody List<String> planCodeList) {
        try {
            List<ActivityPlanItemPushSfaDto> sfaDataByPlanCodes = this.activityPlanService.findSfaDataByPlanCodes(planCodeList);
            return Result.ok(sfaDataByPlanCodes);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 查询可以选择的营销策略
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据ø
     */
    @ApiOperation(value = "查询可以选择的营销策略")
    @GetMapping("findMarketingStrategyByConditions")
    public Result<Page<MarketingStrategyVo>> findMarketingStrategyByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                               @ApiParam(name = "MarketingStrategyDto", value = "策略查询实体") MarketingStrategyDto dto) {
        try {
//            Validate.notBlank(dto.getStrategyType(),"策略类型不能为空！");
            dto.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
            Page<MarketingStrategyVo> page = this.marketingStrategySdkService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询可以选择的营销策略
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据ø
     */
    @ApiOperation(value = "查询可以选择的营销策略")
    @GetMapping("findMarketingStrategyItemByConditions")
    public Result<Page<MarketingStrategyItemVo>> findMarketingStrategyItemByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                       @ApiParam(name = "MarketingStrategyItemDto", value = "策略查询实体") MarketingStrategyItemDto dto) {
        try {
//            Validate.notBlank(dto.getStrategyType(),"策略类型不能为空！");
            dto.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
            //大区方案查询总部测试的时候不根据部门过滤
//            if (StringUtils.isEmpty(dto.getStrategyType())) {
                dto.setAttrOrgQueryExcludeHeadStrategy(1);
//            }
            dto.setIsClose(BooleanEnum.FALSE.getCapital());
            Page<MarketingStrategyItemVo> page = this.marketingStrategyItemSdkService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 活动方案申请可以选择的总部方案
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据ø
     */
    @ApiOperation(value = "活动方案申请可以选择的总部方案")
    @GetMapping("findRelateActivityPlanListByConditions")
    public Result<Page<ActivityPlanVo>> findRelateActivityPlanListByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                               @ApiParam(name = "MarketingStrategyDto", value = "策略查询实体") ActivityPlanDto dto) {
        try {
            dto.setPlanType(ActivityPlanTypeEnum.headquarters.getCode());
            Page<ActivityPlanVo> page = this.activityPlanService.findRelateActivityPlanListByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 查询可以选择的月度预算
     *
     * @param pageable      分页对象
     * @param planBudgetDto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "查询可以选择的月度预算")
    @GetMapping("findMonthBudgetByConditions")
    public Result<Page<MonthBudgetVo>> findMonthBudgetByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                   @ApiParam(name = "MonthBudgetDto", value = "月度预算信息") ActivityPlanBudgetDto planBudgetDto,
                                                                   @ApiParam(name = "MonthBudgetDto", value = "月度预算信息") MonthBudgetDto monthBudgetDto) {
        try {
            Page<MonthBudgetVo> page = this.activityPlanService.findMonthBudgetByConditions(pageable, planBudgetDto, monthBudgetDto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过id获取活动方案数据
     */
    @ApiOperation(value = "通过id获取活动方案数据")
    @GetMapping("findById")
    public Result<ActivityPlanVo> findById(@ApiParam(value = "id") @RequestParam String id) {
        try {
            ActivityPlanVo vo = this.activityPlanService.findById(id);
            return Result.ok(vo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 扣减月度预算
     */
    @ApiOperation(value = "扣减月度预算")
    @PostMapping("useMonthBudgetByPlanCodeList")
    public Result useMonthBudgetByPlanCodeList(@RequestBody List<String> planCodeList) {
        try {
            this.activityPlanBudgetService.useMonthBudgetByPlanCodeList(planCodeList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交流程,多选批量提交
     */
    @ApiOperation(value = "提交流程,多选批量提交")
    @PostMapping(value = "submitApproval")
    public Result submitApproval(@ApiParam(value = "提交流程,多选批量提交")@RequestBody ActivityPlanApproveSubmitDto dto){

        List<String> ids = dto.getIds();
        boolean lockSuccess = false;
        try {
            Assert.notEmpty(ids, "请选择数据,加锁失败,请稍后重试!");
            //提交审批前加锁一个小时
            lockSuccess = redisLockService.batchLock(ActivityPlanConstant.LOCK_ACTIVITY_PLAN_APPROVE,ids, TimeUnit.HOURS, 1);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.activityPlanService.submitApproval(dto.getIds(),dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.batchUnLock(ActivityPlanConstant.LOCK_ACTIVITY_PLAN_APPROVE,ids);
            }
        }
    }

    /**
     * 删除活动方案
     *
     * @param ids 策略ids
     */
    @ApiOperation(value = "删除活动方案")
    @DeleteMapping(value = "delete")
    public Result delete(@ApiParam(value = "策略id集合") @RequestParam List<String> ids) {
        try {
            this.activityPlanService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除活动方案
     *
     * @param ids 策略ids
     */
    @ApiOperation(value = "删除活动方案")
    @PostMapping(value = "deletePos")
    public Result deletePos(@ApiParam(value = "策略id集合") @RequestParam List<String> ids) {
        try {
            this.activityPlanService.deleteForOut(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 更新SFA活动占用状态
     *
     * @param updateList 要更新的数据
     */
    @ApiOperation(value = "更新SFA活动占用状态")
    @PostMapping(value = "updateSfaActivityPlanState")
    public Result updateSfaActivityPlanState(@RequestBody List<SfaActivityPlanItemStatusUpdateDto> updateList) {
        try {
            log.info(JSONObject.toJSONString(updateList));
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * sfa推送活动方案
     */
    @ApiOperation(value = "sfa推送活动方案")
    @PostMapping(value = "pushSfaActByPlanCodeList")
    public Result pushSfaActByPlanCodeList(@RequestBody List<String> planCodeList){
        try {
            List<ActivityPlanItemPushSfaDto> dtos = activityPlanService.findSfaDataByPlanCodes(planCodeList);
            SerializableBiConsumer<ActivityPlanPushSfaEventListener, ActivityPlanItemPushSfaListDto> activityPlanProcessPass =
                    ActivityPlanPushSfaEventListener::pushActivityPlanListToSfa;
            ActivityPlanItemPushSfaListDto activityPlanItemPushSfaListDto = new ActivityPlanItemPushSfaListDto();
            ActivityPlanItemPushSfaDto activityPlanItemPushSfaDto = dtos.get(0);
            activityPlanItemPushSfaDto.setSfaPlanCode("HDFA146942043833483728937");
            activityPlanItemPushSfaDto.setActualAmount(new BigDecimal("5"));
            activityPlanItemPushSfaDto.setSuggestedAmount(new BigDecimal("6"));
            activityPlanItemPushSfaListDto.setList(dtos);
            this.nebulaNetEventClient.publish(activityPlanItemPushSfaListDto, ActivityPlanPushSfaEventListener.class, activityPlanProcessPass);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * sfa推送活动方案
     *
     * @param dtos 方案数据
     */
    @ApiOperation(value = "sfa推送活动方案")
    @PostMapping(value = "pushSfaAct")
    public Result pushSfaAct(@RequestBody List<ActivityPlanItemPushSfaDto> dtos) {
        try {
//            log.info(JSONObject.toJSONString(dtos));
//            log.info(JSON.toJSONString(new ActivityPlanItemPushSfaListDto(dtos)));

            SerializableBiConsumer<ActivityPlanPushSfaEventListener, ActivityPlanItemPushSfaListDto> activityPlanProcessPass =
                    ActivityPlanPushSfaEventListener::pushActivityPlanListToSfa;
            ActivityPlanItemPushSfaListDto activityPlanItemPushSfaListDto = new ActivityPlanItemPushSfaListDto();
            activityPlanItemPushSfaListDto.setList(dtos);
            this.nebulaNetEventClient.publish(activityPlanItemPushSfaListDto, ActivityPlanPushSfaEventListener.class, activityPlanProcessPass);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("合同生成方案")
    @DynamicTaskService(cornExpression = "0 0 2 28 * ?", taskDesc = "每月28号 02:00 执行一次合同生成方案")
    @GetMapping("/contractGenerateActivityPlan")
    public void contractGenerateActivityPlan() {
        boolean lockSuccess = false;
        Date date = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");

        String key = ActivityPlanConstant.LOCK_CONTRACT_GENERATE_ACTIVITY_PLAN + sdf.format(date);
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                return;
            }
            loginUserService.refreshAuthentication(null);
            activityPlanService.contractGenerateActivityPlan();
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }

    @ApiOperation("按岗店分摊")
    @PostMapping("proportionByTerminalEmployee")
    public Result proportionByTerminalEmployee(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                               @ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanDto dto) {
        String key = ActivityPlanConstant.LOCK_ACTIVITY_PLAN_PROPORTION + cacheKey;
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                throw new RuntimeException("操作正在进行中，请稍后查看或重试");
            }
            this.activityPlanService.proportionByTerminalEmployee(dto, cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }


    @ApiOperation("计算活动明细上的金额汇总信息")
    @PostMapping("calculateHeadAmount")
    public Result<ActivityPlanVo> calculateHeadAmount(ActivityPlanDto planDto,
                                                      @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey) {
        try {
            ActivityPlanVo planVo = this.activityPlanService.calculateHeadAmount(planDto, cacheKey);
            return Result.ok(planVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("按销售计划占比分摊")
    @PostMapping("calculateSalePlanProportion")
    public Result calculateSalePlanProportion(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                              @ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanDto dto) {
        String key = ActivityPlanConstant.LOCK_ACTIVITY_PLAN_PROPORTION + cacheKey;
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                throw new RuntimeException("操作正在进行中，请稍后查看或重试");
            }
            this.activityPlanService.calculateSalePlanProportion(dto, cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }

    @ApiOperation("按·门店分摊")
    @PostMapping("proportionByTerminal")
    public Result proportionByTerminal(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                       @ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanDto dto) {
        String key = ActivityPlanConstant.LOCK_ACTIVITY_PLAN_PROPORTION + cacheKey;
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                throw new RuntimeException("操作正在进行中，请稍后查看或重试");
            }
            this.activityPlanService.proportionByTerminal(dto, cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }

    @ApiOperation("退回月度预算")
    @PostMapping("returnMonthBudgetByPlanCodeList")
    public Result returnMonthBudgetByPlanCodeList(@RequestBody List<String> planCodeList) {
        try {
            activityPlanBudgetService.returnMonthBudgetByPlanCodeList(planCodeList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("垂直扣减大日期库存")
    @PostMapping("useInventoryCheck")
    public Result useInventoryCheck(@RequestBody List<String> planCodeList) {
        try {
            activityPlanService.useInventoryCheck(planCodeList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("垂直扣减大日期库存回退")
    @PostMapping("returnInventoryCheck")
    public Result returnInventoryCheck(@RequestBody List<String> planCodeList) {
        try {
            activityPlanService.returnInventoryCheck(planCodeList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("手动调用修正折扣率")
    @GetMapping("manualCorrectionDiscountRate")
    public Result<?> manualCorrectionDiscountRate(@RequestParam("yearMonth") String yearMonth,@RequestParam(value = "regionCode",required = false) String regionCode,@RequestParam("systemCode") String systemCode){
        try {
            correctionDiscountRateQuantumService.frJob(Collections.singletonList(yearMonth),regionCode,systemCode);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
