package com.biz.crm.tpm.business.activity.plan.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.page.cache.controller.MnPageCacheController;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanItemService;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanItemTerminalService;
import com.biz.crm.tpm.business.activity.plan.sdk.constant.ActivityPlanConstant;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanItemTerminalDto;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanItemTerminalVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.concurrent.TimeUnit;


/**
 * dengwei 方案明显门店分页查询
 */
@RestController
@RequestMapping("/v1/activityPlanItemTerminal")
@Slf4j
@Api(tags = "活动方案明细门店")
public class ActivityPlanItemTerminalController{

    @Autowired(required = false)
    private ActivityPlanItemTerminalService activityPlanItemTerminalService;


    @Autowired(required = false)
    private ActivityPlanItemService activityPlanItemService;

    /**
     * redis服务
     */
    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 1、获取明细列表缓存分页接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "获取明细列表缓存分页接口")
    @GetMapping("findCachePageList")
    public Result<Page<ActivityPlanItemTerminalVo>> findCachePageList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                              @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                              @ApiParam(name = "dto", value = "查询实体") ActivityPlanItemTerminalDto dto) {
        try {
            Page<ActivityPlanItemTerminalVo> page = this.activityPlanItemTerminalService.findCachePageList(pageable, dto, cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation("清理缓存接口")
    @PostMapping({"clearCache"})
    public Result clearCache(@ApiParam(name = "cacheKey",value = "缓存键") @RequestParam String cacheKey) {
        try {
            activityPlanItemTerminalService.clearCache(cacheKey);
            return Result.ok();
        } catch (Exception var3) {
            log.error(var3.getMessage(), var3);
            return Result.error(var3.getMessage());
        }
    }

    @ApiOperation(value = "新增一行接口，保存当前页数据后，在缓存中行首插入一条数据并返回第一页数据")
    @PostMapping("addItemCache")
    public Result<Page<ActivityPlanItemTerminalVo>> addItemCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                         @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                 @ApiParam(name = "dto", value = "查询实体") ActivityPlanItemTerminalDto dto,
                                         @ApiParam(value = "当前页数据") @RequestBody List<ActivityPlanItemTerminalDto> itemList) {
        try {
            this.activityPlanItemTerminalService.addItemCache(cacheKey,itemList);
            Page<ActivityPlanItemTerminalVo> page = this.activityPlanItemTerminalService.findCachePageList(pageable, null,cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 3、复制行接口：保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据")
    @PostMapping("copyItemListCache")
    public Result<Page<ActivityPlanItemTerminalVo>> copyItemListCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                              @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                      @ApiParam(name = "dto", value = "查询实体") ActivityPlanItemTerminalDto dto,
                                              @ApiParam(value = "当前页数据，包含要复制的行勾选信息") @RequestBody List<ActivityPlanItemTerminalDto> itemList) {
        try {
            this.activityPlanItemTerminalService.copyItemListCache(cacheKey, itemList);
            Page<ActivityPlanItemTerminalVo> page = this.activityPlanItemTerminalService.findCachePageList(pageable, null,cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "保存当前页数据到缓存并返回指定页数据接口")
    @PostMapping("saveCurrentPageCache")
    public Result<Page<ActivityPlanItemTerminalVo>> saveCurrentPageCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                 @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                         @ApiParam(name = "dto", value = "查询实体") ActivityPlanItemTerminalDto dto,
                                                 @ApiParam(value = "当前页数据") @RequestBody List<ActivityPlanItemTerminalDto> saveList) {
        try {
            this.activityPlanItemTerminalService.saveCurrentPageCache(cacheKey, saveList);
            Page<ActivityPlanItemTerminalVo> page = this.activityPlanItemTerminalService.findCachePageList(pageable, null,cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 7、删除行接口：删除有选中标记的行，更新未标记数据，并返回指定页数据接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "多行删除并返回指定页数据接口")
    @PostMapping("deleteCacheList")
    public Result<Page<ActivityPlanItemTerminalVo>> deleteCacheList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                            @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                                    @ApiParam(name = "dto", value = "查询实体") ActivityPlanItemTerminalDto dto,
                                            @ApiParam(value = "当前页数据，包含要删除的行勾选信息") @RequestBody List<ActivityPlanItemTerminalDto> itemList) {
        try {
            this.activityPlanItemTerminalService.deleteCacheList(cacheKey, itemList);
            Page<ActivityPlanItemTerminalVo> page = this.activityPlanItemTerminalService.findCachePageList(pageable, null,cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation("按·POS分摊")
    @PostMapping("proportionByPos")
    public Result proportionByPos(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                       @ApiParam(name = "ActivityPlanDto", value = "活动方案表头") @RequestBody ActivityPlanDto dto) {
        String key = ActivityPlanConstant.LOCK_ACTIVITY_PLAN_PROPORTION + cacheKey;
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                throw new RuntimeException("操作正在进行中，请稍后查看或重试");
            }
            this.activityPlanItemService.proportionByPos(dto, cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }

    @ApiOperation("初始化行明细涉及到的门店")
    @PostMapping("initActivityPlanItemTerminal")
    public Result initActivityPlanItemTerminal(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                               @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                       @ApiParam(name = "ActivityPlanItemTerminalDto", value = "门店数据查询") @RequestBody ActivityPlanItemTerminalDto dto) {
        String key = ActivityPlanConstant.LOCK_ACTIVITY_PLAN_PROPORTION + cacheKey;
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                throw new RuntimeException("操作正在进行中，请稍后查看或重试");
            }
            this.activityPlanItemTerminalService.initActivityPlanItemTerminal(dto, cacheKey);
            return findCachePageList(pageable,cacheKey,dto);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }

    @ApiOperation("按当前门店重新分摊合同费用")
    @PostMapping("proportionPosByCurrTerminal")
    public Result proportionPosByCurrTerminal(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                               @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                       @ApiParam(name = "ActivityPlanItemTerminalDto", value = "门店数据查询") @RequestBody ActivityPlanItemTerminalDto dto) {
        String key = ActivityPlanConstant.LOCK_ACTIVITY_PLAN_PROPORTION + cacheKey;
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(key, TimeUnit.MINUTES, 30);
            if (!lockSuccess) {
                throw new RuntimeException("操作正在进行中，请稍后查看或重试");
            }
            this.activityPlanItemTerminalService.proportionPosByCurrTerminal(dto, cacheKey);
            return findCachePageList(pageable,cacheKey,dto);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(key);
            }
        }
    }

}
