package com.biz.crm.tpm.business.activity.plan.local.imports;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.promoters.sdk.service.PromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.tpm.business.activity.plan.local.imports.vo.ActivityItemTerminalImportVo;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanItemTerminalService;
import com.biz.crm.tpm.business.activity.plan.local.service.internal.ActivityPlanItemPageCacheHelper;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanItemDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanItemTerminalDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description: 方案终端导入
 * @Author: dengwei
 * @Date: 2023/1/28 23:36
 * @Version: 1.0
 */
@Slf4j
@Component
public class ActivityItemTerminalImportsProcess implements ImportProcess<ActivityItemTerminalImportVo> {


    @Autowired(required = false)
    private ActivityPlanItemTerminalService activityPlanItemTerminalService;

    @Autowired(required = false)
    private TerminalVoService terminalVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private PromotersVoService promotersVoService;

    @Autowired(required = false)
    private ActivityPlanItemPageCacheHelper itemHelper;

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, ActivityItemTerminalImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        String cacheKey = (String)params.get("cacheKey");
        Validate.isTrue(!StringUtils.isEmpty(cacheKey),"缓存key不能空");
        Map<Integer, String> errorMap = Maps.newHashMap();

        List<ActivityPlanItemTerminalDto> itemTerminalDtoList = buildData(data,errorMap,cacheKey);

        if (errorMap.size() == 0){
            activityPlanItemTerminalService.importNewItem(cacheKey, itemTerminalDtoList);
        }
        return errorMap;
    }

    private List<ActivityPlanItemTerminalDto> buildData(LinkedHashMap<Integer, ActivityItemTerminalImportVo> data,Map<Integer, String> errorMap,String cacheKey) {
        if(CollectionUtils.isEmpty(data.values())){
            return null;
        }

        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        List<String> existsTerminalCodeList = Lists.newArrayList();
        List<ActivityPlanItemTerminalDto> itemTerminalList = itemDto.getActivityPlanItemTerminalList();
        if (null != itemTerminalList){
            existsTerminalCodeList = itemTerminalList.stream().map(ActivityPlanItemTerminalDto::getTerminalCode).collect(Collectors.toList());
        }

        Map<Integer, List<String>> errorListMap = Maps.newHashMap();
        Set<Map.Entry<Integer, ActivityItemTerminalImportVo>> dataEntry = data.entrySet();
        Set<String> terminalCodeSet = Sets.newHashSet();
        Set<String> employeeCodeSet = Sets.newHashSet();
        for (Map.Entry<Integer, ActivityItemTerminalImportVo> entry : dataEntry) {
            ActivityItemTerminalImportVo value = entry.getValue();
            Integer key = entry.getKey();
            List<String> errorList = errorListMap.computeIfAbsent(key, k -> Lists.newArrayList());
            try {
                if(StringUtils.isEmpty(value.getTerminalCode())){
                    errorList.add("终端编码必须输入");
                }else{
                    terminalCodeSet.add(value.getTerminalCode());
                }
                if(Objects.nonNull(value.getFloatingRate())){
                    if(value.getFloatingRate().compareTo(BigDecimal.ZERO)<0||value.getFloatingRate().compareTo(BigDecimal.ONE)>0){
                        errorList.add("浮动率只能在0到1之间");
                    }
                }
                if (StringUtils.isEmpty(value.getEmployeeCode())){
                    employeeCodeSet.add(value.getEmployeeCode());
                }
                if (null == value.getAmount()){
                    errorList.add("费用总金额不能为空");
                }
                if (null == value.getFeeAmount()){
                    errorList.add("我方承担金额不能为空");
                }
            }catch (Exception e){
                errorList.add(e.getMessage());
            }
        }

        Map<String, TerminalVo> terminalVoMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(terminalCodeSet)){
            List<TerminalVo> terminalVoList = terminalVoService.findDetailsByIdsOrTerminalCodes(null, new ArrayList<>(terminalCodeSet));
            terminalVoMap = terminalVoList.stream().collect(Collectors.toMap(TerminalVo::getTerminalCode, Function.identity(),(o,n)->n));
        }
        Map<String, PromotersVo> promotersVoMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(employeeCodeSet)){
            List<PromotersVo> promotersVoList = promotersVoService.findByCodes(new ArrayList<>(employeeCodeSet));
            promotersVoMap = promotersVoList.stream().collect(Collectors.toMap(PromotersVo::getEmployeeCode, Function.identity(),(o,n)->n));
        }

        List<ActivityPlanItemTerminalDto> activityPlanItemTerminalDtos = Lists.newArrayList();
        for (Map.Entry<Integer, ActivityItemTerminalImportVo> entry : dataEntry) {
            ActivityItemTerminalImportVo value = entry.getValue();
            Integer key = entry.getKey();
            List<String> errorList = errorListMap.computeIfAbsent(key, k -> Lists.newArrayList());
            try {
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = nebulaToolkitService.copyObjectByWhiteList(value,ActivityPlanItemTerminalDto.class,HashSet.class,ArrayList.class);
                activityPlanItemTerminalDtos.add(activityPlanItemTerminalDto);
                if (StringUtils.isNotEmpty(activityPlanItemTerminalDto.getTerminalCode())){
                    TerminalVo terminalVo = terminalVoMap.get(activityPlanItemTerminalDto.getTerminalCode());
                    if(terminalVo==null){
                        errorList.add("终端"+activityPlanItemTerminalDto.getTerminalCode()+"不存在");
                    }else {
                        if(!EnableStatusEnum.ENABLE.getCode().equals(terminalVo.getEnableStatus()) || !BooleanEnum.TRUE.getNumStr().equals(terminalVo.getTerminalState())){
                            errorList.add("方案中包含已被禁用、已关闭的门店，门店编码为"+terminalVo.getTerminalCode()+",请校对后再提交");
                        }
                        activityPlanItemTerminalDto.setTerminalName(terminalVo.getTerminalName());
                    }
                    if (existsTerminalCodeList.contains(activityPlanItemTerminalDto.getTerminalCode())){
                        errorList.add("门店编码"+terminalVo.getTerminalCode()+"重复添加");
                    }
                    existsTerminalCodeList.add(activityPlanItemTerminalDto.getTerminalCode());
                }
                if(StringUtils.isNotEmpty(activityPlanItemTerminalDto.getEmployeeCode())){
                    PromotersVo promotersVo = promotersVoMap.get(activityPlanItemTerminalDto.getEmployeeCode());
                    if(promotersVo==null){
                        errorList.add("促销人员"+activityPlanItemTerminalDto.getEmployeeCode()+"不存在");
                    }else {
                        activityPlanItemTerminalDto.setEmpId(promotersVo.getEmpId());
                        activityPlanItemTerminalDto.setIdentityCard(promotersVo.getIdentityCard());
                        activityPlanItemTerminalDto.setResponsibleSupervision(promotersVo.getResponsibleSupervision());
                        activityPlanItemTerminalDto.setName(promotersVo.getName());
                        activityPlanItemTerminalDto.setPhone(promotersVo.getPhone());
                        activityPlanItemTerminalDto.setResponsibleSupervision(promotersVo.getResponsibleSupervision());
                    }
                }
            }catch (Exception e){
                log.error(e.getMessage(),e);
                errorList.add(e.getMessage());
            }
        }
        if (errorListMap.size() > 0){
            for (Map.Entry<Integer, List<String>> entry : errorListMap.entrySet()) {
                List<String> value = entry.getValue();
                if (value.size() > 0){
                    errorMap.put(entry.getKey(),String.join(",",value));
                }
            }
        }
        return activityPlanItemTerminalDtos;
    }

    @Override
    public Class<ActivityItemTerminalImportVo> findCrmExcelVoClass() {
        return ActivityItemTerminalImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "ACTIVITY_ITEM_TERMINAL";
    }

    @Override
    public String getTemplateName() {
        return "tpm-方案明细终端导入";
    }
}
