package com.biz.crm.tpm.business.activity.plan.local.repository;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.util.ObjectConvertStringUtil;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanBudget;
import com.biz.crm.tpm.business.activity.plan.local.mapper.ActivityPlanBudgetMapper;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanBudgetDto;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanBudgetVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyBudgetVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.StrategyActivityPlanItemVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.beans.PropertyDescriptor;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 活动方案表(ActivityPlan)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-10-31 15:55:54
 */
@Component
public class ActivityPlanBudgetRepository extends ServiceImpl<ActivityPlanBudgetMapper, ActivityPlanBudget> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 根据方案编码，查询方案对应预算
     * @param planCode 方案编码
     **/
    public List<ActivityPlanBudget> listByPlanCode(String planCode) {
        if (StringUtils.isEmpty(planCode)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(ActivityPlanBudget::getPlanCode, planCode)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据方案编码查询
     **/
    public List<ActivityPlanBudgetDto> listDtoListByPlanCode(String planCode) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(planCode)) {
            return null;
        }
        return baseMapper.listDtoListByPlanCode(planCode);
    }

    public List<ActivityPlanBudget> listByOperateCodeList(List<String> operateCodeList) {
        if (CollectionUtils.isEmpty(operateCodeList)) {
            return null;
        }
        return baseMapper.listByOperateCodeList(operateCodeList);
    }

    public LinkedList<ActivityPlanBudget> listByOperateCodeListVertical(List<String> operateCodeList) {
        if (CollectionUtils.isEmpty(operateCodeList)) {
            return null;
        }
        return baseMapper.listByOperateCodeListVertical(operateCodeList);
    }

    public List<ActivityPlanBudget> listCustomerBudgetByOperateCodeList(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)) {
            return null;
        }
        return baseMapper.listCustomerBudgetByPlanItemCodeList(planItemCodeList);
    }

    /**
     * 根据方案编码，查询方案对应预算
     * @param planCodeList 方案编码集合
     **/
    public List<ActivityPlanBudgetDto> listDtoByPlanCodeList(List<String> planCodeList) {
        List<ActivityPlanBudget> activityPlanBudgets = listByPlanCodeList(planCodeList);
        if (CollectionUtils.isNotEmpty(activityPlanBudgets)){
            List<ActivityPlanBudgetDto> dtoList = (List<ActivityPlanBudgetDto>) nebulaToolkitService.copyCollectionByBlankList(activityPlanBudgets,ActivityPlanBudget.class,ActivityPlanBudgetDto.class, HashSet.class, ArrayList.class);
            return dtoList;
        }
        return Lists.newArrayList();
    }
    /**
     * 根据方案编码，查询方案对应预算
     * @param planItemCodeList 方案明细编码集合
     **/
    public List<ActivityPlanBudgetDto> listDtoByPlanItemCodeList(List<String> planItemCodeList) {
        List<ActivityPlanBudget> activityPlanBudgets = listByPlanItemCodeList(planItemCodeList);
        if (CollectionUtils.isNotEmpty(activityPlanBudgets)){
            List<ActivityPlanBudgetDto> dtoList = (List<ActivityPlanBudgetDto>) nebulaToolkitService.copyCollectionByBlankList(activityPlanBudgets,ActivityPlanBudget.class,ActivityPlanBudgetDto.class, HashSet.class, ArrayList.class);
            return dtoList;
        }
        return Lists.newArrayList();
    }

    public List<ActivityPlanBudget> listByPlanItemCodeList(List<String> planItemCodeList) {
        if (org.springframework.util.CollectionUtils.isEmpty(planItemCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(ActivityPlanBudget::getPlanItemCode, planItemCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据方案编码，查询方案对应预算
     * @param planCodeList 方案编码集合
     **/
    public List<ActivityPlanBudget> listByPlanCodeList(List<String> planCodeList) {
        if (org.springframework.util.CollectionUtils.isEmpty(planCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(ActivityPlanBudget::getPlanCode, planCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据明细编码，批量查询方案明细对应预算
     *
     * @param itemCodeList
     * @return java.util.List<com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanBudget>
     * @author huojia
     * @date 2022/11/28 16:56
     **/
    public List<ActivityPlanBudget> listByDetailCodeList(List<String> itemCodeList) {
        if (CollectionUtils.isEmpty(itemCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(ActivityPlanBudget::getPlanItemCode, itemCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    public void deleteByIds(List<String> ids) {
        if (org.springframework.util.CollectionUtils.isEmpty(ids)){
            return;
        }
        LambdaUpdateWrapper<ActivityPlanBudget> updateWrapper = new UpdateWrapper<ActivityPlanBudget>().lambda();
        updateWrapper.set(ActivityPlanBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(ActivityPlanBudget::getId, ids);
        this.update(updateWrapper);
    }

    public void deletePhysicalByPlanCode(String planCode) {
        if (StringUtils.isEmpty(planCode)){
            return;
        }
        LambdaQueryWrapper<ActivityPlanBudget> updateWrapper = new LambdaQueryWrapper<ActivityPlanBudget>();
//        updateWrapper.set(ActivityPlanBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.eq(ActivityPlanBudget::getPlanCode, planCode);
        this.remove(updateWrapper);
    }

    public void deleteByPlanCodes(List<String> planCodes) {
        if (org.springframework.util.CollectionUtils.isEmpty(planCodes)){
            return;
        }
        LambdaUpdateWrapper<ActivityPlanBudget> updateWrapper = new UpdateWrapper<ActivityPlanBudget>().lambda();
        updateWrapper.set(ActivityPlanBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(ActivityPlanBudget::getPlanCode, planCodes);
        this.update(updateWrapper);
    }

    public Page<ActivityPlanBudgetVo> findByConditions(Pageable pageable, ActivityPlanBudgetDto dto) {
        Page<MarketingStrategyBudgetVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<ActivityPlanBudgetVo> pageList = this.baseMapper.findConditions(page, dto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }


    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoListProperties(List<ActivityPlanBudgetVo> voList) {
        if (org.springframework.util.CollectionUtils.isEmpty(voList)){
            return;
        }
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(ActivityPlanBudgetVo.class);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors).collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        for (ActivityPlanBudgetVo item : voList) {
            this.fillVoProperties(item,propertyMap);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(ActivityPlanBudgetVo item) {
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(ActivityPlanBudgetVo.class);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors).collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        this.fillVoProperties(item,propertyMap);
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(ActivityPlanBudgetVo item,Map<String, PropertyDescriptor> propertyMap) {
        if (null == item){
            return;
        }
        //统一做BigDecimal,integer,Date类型格式转换
        ObjectConvertStringUtil.fillObjectStrProperties(item, ActivityPlanBudgetVo.class,propertyMap);
        BigDecimal useAmount = Optional.ofNullable(item.getUseAmount()).orElse(BigDecimal.ZERO);
        BigDecimal usedAmount = Optional.ofNullable(item.getUsedAmount()).orElse(BigDecimal.ZERO);
        item.setUsedAmount(usedAmount);
        item.setUsableAmount(useAmount.subtract(usedAmount));
    }

    public List<ActivityPlanBudget> findCarGiftCloseReturnBudgetList(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)){
            return Lists.newArrayList();
        }
        return baseMapper.findCarGiftCloseReturnBudgetList(planItemCodeList);
    }

    /**
     * 查询未关闭的方案明细编码（无大区方案关联，或已关联的方案完全关闭）
     *
     * @param itemNos 方案明细编码
     * @return List<String>
     **/
    public List<String> getNoClosePlanItemNo(List<String> itemNos){
        if(CollectionUtils.isEmpty(itemNos)){
            return Lists.newArrayList();
        }
        return baseMapper.getNoClosePlanItemNo(itemNos,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanBudgetVo> findByPlanItemCode(Set<String> codeSet) {
        return baseMapper.findByPlanItemCode(codeSet,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanBudgetVo> findByPlanCode(Set<String> planSet) {
        return baseMapper.findByPlanCode(planSet,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanBudgetVo> findItemBudgetListByPlanCode(String planCode) {
        if(StringUtils.isEmpty(planCode)){
            return Lists.newArrayList();
        }
        List<ActivityPlanBudget> list = this.lambdaQuery().eq(ActivityPlanBudget::getRelatePlanCode, planCode)
                .eq(ActivityPlanBudget::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
        if(CollectionUtils.isNotEmpty(list)){
            Collection<ActivityPlanBudgetVo> activityPlanBudgetVos = this.nebulaToolkitService.copyCollectionByWhiteList(list, ActivityPlanBudget.class, ActivityPlanBudgetVo.class, LinkedHashSet.class, ArrayList.class);
            return (List<ActivityPlanBudgetVo>) activityPlanBudgetVos;
        }
        return Lists.newArrayList();
    }

    public List<ActivityPlanBudget> findItemBudgetListByPlanItemCode(String planItemCode) {
        if(StringUtils.isEmpty(planItemCode)){
            return Lists.newArrayList();
        }
        return this.lambdaQuery().eq(ActivityPlanBudget::getPlanItemCode, planItemCode)
                .eq(ActivityPlanBudget::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityPlanBudget::getTenantCode,TenantUtils.getTenantCode())
                .list();
    }

    public List<String> strategyRelatePlanCodeList(Set<String> planCodeList) {
        return baseMapper.strategyRelatePlanCodeList(planCodeList,TenantUtils.getTenantCode());
    }

    public List<StrategyActivityPlanItemVo> findActivityPlanByStrategyItemCodeList(List<String> strategyItemCodeList) {
        return baseMapper.findActivityPlanByStrategyItemCodeList(strategyItemCodeList,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanBudgetVo> findRegionIsRelateHead(List<String> planCodes, String tenantCode) {
        return this.baseMapper.findRegionIsRelateHead(planCodes,tenantCode);
    }
}

