package com.biz.crm.tpm.business.activity.plan.local.repository;


import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.util.NumberStringDealUtil;
import com.biz.crm.mn.common.base.util.ObjectConvertStringUtil;
import com.biz.crm.mn.common.extend.field.service.ExtendFieldService;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItem;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItemExtend;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItemExtendField;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItemProduct;
import com.biz.crm.tpm.business.activity.plan.local.mapper.ActivityPlanItemMapper;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.*;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.ActivityPlanBudgetOccupyTypeEnum;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.ActivityPlanTypeEnum;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.InterfacePushStateEnum;
import com.biz.crm.tpm.business.activity.plan.sdk.pojo.ActivityPlanItemBase;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanItemBudgetVo;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanItemVo;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.RedPacketQueryScheme2Vo;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.submitreportvo.ActivityPlanSubmitReportSubVo;
import com.biz.crm.tpm.business.month.budget.sdk.constant.MonthBudgetConstant;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.YesOrNoEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import liquibase.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.binding.MapperMethod;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 活动方案明细表(ActivityPlanItem)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-11-01 14:39:19
 */
@Component
@Slf4j
public class ActivityPlanItemRepository extends ServiceImpl<ActivityPlanItemMapper, ActivityPlanItem> {

    @Autowired(required = false)
    private ActivityPlanItemMapper activityPlanItemMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ActivityPlanBudgetRepository activityPlanBudgetRepository;

    @Autowired(required = false)
    private ActivityPlanItemExtendRepository activityPlanItemExtendRepository;

    @Autowired(required = false)
    private ActivityPlanItemExtendFieldRepository activityPlanItemExtendFieldRepository;


    @Autowired(required = false)
    private ActivityPlanItemProductRepository activityPlanItemProductRepository;

    @Autowired(required = false)
    private ActivityPlanItemTerminalRepository activityPlanItemTerminalRepository;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;
    @Autowired(required = false)
    private ExtendFieldService extendFieldService;

    private final static String DATE_FORMAT_STR = "%Y-%m";

    /**
     * 根据方案编码查询方案明细数据
     * @param planCode 方案编码
     * @return 方案明细dtolist
     */
    public List<ActivityPlanItemDto> findDtoListByPlanCode(String planCode) {
        if (StringUtils.isBlank(planCode)){
            return null;
        }
        ActivityPlanItemDto dto = new ActivityPlanItemDto();
        dto.setPlanCode(planCode);
        return findDtoList(dto);
    }
    /**
     * 根据方案编码查询方案明细数据
     * @param planCodeList 方案编码集合
     * @return 方案明细dtolist
     */
    public List<ActivityPlanItemDto> findDtoListByPlanCodeList(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)){
            return null;
        }
        ActivityPlanItemDto dto = new ActivityPlanItemDto();
        dto.setPlanCodeList(planCodeList);
        return findDtoList(dto);
    }

    /**
     * 根据方案编码查询方案明细数据
     * @param planCodeList 方案编码集合
     * @return 方案明细dtolist
     */
    public List<ActivityPlanItemVo> findVoListByPlanCodeList(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)){
            return Collections.emptyList();
        }
        ActivityPlanItemDto dto = new ActivityPlanItemDto();
        dto.setPlanCodeList(planCodeList);
        return findVoList(dto);
    }

    /**
     * 查询dto集合
     * @param planItem 查询参数
     * @return
     */
    public List<ActivityPlanItemVo> findVoList(ActivityPlanItemDto planItem) {
        List<ActivityPlanItem> itemList = findList(planItem);
        List<ActivityPlanItemVo> voList = (List<ActivityPlanItemVo>) nebulaToolkitService.copyCollectionByWhiteList(itemList, ActivityPlanItem.class, ActivityPlanItemVo.class, HashSet.class, ArrayList.class);
        fillVoListProperties(voList);
        return voList;
    }

    /**
     * 根据方案编码查询方案明细数据
     * @param planItemCodeList 方案明细编码
     * @return 方案明细dtolist
     */
    public List<ActivityPlanItemDto> findDtoListByPlanItemCodeList(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)){
            return Lists.newArrayList();
        }
        ActivityPlanItemDto dto = new ActivityPlanItemDto();
        dto.setPlanItemCodeList(planItemCodeList);
        return findDtoList(dto);
    }

    /**
     * 查策略明细信息及其附加其他信息
     * @param planItemCodeList 方案明细编码
     * @return
     */
    public List<ActivityPlanItemDto> findDtoAndAttachListByPlanItemCodeList(List<String> planItemCodeList){
        List<ActivityPlanItemDto> dtoList = findDtoListByPlanItemCodeList(planItemCodeList);
        buildDtoAttach(dtoList);
        return dtoList;
    }

    /**
     * 查策略明细信息及其附加其他信息
     * @param planCode 方案编码
     * @return
     */
    public List<ActivityPlanItemDto> findDtoAndAttachListByPlanCode(String planCode){
        List<ActivityPlanItemDto> dtoList = findDtoListByPlanCode(planCode);
        buildDtoAttach(dtoList);
        return dtoList;
    }
    /**
     * 查策略明细信息及其附加其他信息
     * @param planCodeList 方案编码集合
     */
    public List<ActivityPlanItemDto> findDtoAndAttachListByPlanCodeList(List<String> planCodeList){
        List<ActivityPlanItemDto> dtoList = findDtoListByPlanCodeList(planCodeList);
        buildDtoAttach(dtoList);
        return dtoList;
    }

    private void buildDtoAttach(List<ActivityPlanItemDto> dtoList){
        List<String> planItemCodeList = dtoList.stream().map(ActivityPlanItemDto::getPlanItemCode).collect(Collectors.toList());
        //营销策略就附带一个预算
        List<ActivityPlanBudgetDto> budgetList = activityPlanBudgetRepository.listDtoByPlanItemCodeList(planItemCodeList);
        if (!CollectionUtils.isEmpty(budgetList)){
            convertMonthBudgetProperty(budgetList);
            for (ActivityPlanBudgetDto budgetDto : budgetList) {
                NumberStringDealUtil.setStringIfNotNull(budgetDto.getUseAmount(), budgetDto::setUseAmountStr);
            }
            Map<String, List<ActivityPlanBudgetDto>> budgetMap = budgetList.stream().filter(item -> StringUtils.isNotBlank(item.getPlanItemCode())).collect(Collectors.groupingBy(ActivityPlanBudgetDto::getPlanItemCode));
            for (ActivityPlanItemDto itemDto : dtoList) {
                if (!budgetMap.containsKey(itemDto.getPlanItemCode())){
                    continue;
                }
                List<ActivityPlanBudgetDto> budgetShares = budgetMap.get(itemDto.getPlanItemCode());
                BigDecimal usedHeadFeeAmount = BigDecimal.ZERO;
                BigDecimal usedDepartmentFeeAmount = BigDecimal.ZERO;
                BigDecimal usedCustomerFeeAmount = BigDecimal.ZERO;
                for (ActivityPlanBudgetDto budgetShare : budgetShares) {
                    BigDecimal usedAmount = Optional.ofNullable(budgetShare.getUsedAmount()).orElse(BigDecimal.ZERO);
                    if (StringUtils.isNotBlank(budgetShare.getMonthBudgetCode()) && budgetShare.getMonthBudgetCode().equals(itemDto.getHeadMonthBudgetCode())){
                        usedHeadFeeAmount = usedHeadFeeAmount.add(usedAmount);
                    }
                    if (StringUtils.isNotBlank(budgetShare.getMonthBudgetCode()) && budgetShare.getMonthBudgetCode().equals(itemDto.getMonthBudgetCode())){
                        usedDepartmentFeeAmount = usedDepartmentFeeAmount.add(usedAmount);
                    }
                    if (ActivityPlanBudgetOccupyTypeEnum.CUSTOMER.getCode().equals(budgetShare.getOccupyType())){
                        usedCustomerFeeAmount = usedCustomerFeeAmount.add(usedAmount);
                    }
                }
                itemDto.setUsedHeadFeeAmount(usedHeadFeeAmount);
                itemDto.setUsedHeadFeeAmountStr(usedHeadFeeAmount.toString());
                itemDto.setUsedDepartmentFeeAmount(usedDepartmentFeeAmount);
                itemDto.setUsedDepartmentFeeAmountStr(usedDepartmentFeeAmount.toString());
                itemDto.setUsedCustomerFeeAmount(usedCustomerFeeAmount);
                itemDto.setUsedCustomerFeeAmountStr(usedCustomerFeeAmount.toString());
                itemDto.setBudgetShares(budgetShares);
            }
        }
        List<ActivityPlanItemExtendField> itemExtendFields = activityPlanItemExtendFieldRepository.listByPlanItemCodeList(planItemCodeList);
        if (!CollectionUtils.isEmpty(itemExtendFields)){
            Map<String, List<ActivityPlanItemExtendField>> extendFieldMap = itemExtendFields.stream().collect(Collectors.groupingBy(ActivityPlanItemExtendField::getPlanItemCode));
            for (ActivityPlanItemDto itemDto : dtoList) {
                extendFieldService.fillExtendField(itemDto,ActivityPlanItemDto.class,extendFieldMap.get(itemDto.getPlanItemCode()));
            }
        }
        List<ActivityPlanItemTerminalDto> terminalList = activityPlanItemTerminalRepository.listDtoByPlanItemCodeList(planItemCodeList);
        if (!CollectionUtils.isEmpty(terminalList)){
            Map<String, List<ActivityPlanItemTerminalDto>> terminalMap = terminalList.stream().collect(Collectors.groupingBy(ActivityPlanItemTerminalDto::getPlanItemCode));
            for (ActivityPlanItemDto itemDto : dtoList) {
                if (terminalMap.containsKey(itemDto.getPlanItemCode())){
                    itemDto.setActivityPlanItemTerminalList(terminalMap.get(itemDto.getPlanItemCode()));
                }
            }
        }
    }

    /**
     * 查策略明细信息及其附加其他信息
     * @param planItemCodeList 方案明细编码
     * @return
     */
    public List<ActivityPlanItemDto> findDtoAndAttachListByPlanItemCode(List<String> planItemCodeList){
        List<ActivityPlanItemDto> dtoList = findDtoListByPlanItemCodeList(planItemCodeList);
        //营销策略就附带一个预算
        List<ActivityPlanBudgetDto> budgetList = activityPlanBudgetRepository.listDtoByPlanItemCodeList(planItemCodeList);
        if (!CollectionUtils.isEmpty(budgetList)){
            convertMonthBudgetProperty(budgetList);
            for (ActivityPlanBudgetDto budgetDto : budgetList) {
                NumberStringDealUtil.setStringIfNotNull(budgetDto.getUseAmount(), budgetDto::setUseAmountStr);
            }
            Map<String, List<ActivityPlanBudgetDto>> budgetMap = budgetList.stream().filter(item -> StringUtils.isNotBlank(item.getPlanItemCode())).collect(Collectors.groupingBy(ActivityPlanBudgetDto::getPlanItemCode));
            for (ActivityPlanItemDto itemDto : dtoList) {
                if (!budgetMap.containsKey(itemDto.getPlanItemCode())){
                    continue;
                }
                List<ActivityPlanBudgetDto> budgetShares = budgetMap.get(itemDto.getPlanItemCode());
                BigDecimal usedHeadFeeAmount = BigDecimal.ZERO;
                BigDecimal usedDepartmentFeeAmount = BigDecimal.ZERO;
                BigDecimal usedCustomerFeeAmount = BigDecimal.ZERO;
                for (ActivityPlanBudgetDto budgetShare : budgetShares) {
                    BigDecimal usedAmount = Optional.ofNullable(budgetShare.getUsedAmount()).orElse(BigDecimal.ZERO);
                    if (StringUtils.isNotBlank(budgetShare.getMonthBudgetCode()) && budgetShare.getMonthBudgetCode().equals(itemDto.getHeadMonthBudgetCode())){
                        usedHeadFeeAmount = usedHeadFeeAmount.add(usedAmount);
                    }
                    if (StringUtils.isNotBlank(budgetShare.getMonthBudgetCode()) && budgetShare.getMonthBudgetCode().equals(itemDto.getMonthBudgetCode())){
                        usedDepartmentFeeAmount = usedDepartmentFeeAmount.add(usedAmount);
                    }
                    if (ActivityPlanBudgetOccupyTypeEnum.CUSTOMER.getCode().equals(budgetShare.getOccupyType())){
                        usedCustomerFeeAmount = usedCustomerFeeAmount.add(usedAmount);
                    }
                }
                itemDto.setUsedHeadFeeAmount(usedHeadFeeAmount);
                itemDto.setUsedHeadFeeAmountStr(usedHeadFeeAmount.toString());
                itemDto.setUsedDepartmentFeeAmount(usedDepartmentFeeAmount);
                itemDto.setUsedDepartmentFeeAmountStr(usedDepartmentFeeAmount.toString());
                itemDto.setUsedCustomerFeeAmount(usedCustomerFeeAmount);
                itemDto.setUsedCustomerFeeAmountStr(usedCustomerFeeAmount.toString());
                itemDto.setBudgetShares(budgetShares);
            }
        }
        List<ActivityPlanItemExtendField> itemExtendFields = activityPlanItemExtendFieldRepository.listByPlanItemCodeList(planItemCodeList);
        if (!CollectionUtils.isEmpty(itemExtendFields)){
            Map<String, List<ActivityPlanItemExtendField>> extendFieldMap = itemExtendFields.stream().collect(Collectors.groupingBy(ActivityPlanItemExtendField::getPlanItemCode));
            for (ActivityPlanItemDto itemDto : dtoList) {
                extendFieldService.fillExtendField(itemDto,ActivityPlanItemDto.class,extendFieldMap.get(itemDto.getPlanItemCode()));
            }
        }
        return dtoList;
    }

    /**
     * 转换字段
     */
    public void convertMonthBudgetProperty(List<ActivityPlanBudgetDto> list){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        Map<String,String> feeSourceMap = Maps.newHashMap();
        if (null != dictDataVoService){
            try {
                List<DictDataVo> feeSourceList = dictDataVoService.findByDictTypeCode(MonthBudgetConstant.DICT_TPM_FEE_BELONG_US);
                if (!CollectionUtils.isEmpty(feeSourceList)) {
                    feeSourceMap = feeSourceList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue, (o, n) -> o));
                }

            } catch (Exception e) {
                log.error("费用归口数据字典查询失败：" + e.getMessage());
            }
        }
        for (ActivityPlanBudgetDto vo : list) {
            //费用来源转换
            if (org.apache.commons.lang3.StringUtils.isNotEmpty(vo.getFeeBelongCode())){
                vo.setFeeSourceName(feeSourceMap.getOrDefault(vo.getFeeBelongCode(),vo.getFeeBelongCode()));
            }
        }
    }

    /**
     * 查询dto集合
     * @param planItem 查询参数
     * @return
     */
    public List<ActivityPlanItemDto> findDtoList(ActivityPlanItemDto planItem) {
        List<ActivityPlanItem> itemList = findList(planItem);
        List<ActivityPlanItemDto> dtoList = (List<ActivityPlanItemDto>) nebulaToolkitService.copyCollectionByWhiteList(itemList, ActivityPlanItem.class, ActivityPlanItemDto.class, HashSet.class, ArrayList.class);
        fillDtoListProperties(dtoList);
        return dtoList;
    }

    /**
     * 根据方案编码查询方案明细数据
     * @param planCode 方案编码
     * @return 方案明细dtolist
     */
    public List<ActivityPlanItem> findListByPlanCode(String planCode) {
        if (StringUtils.isBlank(planCode)){
            return null;
        }
        ActivityPlanItemDto activityPlanItem = new ActivityPlanItemDto();
        activityPlanItem.setPlanCode(planCode);
        return findList(activityPlanItem);
    }

    /**
     * 根据查询条件筛选集合
     * @param planItem 查询参数
     */
    public List<ActivityPlanItem> findList(ActivityPlanItemDto planItem) {
        if (StringUtils.isBlank(planItem.getPlanCode())
                && CollectionUtil.isEmpty(planItem.getPlanCodeList())
                && CollectionUtil.isEmpty(planItem.getPlanItemCodeList())) {
            return Collections.emptyList();
        }
        LambdaQueryWrapper<ActivityPlanItem> wrapper = Wrappers.lambdaQuery(ActivityPlanItem.class)
                .eq(StringUtils.isNotBlank(planItem.getPlanCode()), ActivityPlanItem::getPlanCode, planItem.getPlanCode())
                .eq(StringUtils.isNotBlank(planItem.getTemplateConfigCode()), ActivityPlanItem::getTemplateConfigCode, planItem.getTemplateConfigCode())
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityPlanItem::getTenantCode, TenantUtils.getTenantCode());
        if (CollectionUtil.isNotEmpty(planItem.getPlanItemCodeList())) {
            wrapper.in(ActivityPlanItem::getPlanItemCode, planItem.getPlanItemCodeList());
        }
        if (!CollectionUtils.isEmpty(planItem.getPlanCodeList())) {
            wrapper.in(ActivityPlanItem::getPlanCode, planItem.getPlanCodeList());
        }
        return this.list(wrapper);
    }

    /**
     * 根据查询条件筛选集合
     */
    public List<ActivityPlanItem> findListByCodeList(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)){
            return Lists.newArrayList();
        }
        return this.list(Wrappers.lambdaQuery(ActivityPlanItem.class)
                .in(ActivityPlanItem::getPlanCode,planCodeList)
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        );
    }


    /**
     * 通过id删除
     * @param ids id集合
     */
    public void deleteByIds(List<String> ids){
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        LambdaUpdateWrapper<ActivityPlanItem> updateWrapper = new UpdateWrapper<ActivityPlanItem>().lambda();
        updateWrapper.set(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(ActivityPlanItem::getId, ids);
        this.update(updateWrapper);
    }

    /**
     * 通过方案编码删除
     * @param planCodes 方案编码集合
     */
    public void deleteByPlanCodes(List<String> planCodes) {
        if (CollectionUtils.isEmpty(planCodes)){
            return;
        }
        LambdaUpdateWrapper<ActivityPlanItem> updateWrapper = new UpdateWrapper<ActivityPlanItem>().lambda();
        updateWrapper.set(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(ActivityPlanItem::getPlanCode, planCodes);
        this.update(updateWrapper);

        //扩展表直接删掉吧。。。
        LambdaUpdateWrapper<ActivityPlanItemExtend> updateWrapper2 = new UpdateWrapper<ActivityPlanItemExtend>().lambda();
        updateWrapper2.in(ActivityPlanItemExtend::getPlanCode, planCodes);
        activityPlanItemExtendRepository.remove(updateWrapper2);
    }

    /**
     * 根据明细编码批量查询方案明细
     *
     * @author huojia
     * @date 2022/11/27 19:15
     * @param itemCodeList
     * @return java.util.List<com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItem>
     **/
    public List<ActivityPlanItem> listByDetailCodeList(List<String> itemCodeList) {
        if (CollectionUtils.isEmpty(itemCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(ActivityPlanItem::getPlanItemCode, itemCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 分页查询
     *
     * @author huojia
     * @date 2022/12/9 16:53
     * @param pageable
     * @param activityPlanItemDto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanItemVo>
     **/
    public Page<ActivityPlanItemVo> findByConditions(Pageable pageable, ActivityPlanItemDto activityPlanItemDto) {
        Page<ActivityPlanItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        if (pageable.getPageSize() == 999999){
            page.setSize(-1);//根据编码查，不分页
        }
        Page<ActivityPlanItemVo> pageList = this.activityPlanItemMapper.findByConditions(page, activityPlanItemDto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }

    public Page<ActivityPlanItemBudgetVo> findItemBudgetByConditions(Pageable pageable, ActivityPlanItemBudgetDto dto) {
        Page<ActivityPlanItemBudgetVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        dto.setFilterClose(true);
        Page<ActivityPlanItemBudgetVo> pageList = this.activityPlanItemMapper.findItemBudgetByConditions(page, dto);
        if (!CollectionUtils.isEmpty(pageList.getRecords())){
            for (ActivityPlanItemBudgetVo itemBudgetVo : pageList.getRecords()) {
                itemBudgetVo.setUsableAmount(Optional.ofNullable(itemBudgetVo.getTotalFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getUsedAmount()).orElse(BigDecimal.ZERO)));
                itemBudgetVo.setHeadUsableAmount(Optional.ofNullable(itemBudgetVo.getHeadFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getHeadUsedAmount()).orElse(BigDecimal.ZERO)));
                itemBudgetVo.setDepartmentUsableAmount(Optional.ofNullable(itemBudgetVo.getDepartmentFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getDepartmentUsedAmount()).orElse(BigDecimal.ZERO)));
                itemBudgetVo.setCustomerUsableAmount(Optional.ofNullable(itemBudgetVo.getCustomerFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getCustomerUsedAmount()).orElse(BigDecimal.ZERO)));
            }
        }
        return pageList;
    }

    public List<ActivityPlanItemBudgetVo> findItemBudgetListByConditions(ActivityPlanItemBudgetDto dto) {
        List<ActivityPlanItemBudgetVo> pageList = this.activityPlanItemMapper.findItemBudgetListByConditions( dto);
        dto.setFilterClose(false);
        if (!CollectionUtils.isEmpty(pageList)){
            //总部方案，查下关联的大区方案占用金额
            List<String> headPlanItemCodeList = pageList.stream().filter(item -> ActivityPlanTypeEnum.headquarters.getCode().equals(item.getPlanType())).map(ActivityPlanItemBudgetVo::getPlanItemCode).collect(Collectors.toList());
            if(!CollectionUtils.isEmpty(headPlanItemCodeList)){
                List<ActivityPlanItemBudgetVo> headBudgetList = activityPlanItemMapper.findItemHeadBudgetListByConditions(dto);
                Map<String, BigDecimal> headDepartmentUsedAmountMap = headBudgetList.stream()
                        .filter(item -> null != item.getDepartmentUsedAmount())
                        .collect(Collectors.toMap(ActivityPlanItemBudgetVo::getPlanItemCode, ActivityPlanItemBudgetVo::getDepartmentUsedAmount, (o, n) -> n));
                for (ActivityPlanItemBudgetVo planItemBudgetVo : pageList) {
                    if (!headDepartmentUsedAmountMap.containsKey(planItemBudgetVo.getPlanItemCode())){
                        continue;
                    }
                    planItemBudgetVo.setDepartmentUsedAmount(headDepartmentUsedAmountMap.get(planItemBudgetVo.getPlanItemCode()));
                }
            }

            for (ActivityPlanItemBudgetVo itemBudgetVo : pageList) {
                itemBudgetVo.setUsableAmount(Optional.ofNullable(itemBudgetVo.getTotalFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getUsedAmount()).orElse(BigDecimal.ZERO)));
                itemBudgetVo.setHeadUsableAmount(Optional.ofNullable(itemBudgetVo.getHeadFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getHeadUsedAmount()).orElse(BigDecimal.ZERO)));
                itemBudgetVo.setDepartmentUsableAmount(Optional.ofNullable(itemBudgetVo.getDepartmentFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getDepartmentUsedAmount()).orElse(BigDecimal.ZERO)));
                itemBudgetVo.setCustomerUsableAmount(Optional.ofNullable(itemBudgetVo.getCustomerFeeAmount()).orElse(BigDecimal.ZERO).subtract(Optional.ofNullable(itemBudgetVo.getCustomerUsedAmount()).orElse(BigDecimal.ZERO)));
            }
        }
        return pageList;
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoListProperties(List<ActivityPlanItemVo> voList) {
        if (CollectionUtils.isEmpty(voList)){
            return;
        }
        for (ActivityPlanItemVo item : voList) {
            this.fillVoProperties(item);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(ActivityPlanItemVo item) {
        if (null == item){
            return;
        }
        ObjectConvertStringUtil.fillObjectStrProperties(item,ActivityPlanItemVo.class);
        BigDecimal feeAmount = Optional.ofNullable(item.getTotalFeeAmount()).orElse(BigDecimal.ZERO);
        BigDecimal usedAmount = Optional.ofNullable(item.getUsedAmount()).orElse(BigDecimal.ZERO);
        item.setUsedAmount(usedAmount);
        item.setUsableAmount(feeAmount.subtract(usedAmount));
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillDtoListProperties(List<ActivityPlanItemDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)){
            return;
        }
        for (ActivityPlanItemDto item : dtoList) {
            this.fillDtoProperties(item);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillDtoProperties(ActivityPlanItemDto item) {
        if (null == item){
            return;
        }
        ObjectConvertStringUtil.fillObjectStrProperties(item,ActivityPlanItemDto.class);
        item.setActivityType(item.getActivityTypeCode());
        item.setActivityForm(item.getActivityFormCode());
    }

    /**
     * 关闭活动方案明细
     * @param ids
     */
    public void closeItem(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        this.update(Wrappers.lambdaUpdate(ActivityPlanItem.class)
                .set(ActivityPlanItem::getIsClose, BooleanEnum.TRUE.getCapital())
                .in(ActivityPlanItem::getId,ids)
        );
    }
    /**
     * 关闭活动方案明细(垂直)
     * @param ids
     */
    public void closeItemVertical(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        this.update(Wrappers.lambdaUpdate(ActivityPlanItem.class)
                .set(ActivityPlanItem::getIsClose, BooleanEnum.TRUE.getCapital())
                .set(ActivityPlanItem::getIsRollbackBudget, BooleanEnum.FALSE.getCapital())
                .in(ActivityPlanItem::getId,ids)
        );
    }

    public List<RedPacketQueryScheme2Vo> redPacketQueryScheme(RedPacketQueryScheme2Dto dto) {

        List<RedPacketQueryScheme2Vo> redPacketQueryScheme2Vos = this.activityPlanItemMapper.redPacketQueryScheme(dto);
        return redPacketQueryScheme2Vos;
    }

    public Page<ActivityPlanItemVo> findPlanItemNoActivityDetail(Page<ActivityPlanItemVo> page, String beginDate, String endDate, String salesOrgCode, List<String> legalSalesOrgCodeList) {
        return this.activityPlanItemMapper.findPlanItemNoActivityDetail(page, beginDate, endDate, legalSalesOrgCodeList);
    }

    /**
     * 关键指标-申请金额 根据方案明细编码获取活动方案获取费用合计
     * @param dto
     * @return
     */
    public ActivityPlanItemBase applyAmountFindByItemCode(ActivityPlanItemDto dto){
        ActivityPlanItem planItem = this.lambdaQuery()
                .eq(StringUtils.isNotBlank(dto.getPlanItemCode()),ActivityPlanItem::getPlanItemCode,dto.getPlanItemCode())
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).one();
        if(Objects.isNull(planItem)){
            return new ActivityPlanItemVo();
        }
        ActivityPlanItemBase planItemBase = this.nebulaToolkitService.copyObjectByWhiteList(planItem, ActivityPlanItemVo.class, LinkedHashSet.class, ArrayList.class);
        return planItemBase;
    }







    @Override
    public List<ActivityPlanItem> list(Wrapper<ActivityPlanItem> queryWrapper) {
        List<ActivityPlanItem> list = super.list(queryWrapper);
        if (!CollectionUtils.isEmpty(list)){
            List<String> idList = list.stream().map(ActivityPlanItem::getId).collect(Collectors.toList());
            List<ActivityPlanItemExtend> extendEntityList = activityPlanItemExtendRepository.listByIds(idList);
            if (!CollectionUtils.isEmpty(extendEntityList)){
                Map<String, ActivityPlanItemExtend> extendMap = extendEntityList.stream().collect(Collectors.toMap(ActivityPlanItemExtend::getId, Function.identity()));
                String[] ignoreFieldArr = ActivityPlanItemExtendRepository.excludeFieldList.toArray(new String[]{});
                for (ActivityPlanItem activityPlanItem : list) {
                    ActivityPlanItemExtend activityPlanItemExtend = extendMap.get(activityPlanItem.getId());
                    if (null != activityPlanItemExtend){
                        BeanUtil.copyProperties(activityPlanItemExtend,activityPlanItem,ignoreFieldArr);
                    }
                }
            }
            List<String> planItemCodeList = list.stream().map(ActivityPlanItem::getPlanItemCode).collect(Collectors.toList());
            List<ActivityPlanItemProduct> productList = activityPlanItemProductRepository.findListByPlanItemCodeList(planItemCodeList);
            if (!CollectionUtils.isEmpty(productList)){
                Map<String, List<ActivityPlanItemProduct>> productMap = productList.stream().collect(Collectors.groupingBy(ActivityPlanItemProduct::getPlanItemCode));
                for (ActivityPlanItem activityPlanItem : list) {
                    if(!productMap.containsKey(activityPlanItem.getPlanItemCode())){
                        continue;
                    }
                    List<ActivityPlanItemProduct> itemProductList = productMap.get(activityPlanItem.getPlanItemCode());
                    String productCode = itemProductList.stream().map(ActivityPlanItemProduct::getProductCode).collect(Collectors.joining(","));
                    activityPlanItem.setProductCode(productCode);
                    String productName = itemProductList.stream().map(ActivityPlanItemProduct::getProductName).collect(Collectors.joining(","));
                    activityPlanItem.setProductName(productName);
                }
            }
        }
        return list;
    }

    //有副表，重写下几个增删改方法
    @Override
    public boolean save(ActivityPlanItem entity) {
        //先保存下产品
        saveProduct(entity);
        boolean save = super.save(entity);

        //保存下副表
        ActivityPlanItemExtend extendEntity = BeanUtil.copyProperties(entity,ActivityPlanItemExtend.class);
        activityPlanItemExtendRepository.save(extendEntity);
        return save;
    }

    //保存下产品表
    private boolean saveProduct(ActivityPlanItem item){
        return saveProductList(Lists.newArrayList(item));
    }

    //保存下产品表
    private boolean saveProductList(Collection<ActivityPlanItem> itemList){
        if (CollectionUtils.isEmpty(itemList)){
            return false;
        }
        List<String> deleteItemCodeList = itemList.stream().filter(item -> StringUtils.isNotBlank(item.getId())).map(ActivityPlanItem::getPlanItemCode).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(deleteItemCodeList)){
            activityPlanItemProductRepository.deleteByPlanItemCodeList(deleteItemCodeList);
        }
        List<ActivityPlanItemProduct> saveList = com.google.common.collect.Lists.newArrayList();
        for (ActivityPlanItem activityPlanItem : itemList) {
            if (StringUtil.isNotEmpty(activityPlanItem.getProductCode()) && activityPlanItem.getProductCode().contains(",")){
                //是多选的才保存
                String[] productCodeSplit = activityPlanItem.getProductCode().split(",");
                if (StringUtil.isEmpty(activityPlanItem.getProductName())){
                    throw new RuntimeException("产品名称有误！");
                }
                String[] productNameSplit = activityPlanItem.getProductName().split(",");
                if (productNameSplit.length != productCodeSplit.length){
                    throw new RuntimeException("产品名称与编码数量不匹配！");
                }
                for (int i = 0; i < productCodeSplit.length; i++) {
                    ActivityPlanItemProduct itemProduct = new ActivityPlanItemProduct();
                    itemProduct.setPlanCode(activityPlanItem.getPlanCode());
                    itemProduct.setPlanItemCode(activityPlanItem.getPlanItemCode());
                    itemProduct.setProductCode(productCodeSplit[i]);
                    itemProduct.setProductName(productNameSplit[i]);
                    saveList.add(itemProduct);
                }
                activityPlanItem.setProductCode(null);
                activityPlanItem.setProductName(null);
            }
        }
        if (!CollectionUtils.isEmpty(saveList)){
            activityPlanItemProductRepository.saveBatch(saveList);
        }
        return true;
    }

    @Override
    public boolean saveOrUpdate(ActivityPlanItem entity, Wrapper<ActivityPlanItem> updateWrapper) {
        //先保存产品
        saveProduct(entity);
        boolean result = super.saveOrUpdate(entity, updateWrapper);

//        activityPlanItemExtendRepository.removeById(entity.getId());
        //保存下副表
        ActivityPlanItemExtend extendEntity = BeanUtil.copyProperties(entity,ActivityPlanItemExtend.class);
        if (activityPlanItemExtendRepository.hasNotEmptyProperty(extendEntity)){
            activityPlanItemExtendRepository.saveOrUpdate(extendEntity);
        }

        return result;
    }

    @Override
    public boolean saveBatch(Collection<ActivityPlanItem> entityList, int batchSize) {
        saveProductList(entityList);
        List<List<ActivityPlanItem>> partition = Lists.partition(Lists.newArrayList(entityList), batchSize);
        for (List<ActivityPlanItem> listPartition : partition) {
            //真实插入数据
            activityPlanItemMapper.insertBatchSomeColumn(listPartition);
        }
        updateExtendBatchById(entityList,batchSize);
        return true;
    }

    @Override
    public boolean saveOrUpdateBatch(Collection<ActivityPlanItem> entityList, int batchSize) {
        List<ActivityPlanItem> saveList = Lists.newArrayList();
        List<ActivityPlanItem> updateList = Lists.newArrayList();
        for (ActivityPlanItem activityPlanItem : entityList) {
            if (StringUtils.isBlank(activityPlanItem.getId())){
                saveList.add(activityPlanItem);
            }else{
                updateList.add(activityPlanItem);
            }
        }
        if (!CollectionUtils.isEmpty(saveList)){
            this.saveBatch(entityList,batchSize);
        }
        if (!CollectionUtils.isEmpty(updateList)){
            this.updateBatchById(entityList,batchSize);
        }
        return true;
    }

    @Override
    public boolean updateBatchById(Collection<ActivityPlanItem> entityList, int batchSize) {
        saveProductList(entityList);
        boolean result = super.updateBatchById(entityList, batchSize);
        updateExtendBatchById(entityList,batchSize);
        return result;
    }

    protected void updateExtendBatchById(Collection<ActivityPlanItem> entityList, int batchSize){
        List<ActivityPlanItemExtend> extendEntityList = Lists.newArrayList();
        for (ActivityPlanItem entity : entityList) {
            ActivityPlanItemExtend extendEntity = BeanUtil.copyProperties(entity,ActivityPlanItemExtend.class);
            if (activityPlanItemExtendRepository.hasNotEmptyProperty(extendEntity)){
                extendEntityList.add(extendEntity);
            }
        }
        if (!CollectionUtils.isEmpty(extendEntityList)){
            //保存下副表
            activityPlanItemExtendRepository.saveOrUpdateBatch(extendEntityList,batchSize);
        }

    }

    public Integer findTotalByConditions(ActivityPlanItemDto activityPlanItemDto) {
        return  this.activityPlanItemMapper.findTotalByConditions(activityPlanItemDto);
    }

    /**
     * 更新表头关闭状态
     */
    public void updateHeadStatus(List<String> itemCodeList) {
        activityPlanItemMapper.updateHeadStatus(itemCodeList);
    }

    public List<ActivityPlanItemVo> findRelatedPlanItemByCodes(List<String> relatePlanItemCodes) {
        if (CollectionUtils.isEmpty(relatePlanItemCodes)) {
            return Lists.newArrayList();
        }
        List<ActivityPlanItemVo> list = this.baseMapper.findRelatedPlanItemByCodes(relatePlanItemCodes);
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        return list;
    }

    /**
     * 根据关联方案明细编码 查询未关闭且审批通过的活动方案明细编码
     * @param relatePlanItemCodeList 关联方案编码
     * @return
     */
    public List<String> findApprovedItemCodeListByRelatePlanItemCodeList(List<String> relatePlanItemCodeList) {
        return activityPlanItemMapper.findApprovedItemCodeListByRelatePlanItemCodeList(relatePlanItemCodeList);
    }

    public List<ActivityPlanItem> findByIdList(List<String> ids) {
        return this.lambdaQuery()
                .eq(ActivityPlanItem::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(ActivityPlanItem::getId, ids)
                .list();
    }

    public List<ActivityPlanItem> findByHeadquartersPlanItemCode(String planItemCode) {
        if(StringUtils.isBlank(planItemCode)){
            return Lists.newArrayList();
        }
        return this.getBaseMapper().findByHeadquartersPlanItemCode(planItemCode);
    }

    public List<ActivityPlanSubmitReportSubVo> submitPageReportDimension1(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)) {
            return Collections.emptyList();
        }
        return activityPlanItemMapper.submitPageReportDimension1(planCodeList, DATE_FORMAT_STR);
    }

    public List<ActivityPlanSubmitReportSubVo> submitPageReportDimension2(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)) {
            return Collections.emptyList();
        }
        return activityPlanItemMapper.submitPageReportDimension2(planCodeList, DATE_FORMAT_STR);
    }

    public List<ActivityPlanSubmitReportSubVo> submitPageReportDimension3(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)) {
            return Collections.emptyList();
        }
        return activityPlanItemMapper.submitPageReportDimension3(planCodeList, DATE_FORMAT_STR);
    }

    public void updateAuditStatusByCodes(List<String> auditPlanItemCodes) {
        if(!CollectionUtils.isEmpty(auditPlanItemCodes)){
            this.lambdaUpdate().in(ActivityPlanItem::getPlanItemCode,auditPlanItemCodes)
                    .set(ActivityPlanItem::getWholeAudit, YesOrNoEnum.YES.getCode()).update();
            log.info("活动方案预算跟踪表退预算更改方案明细状态");

        }
    }

    public ActivityPlanItem findListByPlanItemCode(String planItemCode) {
        if(StringUtils.isBlank(planItemCode)){
            return null;
        }
        return this.lambdaQuery().eq(ActivityPlanItem::getPlanItemCode,planItemCode).one();
    }


    public List<String> findItemCodeListByRelatePlanItemCodeList(List<String> planItemCodeList) {
        if(CollectionUtils.isEmpty(planItemCodeList)){
            return null;
        }
        return this.lambdaQuery()
                .in(ActivityPlanItem::getRelatePlanItemCode,planItemCodeList)
                .select(ActivityPlanItem::getPlanItemCode)
                .list().stream().map(ActivityPlanItem::getPlanItemCode).collect(Collectors.toList());
    }

    public void updateWholeAuditByPlanItemCode(String businessCode) {
        if(StringUtils.isBlank(businessCode)){
            return;
        }
        this.lambdaUpdate().eq(ActivityPlanItem::getPlanItemCode,businessCode)
                .set(ActivityPlanItem::getWholeAudit,YesOrNoEnum.NO.getCode()).update();
    }
    public List<String> findActivityPlanItemCodeByActivityNumber(List<String> activityNumberList,String planCode) {
        return activityPlanItemMapper.findActivityPlanItemCodeByActivityNumber(activityNumberList,planCode);
    }

    public Collection<String> findPlanCodeListByItemCodeList(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)){
            return Lists.newArrayList();
        }
        return this.list(Wrappers.lambdaQuery(ActivityPlanItem.class)
                .in(ActivityPlanItem::getPlanItemCode, planItemCodeList)
                .eq(ActivityPlanItem::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .select(ActivityPlanItem::getPlanCode)
        ).stream().map(ActivityPlanItem::getPlanCode).distinct().collect(Collectors.toList());
    }

    public List<ActivityPlanItemVo> findCountForCarGift(List<String> planCodeList) {
        if(CollectionUtils.isEmpty(planCodeList)){
            return Lists.newArrayList();
        }
        return this.getBaseMapper().findCountForCarGift(planCodeList);
    }

    public void updatePlanItemRollbackBudget(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)){
            return;
        }
        this.lambdaUpdate()
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityPlanItem::getTenantCode, TenantUtils.getTenantCode())
                .in(ActivityPlanItem::getPlanItemCode, planItemCodeList)
                .set(ActivityPlanItem::getIsRollbackBudget, BooleanEnum.TRUE.getCapital())
                .update();
    }

    public List<ActivityPlanItem> findListByPlanCodes(List<String> planCodes) {
        if(CollectionUtils.isEmpty(planCodes)){
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(ActivityPlanItem::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .eq(ActivityPlanItem::getTenantCode,TenantUtils.getTenantCode())
                .in(ActivityPlanItem::getPlanCode,planCodes)
                .list();
    }

    public void updatePlanItemSapState(String planItemCode, boolean success) {
        if(StringUtils.isBlank(planItemCode)){
            return;
        }
        LambdaUpdateWrapper<ActivityPlanItemExtend> updateWrapper = new UpdateWrapper<ActivityPlanItemExtend>().lambda();
        updateWrapper.set(ActivityPlanItemExtend::getSapInterfaceState, success ? InterfacePushStateEnum.SUCCESS.getCode() : InterfacePushStateEnum.FAIL.getCode());
        updateWrapper.eq(ActivityPlanItemExtend::getTenantCode, TenantUtils.getTenantCode());
        updateWrapper.eq(ActivityPlanItemExtend::getPlanItemCode, planItemCode);
        activityPlanItemExtendRepository.update(updateWrapper);
    }

    public List<String> listIdByPlanCodes(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)){
            return Lists.newArrayList();
        }
        return this.list(Wrappers.lambdaQuery(ActivityPlanItem.class)
                .in(ActivityPlanItem::getPlanCode, planCodeList)
                .eq(ActivityPlanItem::getTenantCode, TenantUtils.getTenantCode())
                .eq(ActivityPlanItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .select(ActivityPlanItem::getId)
        ).stream().map(ActivityPlanItem::getId).collect(Collectors.toList());
    }

    public List<ActivityPlanItemVo> findListForFR(List<ActivityPlanItemDto> activityPlanItemDtos) {
        return this.activityPlanItemMapper.findListForFR(activityPlanItemDtos,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanItemVo> findCostListForFR(List<ActivityPlanItemDto> itemDtos, String code) {
        return this.activityPlanItemMapper.findCostListForFR(itemDtos,code,TenantUtils.getTenantCode());
    }

    @Transactional(rollbackFor = {Exception.class})
    public boolean alwaysUpdateSomeColumnBatchById(Collection<ActivityPlanItem> entityList, int batchSize) {
        saveProductList(entityList);
        String sqlStatement = ActivityPlanItemMapper.class.getName() + ".alwaysUpdateSomeColumnById";
        boolean result = this.executeBatch(entityList, batchSize, (sqlSession, entity) -> {
            MapperMethod.ParamMap<ActivityPlanItem> param = new MapperMethod.ParamMap();
            param.put("et", entity);
            sqlSession.update(sqlStatement, param);
        });
        if(!result){
            return result;
        }
        updateExtendBatchById(entityList,batchSize);
        return result;
    }

    @Transactional(rollbackFor = {Exception.class})
    public int alwaysUpdateSomeColumnById(ActivityPlanItem entity) {
        saveProduct(entity);
        int i = activityPlanItemMapper.alwaysUpdateSomeColumnById(entity);
        updateExtendBatchById(Lists.newArrayList(entity),1000);
        return i;
    }

}

