package com.biz.crm.tpm.business.activity.plan.local.repository;


import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.plan.local.vo.ActivityPlanProcessBusinessForm;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanDto;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlan;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanOrg;
import com.biz.crm.tpm.business.activity.plan.local.mapper.ActivityPlanMapper;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.sfa.ActivityPlanItemPushSfaDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.sfa.display.ActivityPlanItemPushSfaDisplayDto;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 活动方案表(ActivityPlan)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-10-31 15:55:54
 */
@Component
public class ActivityPlanRepository extends ServiceImpl<ActivityPlanMapper, ActivityPlan> {

    @Autowired(required = false)
    private ActivityPlanMapper activityPlanMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ActivityPlanOrgRepository activityPlanOrgRepository;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto 查询实体
     * @return 所有数据
     */
    public Page<ActivityPlanVo> findByConditions(Pageable pageable, ActivityPlanDto dto) {
        Page<ActivityPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<ActivityPlanVo> pageList = this.activityPlanMapper.findByConditions(page, dto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }

    private void setActivityUsableBudget(List<ActivityPlanVo> list){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        List<String> planCodeList = list.stream().map(ActivityPlanVo::getPlanCode).distinct().collect(Collectors.toList());
        List<ActivityPlanVo> budgetList = activityPlanMapper.findActivityPlanUsableBudgetList(planCodeList);
        if (CollectionUtils.isEmpty(budgetList)){
            return;
        }
        Map<String, ActivityPlanVo> budgetMap = budgetList.stream().collect(Collectors.toMap(ActivityPlanVo::getPlanCode, Function.identity(),(o,n) -> n));
        for (ActivityPlanVo planVo : list) {
            ActivityPlanVo budget = budgetMap.get(planVo.getPlanCode());
            if (null == budget){
                continue;
            }
            planVo.setUsableAmount(budget.getUsableAmount());
            planVo.setUsedAmount(budget.getUsedAmount());
        }
    }
    @Override
    public boolean save(ActivityPlan entity) {
        boolean save = super.save(entity);
        saveActivityPlanOrg(entity);
        return save;
    }

    @Override
    public boolean updateById(ActivityPlan entity) {
        boolean b = super.updateById(entity);
        activityPlanOrgRepository.deleteByPlanId(entity.getId());
        saveActivityPlanOrg(entity);
        return b;
    }

    private void saveActivityPlanOrg(ActivityPlan entity){
        if (StringUtils.isBlank(entity.getDepartmentCode())){
            return;
        }
        String[] departmentCodeArr = entity.getDepartmentCode().split(",");
        int length = departmentCodeArr.length;
        String[] departmentNameArr = entity.getDepartmentName().split(",");
        List<ActivityPlanOrg> activityPlanOrgList = Lists.newArrayList();
        for (int i = 0; i < length; i++) {
            ActivityPlanOrg activityPlanOrg = new ActivityPlanOrg();
            activityPlanOrg.setPlanCode(entity.getPlanCode());
            activityPlanOrg.setPlanId(entity.getId());
            activityPlanOrg.setOrgCode(departmentCodeArr[i]);
            activityPlanOrg.setOrgName(departmentNameArr[i]);
            activityPlanOrg.setTenantCode(entity.getTenantCode());
            activityPlanOrgList.add(activityPlanOrg);
        }
        activityPlanOrgRepository.saveBatch(activityPlanOrgList);

    }

    /**
     * 通过id删除
     * @param ids id集合
     */
    public void deleteByIds(List<String> ids){
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        LambdaUpdateWrapper<ActivityPlan> updateWrapper = new UpdateWrapper<ActivityPlan>().lambda();
        updateWrapper.set(ActivityPlan::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(ActivityPlan::getId, ids);
        this.update(updateWrapper);
    }

    public void updateProcessStatus(List<String> codeList, String processStatus) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        activityPlanMapper.updateProcessStatus(codeList,processStatus,TenantUtils.getTenantCode());
    }

    public void updateProcessStatusAndProcessNo(List<String> codeList, String processStatus, String processNo) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        this.update(Wrappers.lambdaUpdate(ActivityPlan.class)
                .set(ActivityPlan::getProcessStatus, processStatus)
                .set(ActivityPlan::getProcessNo, processNo)
                .in(ActivityPlan::getPlanCode, codeList)
                .eq(ActivityPlan::getTenantCode, TenantUtils.getTenantCode())
        );
    }

    public ActivityPlanVo getVoById(Serializable id) {
        ActivityPlan entity = getById(id);
        if (null == entity || !StringUtils.equals(TenantUtils.getTenantCode(), entity.getTenantCode())){
            return null;
        }
        ActivityPlanVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, ActivityPlanVo.class, HashSet.class, ArrayList.class);
        fillVoProperties(vo);
        return vo;
    }

    public ActivityPlanVo getVoByPlanCode(String planCode) {
        ActivityPlan entity = getByPlanCode(planCode);
        if (null == entity){
            return null;
        }
        ActivityPlanVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, ActivityPlanVo.class, HashSet.class, ArrayList.class);
        fillVoProperties(vo);
        return vo;
    }

    public ActivityPlan getByPlanCode(String planCode) {
        if (StringUtils.isBlank(planCode)){
            return null;
        }
        return this.getOne(Wrappers.lambdaQuery(ActivityPlan.class)
                .eq(ActivityPlan::getPlanCode, planCode)
                .eq(ActivityPlan::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()));
    }



    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoListProperties(List<ActivityPlanVo> voList) {
        if (CollectionUtils.isEmpty(voList)){
            return;
        }
        setActivityUsableBudget(voList);
        SimpleDateFormat dayFormat = new SimpleDateFormat(DateUtil.DEFAULT_YEAR_MONTH_DAY);
        for (ActivityPlanVo item : voList) {
            this.fillVoProperties(item,dayFormat);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(ActivityPlanVo item) {
        fillVoProperties(item,null);
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(ActivityPlanVo item,SimpleDateFormat dayFormat) {
        if (null == item){
            return;
        }
        if (null == dayFormat){
            dayFormat = new SimpleDateFormat(DateUtil.DEFAULT_YEAR_MONTH_DAY);
        }
        if (null != item.getBeginDate()){
            item.setBeginDateStr(dayFormat.format(item.getBeginDate()));
        }
        if (null != item.getEndDate()){
            item.setEndDateStr(dayFormat.format(item.getEndDate()));
        }
//
//        BigDecimal feeAmount = Optional.ofNullable(item.getTotalFeeAmount()).orElse(BigDecimal.ZERO);
//        BigDecimal usedAmount = Optional.ofNullable(item.getUsedAmount()).orElse(BigDecimal.ZERO);
//        item.setUsedAmount(usedAmount);
//        item.setUsableAmount(feeAmount.subtract(usedAmount));
    }


    /**
     * 通过方案编码查询推送sfa数据
     * @param planCodes 方案编码
     * @return
     */
    public List<ActivityPlanItemPushSfaDto> findSfaDataByPlanCodes(List<String> planCodes) {
        return activityPlanMapper.findSfaDataByPlanCodes(planCodes);
    }

    public List<ActivityPlanItemPushSfaDisplayDto> findSfaDisplayDataByPlanCodes(List<String> planCodes) {
        return activityPlanMapper.findSfaDisplayDataByPlanCodes(planCodes);
    }

    public List<ActivityPlanVo> findPlanActivityEndTime(Set<String> activityDetailCodes) {

        return activityPlanMapper.findPlanActivityEndTime(activityDetailCodes);
    }

    public ActivityPlanProcessBusinessForm getActivityPlanProcessBusinessForm(List<String> planCodeList) {
        return activityPlanMapper.getActivityPlanProcessBusinessForm(planCodeList);
    }

    /**
     * 根据方案编码查询列表信息
     * @param planCodeList
     * @return
     */
    public List<ActivityPlanDto> findDtoListByPlanCodeList(List<String> planCodeList) {
        List<ActivityPlan> byPlanCodeList = findByPlanCodeList(planCodeList);
        if(CollectionUtils.isEmpty(byPlanCodeList)){
            return Lists.newArrayList();
        }
        return (List<ActivityPlanDto>) nebulaToolkitService.copyCollectionByWhiteList(byPlanCodeList,ActivityPlan.class, ActivityPlanDto.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据方案编码查询列表信息
     * @param planCodeList
     * @return
     */
    public List<ActivityPlan> findByPlanCodeList(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)) {
            return Collections.emptyList();
        }
        List<ActivityPlan> list = this.lambdaQuery()
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(TenantFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(ActivityPlan::getPlanCode, planCodeList)
                .list();
        List<ActivityPlanOrg> orgList = activityPlanOrgRepository.findByPlanCodeList(planCodeList);
        Map<String, List<ActivityPlanOrg>> planOrgMap = orgList.stream().collect(Collectors.groupingBy(ActivityPlanOrg::getPlanCode));
        for (ActivityPlan plan : list) {
            List<ActivityPlanOrg> activityPlanOrgList = planOrgMap.get(plan.getPlanCode());
            if (CollectionUtils.isEmpty(activityPlanOrgList)){
                continue;
            }
            String orgCode = activityPlanOrgList.stream().map(ActivityPlanOrg::getOrgCode).collect(Collectors.joining(","));
            String orgName = activityPlanOrgList.stream().map(ActivityPlanOrg::getOrgName).collect(Collectors.joining(","));
            plan.setDepartmentCode(orgCode);
            plan.setDepartmentName(orgName);
        }
        return list;
    }

    /**
     * 根据流程编码查询列表信息
     * @param processNoList
     * @return
     */
    public List<ActivityPlan> findByProcessNoList(List<String> processNoList) {
        if (CollectionUtils.isEmpty(processNoList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .select(ActivityPlan::getPlanCode, ActivityPlan::getBusinessUnitCode)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(TenantFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(ActivityPlan::getProcessNo, processNoList)
                .list();
    }

    public List<ActivityPlanVo> findByCodes(Set<String> planCodes) {
        List<ActivityPlanVo> byCodes = activityPlanMapper.findByCodes(planCodes);
        setActivityUsableBudget(byCodes);
        return byCodes;
    }


    public List<ActivityPlanRedLineVo> getActivityPlanBreakPrice(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)) {
            return Lists.newArrayList();
        } else {
            return activityPlanMapper.getActivityPlanBreakPrice(planCodeList);
        }
    }

    /**
     * 活动方案申请可以选择的总部方案
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据ø
     */
    public Page<ActivityPlanVo> findRelateActivityPlanListByConditions(Pageable pageable, ActivityPlanDto dto) {
        Page<ActivityPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<ActivityPlanVo> pageList = this.activityPlanMapper.findRelateActivityPlanListByConditions(page, dto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }

    public ActivityPlan findByPlanItemCode(String planItemCode) {
        if(StringUtils.isBlank(planItemCode)){
            return null;
        }
        return this.getBaseMapper().findByPlanItemCode(planItemCode);
    }


    public List<ActivityPlanVo> findByPlanItemCodes(List<String> planItemCodes) {
        if(CollectionUtil.isEmpty(planItemCodes)){
            return Collections.emptyList();
        }
        return this.getBaseMapper().findByPlanItemCodes(planItemCodes);
    }

    /**
     * 大区方案编码查询关联总部方案编码和总部策略编码
     * @param planCodes
     * @return
     */
    public List<ActivityPlanBudgetVo> findHeadquartersByRegionPlanCodes(Set<String> planCodes) {

        if(CollectionUtils.isEmpty(planCodes)){
            return Lists.newArrayList();
        }
        return this.getBaseMapper().findHeadquartersByRegionPlanCodes(planCodes);
    }


    /**
     * 根据大区方案编码查询总部方案信息
     *
     * @param relatePlanCodes 大区方案编码
     * @return
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-11-12 23:27
     */
    public List<ActivityPlanItemVo> findActivityPlanByRelatePlanCodes(List<String> relatePlanCodes) {
        if (CollectionUtils.isEmpty(relatePlanCodes)) {
            return Lists.newArrayList();
        }
        return activityPlanMapper.findActivityPlanByRelatePlanCodes(TenantUtils.getTenantCode(), relatePlanCodes);
    }

    /**
     * 根据总部方案编码查询大区方案信息
     *
     * @param relatePlanItemCodes 总部方案编码
     * @return
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-11-12 23:27
     */
    public List<ActivityPlanVo> findActivityPlanByRelatePlanItemCodes(List<String> relatePlanItemCodes) {
        if (CollectionUtils.isEmpty(relatePlanItemCodes)) {
            return Lists.newArrayList();
        }
        return activityPlanMapper.findActivityPlanByRelatePlanItemCodes(TenantUtils.getTenantCode(), relatePlanItemCodes);
    }

    public List<ActivityPlanVo> findActivityUsableBudget(List<String> planCodeList) {
        if (CollectionUtils.isEmpty(planCodeList)){
            return Lists.newArrayList();
        }
        return activityPlanMapper.findActivityPlanUsableBudgetList(planCodeList);
    }

    public List<ActivityPlanVo>  findOrgCodeByPlanCode(List<String> planCodes) {
        return this.baseMapper.findOrgCodeByPlanCode(planCodes,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanVerticalSchemeForecastVo> findVerticalSchemeForecastByPlanItemCodes(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)){
            return Lists.newArrayList();
        }
        return this.baseMapper.findVerticalSchemeForecastByPlanItemCodes(planItemCodeList,TenantUtils.getTenantCode());
    }

    public List<ActivityPlanAuditVo> findAuditInfoByPlanItemCodes(List<String> planItemCodeList) {
        if (CollectionUtils.isEmpty(planItemCodeList)){
            return Lists.newArrayList();
        }
        return this.baseMapper.findAuditInfoByPlanItemCodes(planItemCodeList,TenantUtils.getTenantCode());
    }
}

