package com.biz.crm.tpm.business.activity.plan.local.service.internal;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanBudget;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItem;
import com.biz.crm.tpm.business.activity.plan.local.repository.ActivityPlanBudgetRepository;
import com.biz.crm.tpm.business.activity.plan.local.repository.ActivityPlanItemRepository;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.OperateActivityPlanBudgetVerticalDto;
import com.biz.crm.tpm.business.activity.plan.sdk.service.ActivityPlanBudgetVerticalService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.OperateMarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategyBudgetSdkService;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto;
import com.biz.crm.tpm.business.month.budget.sdk.dto.OperateMonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/3/14 10:36
 * @description: 方案关联预算数据(垂直)实现
 */
@Service
public class ActivityPlanBudgetVerticalServiceImpl implements ActivityPlanBudgetVerticalService {

    @Autowired(required = false)
    private ActivityPlanBudgetRepository activityPlanBudgetRepository;
    @Autowired(required = false)
    private ActivityPlanBudgetServiceImpl activityPlanBudgetService;
    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;
    @Autowired(required = false)
    private MarketingStrategyBudgetSdkService marketingStrategyBudgetSdkService;
    @Autowired(required = false)
    private ActivityPlanItemRepository activityPlanItemRepository;

    /**
     * 操作方案预算(垂直方案关闭后)
     *
     * @param operateList 操作dto
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void operateBudget(List<OperateActivityPlanBudgetVerticalDto> operateList) {
        //查预算
        List<ActivityPlanBudget> activityPlanBudgets = validateOperateBudget(operateList);
        //按照方案明细分组
        Map<String, List<ActivityPlanBudget>> budgetEntityMap = activityPlanBudgets.stream().collect(Collectors.groupingBy(ActivityPlanBudget::getPlanItemCode));

        List<MonthBudgetDetailDto> detailList = Lists.newArrayList();
        List<OperateMarketingStrategyBudgetDto> strategyOperateList = Lists.newArrayList();
        List<OperateMonthBudgetDto> budgetOperateList = Lists.newArrayList();
        Set<String> planItemSet = new HashSet<>();
        //循环预算操作
        for (OperateActivityPlanBudgetVerticalDto operateMonthBudgetDto : operateList) {
            BigDecimal operationAmount = operateMonthBudgetDto.getOperationAmount();
            if (BigDecimal.ZERO.compareTo(operationAmount) >= 0) {
                break;
            }
            List<ActivityPlanBudget> monthBudgetEntities = budgetEntityMap.get(operateMonthBudgetDto.getPlanItemCode());
            if (CollectionUtils.isEmpty(monthBudgetEntities)){
                break;
            }
            //循环单个方案的预算
            for (int i = 0; i < monthBudgetEntities.size(); i++) {
                ActivityPlanBudget monthBudgetEntity = monthBudgetEntities.get(i);
                //可用金额
                BigDecimal usableAmount = monthBudgetEntity.getUseAmount().subtract(monthBudgetEntity.getReturnAmount());
                BigDecimal thisOperationAmount = operationAmount;
                if (BigDecimal.ZERO.compareTo(operationAmount) >= 0) {
                    break;
                }
                //本次操作大于可用金额
                if (thisOperationAmount.compareTo(usableAmount) > 0) {
                    thisOperationAmount = usableAmount;
                }
                operationAmount = operationAmount.subtract(thisOperationAmount);
                monthBudgetEntity.setReturnAmount(monthBudgetEntity.getReturnAmount().subtract(thisOperationAmount));

                BigDecimal finalThisOperationAmount = thisOperationAmount;
                //这里组装月度预算的参数
                budgetOperateList.add(new OperateMonthBudgetDto() {{
                    this.setBusinessCode(monthBudgetEntity.getPlanItemCode());
                    this.setMonthBudgetCode(monthBudgetEntity.getMonthBudgetCode());
                    this.setOperationType(BudgetOperationTypeEnum.RETURN.getCode());
                    this.setOperationAmount(finalThisOperationAmount);
                }});
                //这里组装策略的参数
                if (StringUtils.isNotBlank(monthBudgetEntity.getRelateStrategyCode())) {

                    strategyOperateList.add(new OperateMarketingStrategyBudgetDto() {{
                        this.setBusinessCode(monthBudgetEntity.getPlanItemCode());
                        this.setStrategyCode(monthBudgetEntity.getRelateStrategyCode());
                        this.setStrategyItemCode(monthBudgetEntity.getRelateStrategyItemCode());
                        this.setMonthBudgetCode(monthBudgetEntity.getMonthBudgetCode());
                        this.setOperationType(BudgetOperationTypeEnum.RETURN.getCode());
                        this.setOperationAmount(finalThisOperationAmount);
                    }});
                }
            }
            //这里组装方案明细得参数
            if (operateMonthBudgetDto.getIsRollbackBudget()) {
                planItemSet.add(operateMonthBudgetDto.getPlanItemCode());
            }
        }
        activityPlanBudgetService.updateOperateBudget(activityPlanBudgets, detailList);
        //1、退回策略预算
        if (!CollectionUtils.isEmpty(strategyOperateList)) {
            marketingStrategyBudgetSdkService.operateAmount(strategyOperateList);
        }
        //2、退回月度预算
        if (!CollectionUtils.isEmpty(budgetOperateList)) {
            monthBudgetService.operateBudget(budgetOperateList);
        }
    }

    /**
     * 操作方案预算(垂直细案单独关闭后)
     *
     * @param operateList 操作dto
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void operateBudgetDetailPlan(List<OperateActivityPlanBudgetVerticalDto> operateList) {
        //查预算
        List<ActivityPlanBudget> activityPlanBudgets = validateOperateBudget(operateList);
        //按照方案明细分组
        Map<String, List<ActivityPlanBudget>> budgetEntityMap = activityPlanBudgets.stream().collect(Collectors.groupingBy(ActivityPlanBudget::getPlanItemCode));

        List<MonthBudgetDetailDto> detailList = Lists.newArrayList();
        List<OperateMarketingStrategyBudgetDto> strategyOperateList = Lists.newArrayList();
        List<OperateMonthBudgetDto> budgetOperateList = Lists.newArrayList();
        //循环预算操作
        for (OperateActivityPlanBudgetVerticalDto operateMonthBudgetDto : operateList) {

            BigDecimal operationAmount = operateMonthBudgetDto.getOperationAmount();
            List<ActivityPlanBudget> monthBudgetEntities = budgetEntityMap.get(operateMonthBudgetDto.getPlanItemCode());
            //循环单个方案的预算
            for (int i = 0; i < monthBudgetEntities.size(); i++) {
                ActivityPlanBudget monthBudgetEntity = monthBudgetEntities.get(i);
                //可用金额
                BigDecimal usableAmount = monthBudgetEntity.getUseAmount().subtract(monthBudgetEntity.getUsedAmount());
                BigDecimal thisOperationAmount = operationAmount;
                if (BigDecimal.ZERO.compareTo(operationAmount) >= 0) {
                    break;
                }
                //本次操作大于可用金额
                if (thisOperationAmount.compareTo(usableAmount) > 0) {
                    thisOperationAmount = usableAmount;
                    operationAmount = operationAmount.subtract(thisOperationAmount);
                }
                monthBudgetEntity.setUsedAmount(monthBudgetEntity.getUsedAmount().subtract(thisOperationAmount));

                BigDecimal finalThisOperationAmount = thisOperationAmount;
                //这里组装月度预算的参数
                budgetOperateList.add(new OperateMonthBudgetDto() {{
                    this.setBusinessCode(monthBudgetEntity.getPlanItemCode());
                    this.setMonthBudgetCode(monthBudgetEntity.getMonthBudgetCode());
                    this.setOperationType(BudgetOperationTypeEnum.RETURN.getCode());
                    this.setOperationAmount(finalThisOperationAmount);
                }});
                //这里组装策略的参数
                if (StringUtils.isNotBlank(monthBudgetEntity.getRelateStrategyCode())) {

                    strategyOperateList.add(new OperateMarketingStrategyBudgetDto() {{
                        this.setBusinessCode(monthBudgetEntity.getPlanItemCode());
                        this.setStrategyCode(monthBudgetEntity.getRelateStrategyCode());
                        this.setStrategyItemCode(monthBudgetEntity.getRelateStrategyItemCode());
                        this.setMonthBudgetCode(monthBudgetEntity.getMonthBudgetCode());
                        this.setOperationType(BudgetOperationTypeEnum.RETURN.getCode());
                        this.setOperationAmount(finalThisOperationAmount);
                    }});
                }
            }
        }
        activityPlanBudgetService.updateOperateBudget(activityPlanBudgets, detailList);
        //1、退回策略预算
        if (!CollectionUtils.isEmpty(strategyOperateList)) {
            marketingStrategyBudgetSdkService.operateAmount(strategyOperateList);
        }
        //2、退回月度预算
        if (!CollectionUtils.isEmpty(budgetOperateList)) {
            monthBudgetService.operateBudget(budgetOperateList);
        }
    }

    @Override
    public void updatePlanItemRollbackBudget(List<String> planItemCodeList) {
        activityPlanItemRepository.updatePlanItemRollbackBudget(planItemCodeList);
    }


    public List<ActivityPlanBudget> validateOperateBudget(List<OperateActivityPlanBudgetVerticalDto> operateList) {
        for (OperateActivityPlanBudgetVerticalDto operateMonthBudgetDto : operateList) {
//            Validate.notEmpty(operateMonthBudgetDto.getMonthBudgetCode(), "操作预算时，预算编码不能为空！");
            Validate.notNull(operateMonthBudgetDto.getOperationAmount(), "操作预算时，操作金额不能为空！");
            Validate.notEmpty(operateMonthBudgetDto.getOperationType(), "操作预算时，操作类型不能为空！");
            Validate.notEmpty(operateMonthBudgetDto.getPlanItemCode(), "操作方案预算时，方案明细编码不能为空！");
        }
        List<String> operateCodeList = operateList.stream().map(OperateActivityPlanBudgetVerticalDto::getPlanItemCode).distinct().collect(Collectors.toList());
        LinkedList<ActivityPlanBudget> budgetEntityList = activityPlanBudgetRepository.listByOperateCodeListVertical(operateCodeList);
//        if (budgetEntityList.size() < operateCodeList.size()) {
//            List<String> existsCodes = budgetEntityList.stream().map(ActivityPlanBudget::getPlanItemCode).collect(Collectors.toList());
//            String notExistsJoinCodesStr = operateCodeList.stream().filter(item -> !existsCodes.contains(item)).collect(Collectors.joining(","));
//            throw new RuntimeException("预算操作失败，活动明细[" + notExistsJoinCodesStr + "]查询失败，请检查预算是否启用或是否存在！！");
//        }
        for (ActivityPlanBudget activityPlanItem : budgetEntityList) {
            if (null == activityPlanItem.getUseAmount()) {
                activityPlanItem.setUseAmount(BigDecimal.ZERO);
            }
            if (null == activityPlanItem.getUsedAmount()) {
                activityPlanItem.setUsedAmount(BigDecimal.ZERO);
            }
            if (null == activityPlanItem.getReturnAmount()) {
                activityPlanItem.setReturnAmount(BigDecimal.ZERO);
            }
        }
        return budgetEntityList;
    }
}
