package com.biz.crm.tpm.business.activity.plan.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.invoice.sales.data.sdk.dto.SalesDataDto;
import com.biz.crm.kms.business.invoice.sales.data.sdk.service.InvoiceSalesDataVoService;
import com.biz.crm.kms.business.invoice.sales.data.sdk.vo.SalesDataVo;
import com.biz.crm.mdm.business.promoters.sdk.dto.PromotersDto;
import com.biz.crm.mdm.business.promoters.sdk.service.PromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalPaginationDto;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.util.ObjectConvertStringUtil;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlan;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItemTerminal;
import com.biz.crm.tpm.business.activity.plan.local.repository.ActivityPlanItemTerminalRepository;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanItemTerminalService;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanItemDto;
import com.biz.crm.tpm.business.activity.plan.sdk.dto.ActivityPlanItemTerminalDto;
import com.biz.crm.tpm.business.activity.plan.sdk.enums.WriteOffMethodEnum;
import com.biz.crm.tpm.business.activity.plan.sdk.vo.ActivityPlanItemTerminalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import liquibase.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author ：dengwei
 * @date ：Created in 2023/1/12 11:03
 * @description：方案明细门店
 */
@Slf4j
@Service
public class ActivityPlanItemTerminalServiceImpl implements ActivityPlanItemTerminalService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private ActivityPlanItemTerminalRepository activityPlanItemTerminalRepository;

    @Autowired(required = false)
    private ActivityPlanItemPageCacheHelper itemHelper;

    @Autowired(required = false)
    private TerminalVoService terminalVoService;

    @Autowired(required = false)
    private InvoiceSalesDataVoService invoiceSalesDataVoService;

    @Autowired(required = false)
    private PromotersVoService promotersVoService;

    /**
     * 保存更新门店明细
     *
     * @param entity
     * @param update
     * @param itemDtoList
     * @param cacheKey
     */
    @Override
    public void saveActivityPlanItemTerminal(ActivityPlan entity, boolean update, List<ActivityPlanItemDto> itemDtoList, String cacheKey) {
        if(update){
            activityPlanItemTerminalRepository.lambdaUpdate().eq(ActivityPlanItemTerminal::getPlanCode, entity.getPlanCode()).remove();
        }

        //可能要删除终端的方案明细code
        List<ActivityPlanItemTerminal> allList = new ArrayList<>();
        for (ActivityPlanItemDto itemDto : itemDtoList) {
            List<ActivityPlanItemTerminalDto> terminalList = itemDto.getActivityPlanItemTerminalList();
            if(!CollectionUtils.isEmpty(terminalList)){
                for (ActivityPlanItemTerminalDto dto : terminalList) {
                    dto.setPlanCode(entity.getPlanCode());
                    dto.setPlanItemCode(itemDto.getPlanItemCode());
                    dto.setBeginDate(itemDto.getActivityBeginDate());
                    dto.setEndDate(itemDto.getActivityEndDate());
                    dto.setTenantCode(TenantUtils.getTenantCode());
                }
                List<ActivityPlanItemTerminal> activityPlanItemTerminals = (List<ActivityPlanItemTerminal>)this.nebulaToolkitService.copyCollectionByBlankList(terminalList, ActivityPlanItemTerminalDto.class, ActivityPlanItemTerminal.class, LinkedHashSet.class, ArrayList.class);
                allList.addAll(activityPlanItemTerminals);
            }
        }
        if(!CollectionUtils.isEmpty(allList)){
            List<List<ActivityPlanItemTerminal>> partition = com.google.common.collect.Lists.partition(allList, 1000);
            int i = 0;
            int size = allList.size();
            for (List<ActivityPlanItemTerminal> activityPlanItemTerminals : partition) {
                itemHelper.sendMsg("正在更新第["+1000*(i++)+"/"+size+"]条方案门店明细...");
                this.activityPlanItemTerminalRepository.saveBatch(activityPlanItemTerminals);
            }
        }
    }


    /**
     * 营销策略明细新增校验逻辑
     *
     * @param dtoList 营销策略明细数据
     */
    @Override
    public void createValidateList(ActivityPlanDto planDto, ActivityPlanItemDto itemDto, List<ActivityPlanItemTerminalDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }
        ObjectConvertStringUtil.convertObjectListStrProperties(dtoList, ActivityPlanItemTerminalDto.class );
    }

    @Override
    public void addTerminalFromCache(ActivityPlanItemDto dto) {
        String cacheKey = dto.getCacheKey();
        if(!StringUtils.hasText(cacheKey)){
            return;
        }
        List<ActivityPlanItemTerminalDto> cacheList = this.findCacheList(cacheKey + ":" + dto.getId());
        dto.setActivityPlanItemTerminalList(cacheList);
    }

    @Override
    public void changeTerminalDetail(ActivityPlanItemDto dto, boolean autoFill) {
        String cacheKey = dto.getCacheKey();
        log.info("更新方案明细["+cacheKey+"]门店数据1");
        if(!StringUtils.hasText(cacheKey)){
            return;
        }
        log.info("更新方案明细["+cacheKey+"]门店数据2");
        cacheKey = cacheKey + ":" + dto.getId();
        if(autoFill) {
            this.saveListCache(cacheKey, dto.getActivityPlanItemTerminalList());
        }else {
            this.saveCurrentPageCache(cacheKey,dto.getActivityPlanItemTerminalList());
        }
    }

    @Override
    public void importNewItem(String cacheKey, List<ActivityPlanItemTerminalDto> itemTerminalDtoList) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        List<ActivityPlanItemTerminalDto> itemTerminalList = itemDto.getActivityPlanItemTerminalList();
        if (null == itemTerminalList){
            itemTerminalList = Lists.newArrayList();
            itemDto.setActivityPlanItemTerminalList(itemTerminalList);
        }
        for (ActivityPlanItemTerminalDto terminalDto : itemTerminalDtoList) {
            terminalDto.setId(UUID.randomUUID().toString().replace("-", ""));
            terminalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            terminalDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        itemTerminalList.addAll(0,itemTerminalDtoList);
        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public void clearCache(String cacheKey) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        itemDto.setActivityPlanItemTerminalList(null);
        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public void addItemCache(String cacheKey, List<ActivityPlanItemTerminalDto> itemList) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        List<ActivityPlanItemTerminalDto> itemTerminalList = itemDto.getActivityPlanItemTerminalList();
        if (null == itemTerminalList){
            itemTerminalList = Lists.newArrayList();
            itemDto.setActivityPlanItemTerminalList(itemTerminalList);
        }

        ActivityPlanItemTerminalDto newItem = new ActivityPlanItemTerminalDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        itemTerminalList.add(0,newItem);

        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public Page<ActivityPlanItemTerminalVo> findCachePageList(Pageable pageable, ActivityPlanItemTerminalDto dto, String cacheKey) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        Page<ActivityPlanItemTerminalVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(com.google.common.collect.Lists.newArrayList());
        List<ActivityPlanItemTerminalDto> dtoList = itemDto.getActivityPlanItemTerminalList();
        if (CollectionUtils.isEmpty(dtoList)){
            return page;
        }

        //放到缓存里面
        page.setTotal(dtoList.size());
        long start = page.offset();
        if (page.getTotal() > start) {
            long end = page.offset() + page.getSize();
            if (page.getTotal() < end) {
                end = page.getTotal();
            }
            List<ActivityPlanItemTerminalDto> recordDtoList = dtoList.subList((int) page.offset(), (int) end);
            List<ActivityPlanItemTerminalVo> voList = (List<ActivityPlanItemTerminalVo>) nebulaToolkitService.copyCollectionByWhiteList(recordDtoList, ActivityPlanItemTerminalDto.class, ActivityPlanItemTerminalVo.class, HashSet.class, ArrayList.class);
            page.setRecords(voList);
        }
        return page;
    }

    @Override
    public List<ActivityPlanItemTerminalDto> findCacheList(String cacheKey) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        return itemDto.getActivityPlanItemTerminalList();
    }

    @Override
    public void saveListCache(String cacheKey, List<ActivityPlanItemTerminalDto> itemList) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        itemDto.setActivityPlanItemTerminalList(itemList);

        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        log.info("更新方案明细["+itemId+"]门店数据"+ JSON.toJSONString(itemDto.getActivityPlanItemTerminalList()));
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public void copyItemListCache(String cacheKey, List<ActivityPlanItemTerminalDto> updateList) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        List<ActivityPlanItemTerminalDto> itemTerminalList = itemDto.getActivityPlanItemTerminalList();
        if (null == itemTerminalList){
            itemTerminalList = Lists.newArrayList();
            itemDto.setActivityPlanItemTerminalList(itemTerminalList);
        }

        Map<String, ActivityPlanItemTerminalDto> existsMap = itemTerminalList.stream().collect(Collectors.toMap(ActivityPlanItemTerminalDto::getId, Function.identity()));
        for (ActivityPlanItemTerminalDto terminalDto : updateList) {
            ActivityPlanItemTerminalDto oldTerminalDto = existsMap.get(terminalDto.getId());
            if (BooleanEnum.TRUE.getNumStr().equals(terminalDto.getChecked())) {
                //选中状态，代表该行数据要复制
                ActivityPlanItemTerminalDto copyTerminalDto = nebulaToolkitService.copyObjectByWhiteList(terminalDto, ActivityPlanItemTerminalDto.class, HashSet.class, ArrayList.class);
                copyTerminalDto.setId(UUID.randomUUID().toString().replace("-", ""));
                copyTerminalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                copyTerminalDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                itemTerminalList.add(0,copyTerminalDto);
            } else {
                if (null == oldTerminalDto){
                    continue;//不应该出现
                }
                BeanUtils.copyProperties(terminalDto,oldTerminalDto);
            }
        }

        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public void saveCurrentPageCache(String cacheKey, List<ActivityPlanItemTerminalDto> updateList) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }
        List<ActivityPlanItemTerminalDto> itemTerminalList = itemDto.getActivityPlanItemTerminalList();
        if (null == itemTerminalList){
            itemTerminalList = Lists.newArrayList();
            itemDto.setActivityPlanItemTerminalList(itemTerminalList);
        }
        Map<String, ActivityPlanItemTerminalDto> existsMap = itemTerminalList.stream().collect(Collectors.toMap(ActivityPlanItemTerminalDto::getId, Function.identity()));
        for (ActivityPlanItemTerminalDto terminalDto : updateList) {
            ActivityPlanItemTerminalDto oldTerminalDto = existsMap.get(terminalDto.getId());
            if (null == oldTerminalDto){
                continue;//不应该出现
            }
            BeanUtils.copyProperties(terminalDto,oldTerminalDto);
        }

        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public void deleteCacheList(String cacheKey, List<ActivityPlanItemTerminalDto> updateList) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        if (null == itemDto){
            throw new RuntimeException("明细数据未保存，请检查");
        }

        List<ActivityPlanItemTerminalDto> itemTerminalList = itemDto.getActivityPlanItemTerminalList();
        List<ActivityPlanItemTerminalDto> newItemTerminalList = Lists.newArrayList();
        Map<String, ActivityPlanItemTerminalDto> updateTerminalMap = updateList.stream().collect(Collectors.toMap(ActivityPlanItemTerminalDto::getId, Function.identity()));
        for (ActivityPlanItemTerminalDto terminalDto : itemTerminalList) {
            if (updateTerminalMap.containsKey(terminalDto.getId())){
                ActivityPlanItemTerminalDto newTerminalDto = updateTerminalMap.get(terminalDto.getId());
                if (!BooleanEnum.TRUE.getNumStr().equals(newTerminalDto.getChecked())){
                    newItemTerminalList.add(newTerminalDto);
                }
            }else{
                newItemTerminalList.add(terminalDto);
            }
        }
        itemDto.setActivityPlanItemTerminalList(newItemTerminalList);

        Map<Object, ActivityPlanItemDto> updateMap = Maps.newHashMap();
        updateMap.put(itemId,itemDto);
        itemHelper.doSaveItem(itemCacheKey,updateMap );
    }

    @Override
    public void proportionByTerminal(ActivityPlanItemDto dto) {
        Validate.notBlank(dto.getRegion(), "区域不能为空");
        Validate.notBlank(dto.getSystemCode(), "零售商不能为空");
        BigDecimal totalFeeAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getTotalFeeAmountStr())) {
            try {
                String totalFeeAmountStr = dto.getTotalFeeAmountStr();
                totalFeeAmount = new BigDecimal(totalFeeAmountStr);
                dto.setTotalFeeAmount(totalFeeAmount);
            } catch (Exception e) {
                throw new RuntimeException("费用合计格式有误！");
            }
        }else{
            throw new RuntimeException("费用合计不能为空");
        }
        //获取门店明细
        this.addTerminalFromCache(dto);
        //是否自动填充标识
        boolean autoFill = false;
        if (CollectionUtils.isEmpty(dto.getActivityPlanItemTerminalList())) {
            List<TerminalVo> terminalVoList = this.findAvaiableTerminal(dto);
            List<ActivityPlanItemTerminalDto> activityPlanItemTerminalDtoList = new ArrayList<>();
            for (TerminalVo terminalVo : terminalVoList) {
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = new ActivityPlanItemTerminalDto();
                activityPlanItemTerminalDto.setTerminalCode(terminalVo.getTerminalCode());
                activityPlanItemTerminalDto.setTerminalName(terminalVo.getTerminalName());
                activityPlanItemTerminalDto.setId(UUID.randomUUID().toString().replace("-", ""));
                activityPlanItemTerminalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                activityPlanItemTerminalDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                activityPlanItemTerminalDtoList.add(activityPlanItemTerminalDto);
            }
            dto.setActivityPlanItemTerminalList(activityPlanItemTerminalDtoList);
            autoFill = true;
        }
        Validate.isTrue(!CollectionUtils.isEmpty(dto.getActivityPlanItemTerminalList()), "门店明细不能为空");
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getPeriodPromotionalNumberStr())) {
            try {
                dto.setPeriodPromotionalNumber(Integer.parseInt(dto.getPeriodPromotionalNumberStr()));
            } catch (Exception e) {
                throw new RuntimeException("期间促销件数格式有误！");
            }
        }
        //2023-05-06 按门店分摊时的四舍五入逻辑修改：
        //按照依次发放的逻辑进行分摊，20件产品，9个门店，依次发放1，再依次发放1，最后2个给第1家和第二家门店
        //需要区分模板：促销模板按照期间促销件数+单间申请费用分摊；如果是陈列模板的话用陈列单价和陈列数量分摊
        Integer periodPromotionalNumberInteger;
        if ("陈列方案模板".equals(dto.getTemplateConfigName())) {
            Validate.notBlank(dto.getDisplayQuantityStr(), "陈列数量不能为空");
            periodPromotionalNumberInteger = new BigDecimal(dto.getDisplayQuantityStr()).intValue();
        } else {
            periodPromotionalNumberInteger = dto.getPeriodPromotionalNumber();
            Validate.notNull(periodPromotionalNumberInteger, "期间促销件数不能为空");
        }
        int terminalNum = dto.getActivityPlanItemTerminalList().size();
        int average = periodPromotionalNumberInteger / terminalNum;
        int averageAddOne = 0;
        if (periodPromotionalNumberInteger > 0){
            averageAddOne = average + 1;
        }else if (periodPromotionalNumberInteger < 0){
            averageAddOne = average - 1;
        }
        BigDecimal averageDecimal = new BigDecimal(average);
        BigDecimal averageAddOneDecimal = new BigDecimal(averageAddOne);
        int remainder = Math.abs(periodPromotionalNumberInteger % terminalNum);
        Iterator<ActivityPlanItemTerminalDto> iterator = dto.getActivityPlanItemTerminalList().iterator();
        int i = 0;
        while (iterator.hasNext()){
            ActivityPlanItemTerminalDto activityPlanItemTerminalDto = iterator.next();
            if ("陈列方案模板".equals(dto.getTemplateConfigName())) {
                if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getPriceStr())) {
                    activityPlanItemTerminalDto.setStandard(new BigDecimal(dto.getPriceStr()));
                }
            } else {
                if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getSingleApplicationFeeStr())) {
                    activityPlanItemTerminalDto.setStandard(new BigDecimal(dto.getSingleApplicationFeeStr()));
                }
            }
            //处理尾差
            if (i < remainder) {
                activityPlanItemTerminalDto.setQuantity(averageAddOneDecimal);
            } else {
                activityPlanItemTerminalDto.setQuantity(averageDecimal);
            }
            if (Objects.nonNull(activityPlanItemTerminalDto.getStandard()) && Objects.nonNull(activityPlanItemTerminalDto.getQuantity())) {
                activityPlanItemTerminalDto.setAmount(activityPlanItemTerminalDto.getStandard().multiply(activityPlanItemTerminalDto.getQuantity()));
            }
            i++;
            if (null == activityPlanItemTerminalDto.getAmount() || activityPlanItemTerminalDto.getAmount().compareTo(BigDecimal.ZERO) == 0){
                iterator.remove();
                continue;
            }
            activityPlanItemTerminalDto.setFloatingRateStr(dto.getFloatingRateStr());
            activityPlanItemTerminalDto.setFloatingRate(dto.getFloatingRate());
        }
        proportionAmountByTotalFeeAmount(dto);
        this.changeTerminalDetail(dto, autoFill);
    }

    @Override
    public void proportionByTerminalEmployee(ActivityPlanItemDto dto) {
        Validate.notBlank(dto.getRegion(), "区域不能为空");
//        Validate.notBlank(dto.getRetailBusinessmanCode(),"零售商不能为空");
        BigDecimal totalFeeAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getTotalFeeAmountStr())) {
            try {
                String totalFeeAmountStr = dto.getTotalFeeAmountStr();
                totalFeeAmount = new BigDecimal(totalFeeAmountStr);
                dto.setTotalFeeAmount(totalFeeAmount);
            } catch (Exception e) {
                throw new RuntimeException("费用合计格式有误！");
            }
        }else{
            throw new RuntimeException("费用合计不能为空");
        }
        //获取门店明细
        this.addTerminalFromCache(dto);
        //是否自动填充标识
        boolean autoFill = false;
        if (CollectionUtils.isEmpty(dto.getActivityPlanItemTerminalList())) {

            PromotersDto promotersDto = new PromotersDto();
            promotersDto.setRegionCode(dto.getRegion());
            promotersDto.setCustomerRetailerCode(dto.getSystemCode());
            promotersDto.setDistrictLevelCode(dto.getCityLevel());
            promotersDto.setTerminalTypeCode(dto.getAcStoreType());
            promotersDto.setPointsWarehouseCode(dto.getAcWarehouseCode());
            promotersDto.setBusinessUnitCode(BusinessUnitEnum.VERTICAL.getCode());//这个功能只有垂直在用，写死垂直吧

            List<PromotersVo> promotersVoList = promotersVoService.findTerminalPromoters(promotersDto);
            List<ActivityPlanItemTerminalDto> activityPlanItemTerminalDtoList = new ArrayList<>();
            for (PromotersVo promotersVo : promotersVoList) {
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = new ActivityPlanItemTerminalDto();
                BeanUtils.copyProperties(promotersVo, activityPlanItemTerminalDto);
                activityPlanItemTerminalDto.setId(UUID.randomUUID().toString().replace("-", ""));
                activityPlanItemTerminalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                activityPlanItemTerminalDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                activityPlanItemTerminalDtoList.add(activityPlanItemTerminalDto);
            }
            dto.setActivityPlanItemTerminalList(activityPlanItemTerminalDtoList);
            autoFill = true;
        }
        Validate.isTrue(!CollectionUtils.isEmpty(dto.getActivityPlanItemTerminalList()), "门店明细不能为空");
        Iterator<ActivityPlanItemTerminalDto> iterator = dto.getActivityPlanItemTerminalList().iterator();
        int i = 0;
        while (iterator.hasNext()){
            ActivityPlanItemTerminalDto activityPlanItemTerminalDto = dto.getActivityPlanItemTerminalList().get(i);
            activityPlanItemTerminalDto.setQuantity(BigDecimal.ONE);
            if (Objects.nonNull(activityPlanItemTerminalDto.getStandard()) && Objects.nonNull(activityPlanItemTerminalDto.getQuantity())) {
                activityPlanItemTerminalDto.setAmount(activityPlanItemTerminalDto.getStandard().multiply(activityPlanItemTerminalDto.getQuantity()));
            }
            i++;
            if (null == activityPlanItemTerminalDto.getAmount() || activityPlanItemTerminalDto.getAmount().compareTo(BigDecimal.ZERO) == 0){
                iterator.remove();
                continue;
            }
            activityPlanItemTerminalDto.setFloatingRateStr(dto.getFloatingRateStr());
            activityPlanItemTerminalDto.setFloatingRate(dto.getFloatingRate());
        }
        proportionAmountByTotalFeeAmount(dto);
        this.changeTerminalDetail(dto, autoFill);
    }

    @Override
    public void proportionByPos(ActivityPlanDto dto, String cacheKey, List<ActivityPlanItemDto> itemCacheList) {
        //垂直
        if (!BusinessUnitEnum.VERTICAL.getCode().equals(dto.getBusinessUnitCode())) {
            return;
        }
        if (CollectionUtils.isEmpty(itemCacheList)) {
            return;
        }
        itemCacheList.forEach(item -> this.clearCache(cacheKey + ":" + item.getId()));
        for (int i = 0; i < itemCacheList.size(); i++) {
            try {
                ActivityPlanItemDto item = itemCacheList.get(i);
                if (!org.apache.commons.lang3.StringUtils.isEmpty(item.getTerminalCode())) {
                    continue;
                }
                item.setCacheKey(cacheKey);
                item.setTemplateConfigName(dto.getTemplateConfigName());
                this.proportionByPos(item);
            } catch (Exception e) {
                List<String> itemIdList = itemCacheList.stream().map(ActivityPlanItemDto::getId).collect(Collectors.toList());
                for (int k = 0; k <= i; k++) {
                    this.clearCache(cacheKey + ":" + itemIdList.get(k));
                }
                log.error(e.getMessage(), e);
                throw new RuntimeException("第【" + (i + 1) + "】行，" + e.getMessage());
            }
        }
    }

    public void proportionByPos(ActivityPlanItemDto dto) {
        Validate.notBlank(dto.getRegion(), "区域不能为空");
        Validate.notBlank(dto.getSystemCode(), "零售商不能为空");
        Validate.notBlank(dto.getActivityBeginDateStr(), "活动开始时间不能为空");
        Validate.notBlank(dto.getActivityEndDateStr(), "活动结束时间不能为空");
        Validate.notBlank(dto.getProductCode(), "产品编码不能为空");
        //是否自动填充标识
        List<ActivityPlanItemTerminalDto> activityPlanItemTerminalDtoList = new ArrayList<>();
        List<TerminalVo> terminalVoList = findAvaiableTerminal(dto);
        for (TerminalVo terminalVo : terminalVoList) {
            ActivityPlanItemTerminalDto activityPlanItemTerminalDto = new ActivityPlanItemTerminalDto();
            activityPlanItemTerminalDto.setTerminalCode(terminalVo.getTerminalCode());
            activityPlanItemTerminalDto.setTerminalName(terminalVo.getTerminalName());
            activityPlanItemTerminalDto.setId(UUID.randomUUID().toString().replace("-", ""));
            activityPlanItemTerminalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            activityPlanItemTerminalDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            activityPlanItemTerminalDtoList.add(activityPlanItemTerminalDto);
        }
        dto.setActivityPlanItemTerminalList(activityPlanItemTerminalDtoList);
        proportionByTerminalPos(dto);
    }

    public void proportionByTerminalPos(ActivityPlanItemDto dto) {
        Validate.isTrue(!CollectionUtils.isEmpty(dto.getActivityPlanItemTerminalList()), "门店明细不能为空");
        SalesDataDto salesDataDto = new SalesDataDto();
        try {
            SimpleDateFormat dayFormat = new SimpleDateFormat(DateUtil.DEFAULT_YEAR_MONTH_DAY);
            Date beginDate = dayFormat.parse(dto.getActivityBeginDateStr());
            Date endDate = dayFormat.parse(dto.getActivityEndDateStr());
            Calendar instance = Calendar.getInstance();
            instance.setTime(beginDate);
            instance.add(Calendar.MONTH,-1);
            salesDataDto.setStartTime(dayFormat.format(instance.getTime()));
            instance.setTime(endDate);
            instance.add(Calendar.MONTH,-1);
            salesDataDto.setEndTime(dayFormat.format(instance.getTime()) + " 23:59:59");
        }catch (Exception e){
            throw new RuntimeException("活动时间格式有误！");
        }
        BigDecimal feeAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getFeeAmountStr())) {
            try {
                String feeAmountStr = dto.getFeeAmountStr();
                feeAmount = new BigDecimal(feeAmountStr);
                dto.setFeeAmount(feeAmount);
            } catch (Exception e) {
                throw new RuntimeException("我方承担金额格式有误！");
            }
        }else{
            throw new RuntimeException("我方承担金额不能为空");
        }


        //查询pos数据
        List<String> terminalCodeList = dto.getActivityPlanItemTerminalList().stream().map(ActivityPlanItemTerminalDto::getTerminalCode).collect(Collectors.toList());
        salesDataDto.setDeliveryPartyCodeList(terminalCodeList);
        salesDataDto.setGoodsCode(dto.getProductCode());
        List<SalesDataVo> kmsPosList = invoiceSalesDataVoService.findKmsPosList(salesDataDto);
        boolean proportionAvage = false;
        int terminalSize = dto.getActivityPlanItemTerminalList().size();
        BigDecimal avageAmount = feeAmount.divide(new BigDecimal(terminalSize),2, RoundingMode.HALF_UP);//舍位
        BigDecimal totalSalesAmount = BigDecimal.ONE;
        Map<String, SalesDataVo> kmsPosMap = Maps.newHashMap();
        if (CollectionUtils.isEmpty(kmsPosList)){
            //没有pos数据就均摊
            proportionAvage = true;
        }else{
            totalSalesAmount = kmsPosList.stream().map(SalesDataVo::getSalesAmount).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
            if (totalSalesAmount.compareTo(BigDecimal.ZERO) == 0){
                proportionAvage = true;
            }
            kmsPosMap = kmsPosList.stream().collect(Collectors.toMap(item -> item.getDeliveryPartyCode() + item.getGoodsCode(), Function.identity()));
        }


        Iterator<ActivityPlanItemTerminalDto> iterator = dto.getActivityPlanItemTerminalList().iterator();
        BigDecimal leaveFeeAmount = feeAmount;
        while (iterator.hasNext()){
            ActivityPlanItemTerminalDto activityPlanItemTerminalDto = iterator.next();
            if (proportionAvage){
                activityPlanItemTerminalDto.setFeeAmount(avageAmount);
            }else{
                String key = activityPlanItemTerminalDto.getTerminalCode() + dto.getProductCode();
                if (kmsPosMap.containsKey(key)){
                    //有pos数据，分摊金额= 总费用金额 * (门店销售额/总销售额)
                    BigDecimal salesAmount = Optional.ofNullable(kmsPosMap.get(key).getSalesAmount()).orElse(BigDecimal.ZERO);
                    BigDecimal amount = feeAmount.multiply(salesAmount).divide(totalSalesAmount,2,RoundingMode.HALF_UP);
                    activityPlanItemTerminalDto.setFeeAmount(amount);
                }
            }
            //分摊金额为0，排除掉
            if (null == activityPlanItemTerminalDto.getFeeAmount() || activityPlanItemTerminalDto.getFeeAmount().compareTo(BigDecimal.ZERO) == 0){
                iterator.remove();
                continue;
            }
            leaveFeeAmount = leaveFeeAmount.subtract(activityPlanItemTerminalDto.getFeeAmount());
            activityPlanItemTerminalDto.setFloatingRateStr(dto.getFloatingRateStr());
            activityPlanItemTerminalDto.setFloatingRate(dto.getFloatingRate());
        }
        if (leaveFeeAmount.compareTo(BigDecimal.ZERO) != 0){
            ActivityPlanItemTerminalDto lastTerminalDto = dto.getActivityPlanItemTerminalList().get(dto.getActivityPlanItemTerminalList().size() - 1);
            lastTerminalDto.setFeeAmount(lastTerminalDto.getFeeAmount().add(leaveFeeAmount));
        }
        proportionAmountByFeeAmount(dto);
        this.changeTerminalDetail(dto, true);
    }

    /**
     * 根据费用金额的分摊比例分摊下费用合计和系统承担金额
     */
    private void proportionAmountByFeeAmount(ActivityPlanItemDto dto){
        List<ActivityPlanItemTerminalDto> activityPlanItemTerminalList = dto.getActivityPlanItemTerminalList();
        if (CollectionUtils.isEmpty(activityPlanItemTerminalList)){
            return;
        }
        Map<String, BigDecimal> feeAmountMap = activityPlanItemTerminalList.stream().filter(item -> null != item.getFeeAmount()).collect(Collectors.toMap(ActivityPlanItemTerminalDto::getId,ActivityPlanItemTerminalDto::getFeeAmount));
        BigDecimal feeAmount = dto.getFeeAmount();
        BigDecimal totalFeeAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getTotalFeeAmountStr())) {
            try {
                String totalFeeAmountStr = dto.getTotalFeeAmountStr();
                totalFeeAmount = new BigDecimal(totalFeeAmountStr);
                dto.setTotalFeeAmount(totalFeeAmount);
            } catch (Exception e) {
                throw new RuntimeException("费用合计格式有误！");
            }
            Iterator<ActivityPlanItemTerminalDto> iterator = activityPlanItemTerminalList.iterator();
            BigDecimal leaveAmount = totalFeeAmount;
            while (iterator.hasNext()){
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = iterator.next();
                String key = activityPlanItemTerminalDto.getId();
                BigDecimal thisFeeAmount = Optional.ofNullable(feeAmountMap.get(key)).orElse(BigDecimal.ZERO);
                BigDecimal thisAmount = totalFeeAmount.multiply(thisFeeAmount).divide(feeAmount,2,RoundingMode.HALF_UP);
                activityPlanItemTerminalDto.setAmount(thisAmount);
                leaveAmount = leaveAmount.subtract(thisAmount);
            }
            if (leaveAmount.compareTo(BigDecimal.ZERO) != 0){
                ActivityPlanItemTerminalDto lastTerminalDto = activityPlanItemTerminalList.get(activityPlanItemTerminalList.size() - 1);
                lastTerminalDto.setAmount(lastTerminalDto.getAmount().add(leaveAmount));
            }
        }
        BigDecimal systemBorneAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getSystemBorneAmountStr())) {
            try {
                String systemBorneAmountStr = dto.getSystemBorneAmountStr();
                systemBorneAmount = new BigDecimal(systemBorneAmountStr);
                dto.setSystemBorneAmount(systemBorneAmount);
            } catch (Exception e) {
                throw new RuntimeException("系统承担金额格式有误！");
            }
            Iterator<ActivityPlanItemTerminalDto> iterator = activityPlanItemTerminalList.iterator();
            BigDecimal leaveAmount = systemBorneAmount;
            while (iterator.hasNext()){
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = iterator.next();
                String key = activityPlanItemTerminalDto.getId();
                BigDecimal thisFeeAmount = Optional.ofNullable(feeAmountMap.get(key)).orElse(BigDecimal.ZERO);
                BigDecimal thisAmount = systemBorneAmount.multiply(thisFeeAmount).divide(feeAmount,2,RoundingMode.HALF_UP);
                activityPlanItemTerminalDto.setSystemBorneAmount(thisAmount);
                leaveAmount = leaveAmount.subtract(thisAmount);
            }
            if (leaveAmount.compareTo(BigDecimal.ZERO) != 0){
                ActivityPlanItemTerminalDto lastTerminalDto = activityPlanItemTerminalList.get(activityPlanItemTerminalList.size() - 1);
                lastTerminalDto.setSystemBorneAmount(lastTerminalDto.getSystemBorneAmount().add(leaveAmount));
            }
        }
    }


    /**
     * 根据费用金额的分摊比例分摊下费用合计和系统承担金额
     */
    @Override
    public void proportionAmountByTotalFeeAmount(ActivityPlanItemDto dto){
        List<ActivityPlanItemTerminalDto> activityPlanItemTerminalList = dto.getActivityPlanItemTerminalList();
        if (CollectionUtils.isEmpty(activityPlanItemTerminalList)){
            return;
        }
        Map<String, BigDecimal> feeAmountMap = activityPlanItemTerminalList.stream().filter(item -> null != item.getAmount()).collect(Collectors.toMap(ActivityPlanItemTerminalDto::getId,ActivityPlanItemTerminalDto::getAmount));
        BigDecimal totalFeeAmount = dto.getTotalFeeAmount();
        BigDecimal feeAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getFeeAmountStr())) {
            try {
                String feeAmountStr = dto.getFeeAmountStr();
                feeAmount = new BigDecimal(feeAmountStr);
                dto.setFeeAmount(feeAmount);
            } catch (Exception e) {
                throw new RuntimeException("我方承担金额格式有误！");
            }
            Iterator<ActivityPlanItemTerminalDto> iterator = activityPlanItemTerminalList.iterator();
            BigDecimal leaveAmount = feeAmount;
            while (iterator.hasNext()){
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = iterator.next();
                String key = activityPlanItemTerminalDto.getId();
                BigDecimal thisTotalAmount = Optional.ofNullable(feeAmountMap.get(key)).orElse(BigDecimal.ZERO);
                BigDecimal thisAmount = feeAmount.multiply(thisTotalAmount).divide(totalFeeAmount,2,RoundingMode.HALF_UP);
                activityPlanItemTerminalDto.setFeeAmount(thisAmount);
                leaveAmount = leaveAmount.subtract(thisAmount);
            }
            if (leaveAmount.compareTo(BigDecimal.ZERO) != 0){
                ActivityPlanItemTerminalDto lastTerminalDto = activityPlanItemTerminalList.get(activityPlanItemTerminalList.size() - 1);
                lastTerminalDto.setFeeAmount(lastTerminalDto.getFeeAmount().add(leaveAmount));
            }
        }
        BigDecimal systemBorneAmount = null;
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getSystemBorneAmountStr())) {
            try {
                String systemBorneAmountStr = dto.getSystemBorneAmountStr();
                systemBorneAmount = new BigDecimal(systemBorneAmountStr);
                dto.setSystemBorneAmount(systemBorneAmount);
            } catch (Exception e) {
                throw new RuntimeException("系统承担金额格式有误！");
            }
            Iterator<ActivityPlanItemTerminalDto> iterator = activityPlanItemTerminalList.iterator();
            BigDecimal leaveAmount = systemBorneAmount;
            while (iterator.hasNext()){
                ActivityPlanItemTerminalDto activityPlanItemTerminalDto = iterator.next();
                String key = activityPlanItemTerminalDto.getId();
                BigDecimal thisFeeAmount = Optional.ofNullable(feeAmountMap.get(key)).orElse(BigDecimal.ZERO);
                BigDecimal thisAmount = systemBorneAmount.multiply(thisFeeAmount).divide(totalFeeAmount,2,RoundingMode.HALF_UP);
                activityPlanItemTerminalDto.setSystemBorneAmount(thisAmount);
                leaveAmount = leaveAmount.subtract(thisAmount);
            }
            if (leaveAmount.compareTo(BigDecimal.ZERO) != 0){
                ActivityPlanItemTerminalDto lastTerminalDto = activityPlanItemTerminalList.get(activityPlanItemTerminalList.size() - 1);
                lastTerminalDto.setSystemBorneAmount(lastTerminalDto.getSystemBorneAmount().add(leaveAmount));
            }
        }
    }

    @Override
    public List<TerminalVo> findAvaiableTerminal(ActivityPlanItemDto dto){
        TerminalPaginationDto terminalPaginationDto = new TerminalPaginationDto();
        terminalPaginationDto.setRegionCode(dto.getRegion());
        terminalPaginationDto.setCustomerRetailerCode(dto.getSystemCode());
        terminalPaginationDto.setTerminalTypeCode(dto.getAcStoreType());
        terminalPaginationDto.setDistrictLevelCode(dto.getCityLevel());
        terminalPaginationDto.setTerminalTypeCode(dto.getAcStoreType());
        terminalPaginationDto.setPointsWarehouseCode(dto.getAcWarehouseCode());
        terminalPaginationDto.setTerminalState("1");
        terminalPaginationDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        terminalPaginationDto.setBusinessUnitCode(BusinessUnitEnum.VERTICAL.getCode());//这个功能只有垂直在用，写死垂直吧

        if (WriteOffMethodEnum.THINGS.getCode().equals(dto.getWriteOffMethod())) {
            terminalPaginationDto.setSettleDestination("1");
        }
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getTerminalCode())){
            terminalPaginationDto.setTerminalCode(dto.getTerminalCode());
        }
        List<TerminalVo> terminalVoList = terminalVoService.findListByConditions(terminalPaginationDto);
        if (!CollectionUtils.isEmpty(terminalVoList)){
//            1035250 【活动方案-新增/编辑】活动方案申请时“按岗店分摊”、“按门店分摊”和“按POS分摊”功能过滤掉门店增加条件
            terminalVoList = terminalVoList.stream().filter(item -> StringUtil.isNotEmpty(item.getSellerCode()) && StringUtil.isNotEmpty(item.getSalesInstitutionCode())).collect(Collectors.toList());
        }
        return terminalVoList;
    }

    @Override
    public void initActivityPlanItemTerminal(ActivityPlanItemTerminalDto dto, String cacheKey) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        List<TerminalVo> terminalVoList = findAvaiableTerminal(itemDto);
        List<ActivityPlanItemTerminalDto> activityPlanItemTerminalDtoList = Lists.newArrayList();
        for (TerminalVo terminalVo : terminalVoList) {
            ActivityPlanItemTerminalDto activityPlanItemTerminalDto = new ActivityPlanItemTerminalDto();
            activityPlanItemTerminalDto.setTerminalCode(terminalVo.getTerminalCode());
            activityPlanItemTerminalDto.setTerminalName(terminalVo.getTerminalName());
            activityPlanItemTerminalDto.setId(UUID.randomUUID().toString().replace("-", ""));
            activityPlanItemTerminalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            activityPlanItemTerminalDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            activityPlanItemTerminalDtoList.add(activityPlanItemTerminalDto);
        }
        itemDto.setActivityPlanItemTerminalList(activityPlanItemTerminalDtoList);
        this.changeTerminalDetail(itemDto, true);
    }

    @Override
    public void proportionPosByCurrTerminal(ActivityPlanItemTerminalDto dto, String cacheKey) {
        String itemCacheKey = cacheKey.substring(0,cacheKey.lastIndexOf(":"));
        String itemId = cacheKey.substring(cacheKey.lastIndexOf(":")+1);
        ActivityPlanItemDto itemDto = itemHelper.getDtoByKey(itemCacheKey, itemId);
        itemDto.setCacheKey(itemCacheKey);
        proportionByTerminalPos(itemDto);
    }
}
