package com.biz.crm.tpm.business.activity.plan.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlan;
import com.biz.crm.tpm.business.activity.plan.local.entity.ActivityPlanItem;
import com.biz.crm.tpm.business.activity.plan.local.repository.ActivityPlanItemRepository;
import com.biz.crm.tpm.business.activity.plan.local.repository.ActivityPlanRepository;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanItemService;
import com.biz.crm.tpm.business.activity.plan.local.service.ActivityPlanLockService;
import com.biz.crm.tpm.business.activity.plan.sdk.constant.ActivityPlanConstant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 活动方案加锁处理service
 *
 * @author wanghaojia
 * @date 2023/9/9 16:28
 */
@Service
public class ActivityPlanLockServiceImpl implements ActivityPlanLockService {

    @Autowired(required = false)
    private ActivityPlanItemService activityPlanItemService;

    @Autowired(required = false)
    private ActivityPlanRepository activityPlanRepository;

    @Autowired(required = false)
    private ActivityPlanItemRepository activityPlanItemRepository;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private ActivityPlanItemPageCacheHelper activityPlanItemPageCacheHelper;

    /**
     * 活动方案关闭明细
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void closeItemList(List<ActivityPlanItem> itemList) {
        activityPlanItemPageCacheHelper.sendCloseMsg("开始执行活动方案一键关闭...");
        if (CollectionUtil.isEmpty(itemList)) {
            activityPlanItemPageCacheHelper.sendCloseMsg("请选择方案明细,活动方案关闭处理结束!", ExecStatusEnum.FINISH.getKey());
            return;
        }
        boolean lockSuccess = false;
        List<String> ids = itemList.stream().map(ActivityPlanItem::getId).collect(Collectors.toList());
        try {
            //锁1小时
            lockSuccess = redisLockService.batchLock(ActivityPlanConstant.LOCK_ACTIVITY_PLAN_ITEM_CLOSE, ids, TimeUnit.HOURS, 1);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            activityPlanItemService.closeItemList(itemList, true);
            activityPlanItemPageCacheHelper.sendCloseMsg("活动方案关闭处理成功!", ExecStatusEnum.FINISH.getKey());
        } catch (Exception e) {
            activityPlanItemPageCacheHelper.sendCloseMsg("活动方案关闭处理失败：" + e.getMessage(), ExecStatusEnum.FINISH.getKey());
            throw e;
        } finally {
            if (lockSuccess) {
                redisLockService.batchUnLock(ActivityPlanConstant.LOCK_ACTIVITY_PLAN_ITEM_CLOSE, ids);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void closeItemByPlanIds(List<String> ids) {
        activityPlanItemPageCacheHelper.sendCloseMsg("开始执行活动方案一键关闭...", ExecStatusEnum.RUNNING.getKey());
        if (CollectionUtils.isEmpty(ids)) {
            activityPlanItemPageCacheHelper.sendCloseMsg("请选择方案,活动方案关闭处理结束", ExecStatusEnum.FINISH.getKey());
            return;
        }
        List<ActivityPlan> planList = activityPlanRepository.listByIds(ids);
        if (CollectionUtils.isEmpty(planList)) {
            activityPlanItemPageCacheHelper.sendCloseMsg("选择的方案不存在,活动方案关闭处理结束", ExecStatusEnum.FINISH.getKey());
            return;
        }
        List<String> planCodeList = planList.stream().map(ActivityPlan::getPlanCode).collect(Collectors.toList());
        List<String> itemIds = activityPlanItemRepository.listIdByPlanCodes(planCodeList);
        if (CollectionUtils.isEmpty(itemIds)) {
            activityPlanItemPageCacheHelper.sendCloseMsg("选择的方案未找到方案明细,活动方案关闭处理结束", ExecStatusEnum.FINISH.getKey());
            return;
        }
        boolean lockSuccess = false;
        try {
            //锁1小时
            lockSuccess = redisLockService.batchLock(ActivityPlanConstant.LOCK_ACTIVITY_PLAN_ITEM_CLOSE, ids, TimeUnit.HOURS, 1);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            List<String> errorMsgList = activityPlanItemService.closeItem(itemIds, false);
            Assert.isTrue(CollectionUtils.isEmpty(errorMsgList), String.join(",", errorMsgList));
            activityPlanItemPageCacheHelper.sendCloseMsg("活动方案关闭处理成功", ExecStatusEnum.FINISH.getKey());
        } catch (Exception e) {
            activityPlanItemPageCacheHelper.sendCloseMsg("活动方案关闭处理结束:" + e.getMessage(), ExecStatusEnum.FINISH.getKey());
            throw e;
        } finally {
            if (lockSuccess) {
                redisLockService.batchUnLock(ActivityPlanConstant.LOCK_ACTIVITY_PLAN_ITEM_CLOSE, ids);
            }
        }
    }

}
