package com.biz.crm.tpm.business.activity.put.cost.local.service.internal;

import com.alibaba.excel.util.Validate;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.put.cost.sdk.constant.ActivityPutCostConstant;
import com.biz.crm.tpm.business.activity.put.cost.sdk.dto.ActivityPutCostDto;
import com.biz.crm.tpm.business.activity.put.cost.sdk.service.ActivityPutCostService;
import com.biz.crm.tpm.business.activity.put.cost.sdk.vo.ActivityPutCostImportVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2022/11/11 16:32
 * @description: 电商活动投放费用管理(ActivityPutCost)导入实现层
 */
@Slf4j
@Component
public class ActivityPutCostImportProcess implements ImportProcess<ActivityPutCostImportVo> {

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    @Autowired
    private ActivityPutCostService activityPutCostService;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private CustomerVoService customerVoService;
    @Autowired(required = false)
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    private static final String MDM_BUSINESS_FORMAT = "mdm_business_format";

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, ActivityPutCostImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {

        try {
            log.info("开始数据校验-----------------");
            List<ActivityPutCostDto> importList = this.validate(data);
            log.info("开始执行大批量保存-----------------");
            //大批量保存
            activityPutCostService.bulkImportSave(importList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }


        return null;
    }

    /**
     * 校验数据
     *
     * @param data data
     * @return List<ActivityPutCostDto>
     **/
    private List<ActivityPutCostDto> validate(LinkedHashMap<Integer, ActivityPutCostImportVo> data) {
        List<ActivityPutCostImportVo> importsVos = new ArrayList<>(data.values());
        Set<String> cusCodes = new HashSet<>();
        List<String> actFormCodes = new ArrayList<>();
        Set<String> orgCodeSet = new HashSet<>();
        Set<String> channelCodeSet = new HashSet<>();
        for (ActivityPutCostImportVo vo : importsVos) {
            this.validateData(vo);
            cusCodes.add(vo.getCustomerCode());
            actFormCodes.add(vo.getActivityFormCode());
            orgCodeSet.add(vo.getSalesInstitutionCode());
            channelCodeSet.add(vo.getChannelCode());
        }
        // 查询数据字典
        Map<String, List<CustomerVo>> customerVoMap;
        Map<String, List<DictDataVo>> dictDataMap = dictDataVoService.findByDictTypeCodeList(Arrays.asList(MDM_BUSINESS_FORMAT, ActivityPutCostConstant.ACTIVITY_PUT_COST_PLATFORM_DICT));
        //获取客户信息
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(cusCodes));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        } else {
            throw new IllegalArgumentException("未找到任意客户，请检查！");
        }
        //获取活动形式信息
        List<List<String>> actFormList;
        if (!CollectionUtils.isEmpty(actFormCodes)) {
            actFormList = Lists.partition(actFormCodes, 500);
        } else {
            actFormList = new ArrayList<>();
        }
        Map<String, String> formMap = activityFormService.findFormByCodes(actFormList);
        //销售机构
        Map<String, SalesOrgVo> orgVoMap;
        List<SalesOrgVo> orgVos = salesOrgVoService.findBySalesOrgCodes(new ArrayList<>(orgCodeSet));
        if (!CollectionUtils.isEmpty(orgVos)) {
            orgVoMap = orgVos.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity()));
        } else {
            throw new IllegalArgumentException("未找到任意销售机构，请检查！");
        }
        //查询渠道
        Map<String, CustomerChannelVo> channelVoMap;
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (!CollectionUtils.isEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(e -> e.getCustomerChannelCode(), Function.identity()));
        } else {
            throw new IllegalArgumentException("未找到任意渠道，请检查！");
        }

        importsVos.forEach(importsVo -> {
            this.validateCorrect(importsVo, customerVoMap, dictDataMap, formMap, orgVoMap, channelVoMap);
        });
        Collection<ActivityPutCostDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(importsVos, ActivityPutCostImportVo.class, ActivityPutCostDto.class, LinkedHashSet.class, ArrayList.class);
        return new ArrayList<>(dtoList);
    }


    /**
     * 校验数据
     *
     * @param dto 导入数据
     **/
    private void validateData(ActivityPutCostImportVo dto) {
        Validate.isTrue(StringUtils.isNotBlank(dto.getPutCostDateStr()), "日期不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getPlatformCode()), "平台不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getCustomerCode()), "客户编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getCostItem()), "费用项不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getActivityFormCode()), "活动形式编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getChannelName()), "渠道不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getSalesInstitutionCode()), "销售机构编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getBusinessFormatCode()), "业态不能为空");
        Validate.notNull(dto.getCostAmount(), "花费金额不能为空");
        Validate.notNull(dto.getOrderAmount(), "订单金额不能为空");
    }

    /**
     * 校验数据正确性
     *
     * @param dto
     * @param customerVoMap
     * @param dictDataMap
     * @param formMap
     * @param orgVoMap
     */
    private void validateCorrect(ActivityPutCostImportVo dto, Map<String, List<CustomerVo>> customerVoMap,
                                 Map<String, List<DictDataVo>> dictDataMap, Map<String, String> formMap, Map<String, SalesOrgVo> orgVoMap,
                                 Map<String, CustomerChannelVo> channelVoMap) {

        try {
            dto.setPutCostDate(DateUtil.date_yyyy_MM_dd.parse(dto.getPutCostDateStr()));
        } catch (Exception e) {
            throw new IllegalArgumentException("年月格式错误【yyyy-MM-dd】");
        }
        //平台
        DictDataVo platform = dictDataMap.get(ActivityPutCostConstant.ACTIVITY_PUT_COST_PLATFORM_DICT).stream().filter(e -> e.getDictValue().equals(dto.getPlatformCode())).findFirst().orElse(null);
        if (platform != null) {
            dto.setPlatformCode(platform.getDictCode());
        } else {
            throw new IllegalArgumentException("平台【" + dto.getPlatformCode() + "】错误，请检查！");
        }
        //业态
        DictDataVo promotionType = dictDataMap.get(MDM_BUSINESS_FORMAT).stream().filter(e -> e.getDictValue().equals(dto.getBusinessFormatCode())).findFirst().orElse(null);
        if (promotionType != null) {
            dto.setBusinessFormatCode(promotionType.getDictCode());
        } else {
            throw new IllegalArgumentException("业态【" + dto.getBusinessFormatCode() + "】错误，请检查！");
        }
        //查询渠道
        if (channelVoMap.containsKey(dto.getChannelCode())) {
            dto.setChannelName(channelVoMap.get(dto.getChannelCode()).getCustomerChannelName());
        } else {
            throw new IllegalArgumentException("渠道编码【" + dto.getChannelCode() + "】错误，请检查！");
        }
        //活动形式
        if (formMap.containsKey(dto.getActivityFormCode())) {
            dto.setActivityFormName(formMap.get(dto.getActivityFormCode()));
        } else {
            Validate.isTrue(false, "根据活动形式编码未能查找到活动形式信息");
        }
        if (BigDecimal.ZERO.compareTo(dto.getOrderAmount()) == 0 || BigDecimal.ZERO.compareTo(dto.getCostAmount()) == 0) {
            dto.setRoi(BigDecimal.ZERO);
        } else {
            dto.setRoi(dto.getOrderAmount().multiply(new BigDecimal(100)).divide(dto.getCostAmount(), 2));
        }
        //销售机构
        if (orgVoMap.containsKey(dto.getSalesInstitutionCode())) {
            dto.setSalesInstitutionName(orgVoMap.get(dto.getSalesInstitutionCode()).getSalesOrgName());
        } else {
            throw new IllegalArgumentException("销售机构编码【" + dto.getSalesInstitutionCode() + "】错误，请检查！");
        }
        //客户
        if (customerVoMap.containsKey(dto.getCustomerCode())) {
            List<CustomerVo> list = customerVoMap.get(dto.getCustomerCode());
            CustomerVo customerVo = list.stream().filter(e -> dto.getChannelCode().equals(e.getCustomerChannelCode()) && dto.getSalesInstitutionCode().equals(e.getSalesInstitutionCode()) && dto.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                    .findFirst().orElse(null);
            if (customerVo == null) {
                throw new IllegalArgumentException("客户编码【" + dto.getCustomerCode() + "】未找到对应客户，请检查！");
            }
            dto.setCustomerName(customerVo.getCustomerName());
        } else {
            throw new IllegalArgumentException("客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
        }
        dto.setRoiStr(dto.getRoi().toPlainString() + "%");
    }


    /**
     * 获取数据实体
     *
     * @return Class<ActivityPutCostImportVo>
     */
    @Override
    public Class<ActivityPutCostImportVo> findCrmExcelVoClass() {
        return ActivityPutCostImportVo.class;
    }

    /**
     * 获取数据实体
     *
     * @return String
     */
    @Override
    public String getTemplateCode() {
        return "TPM_ACTIVITY_PUT_COST_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return String
     */
    @Override
    public String getTemplateName() {
        return "TPM-电商活动投放费用导入模板";
    }
}
