package com.biz.crm.tpm.business.agency.operation.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.agency.operation.local.entity.AgencyOperationFee;
import com.biz.crm.tpm.business.agency.operation.local.entity.AgencyOperationGoal;
import com.biz.crm.tpm.business.agency.operation.local.entity.AgencyOperationServiceWeight;
import com.biz.crm.tpm.business.agency.operation.local.repository.AgencyOperationFeeRepository;
import com.biz.crm.tpm.business.agency.operation.local.repository.AgencyOperationGoalRepository;
import com.biz.crm.tpm.business.agency.operation.local.repository.AgencyOperationServiceWeightRepository;
import com.biz.crm.tpm.business.agency.operation.sdk.constant.AgencyOperationConstant;
import com.biz.crm.tpm.business.agency.operation.sdk.dto.AgencyOperationFeeDto;
import com.biz.crm.tpm.business.agency.operation.sdk.enums.AgencyOperationEnum;
import com.biz.crm.tpm.business.agency.operation.sdk.service.AgencyOperationFeeService;
import com.biz.crm.tpm.business.agency.operation.sdk.vo.AgencyOperationFeeTableVo;
import com.biz.crm.tpm.business.agency.operation.sdk.vo.AgencyOperationFeeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.*;

/**
 * 代运营费用表实现类
 *
 * @author: yaoyongming
 * @date: 2023/8/23 16:29
 */
@Slf4j
@Service("agencyOperationFeeService")
public class AgencyOperationFeeServiceImpl implements AgencyOperationFeeService {

    @Autowired(required = false)
    AgencyOperationGoalRepository agencyOperationGoalRepository;

    @Autowired(required = false)
    AgencyOperationServiceWeightRepository agencyOperationServiceWeightRepository;

    @Autowired(required = false)
    AgencyOperationFeeRepository agencyOperationFeeRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 新增
     *
     * @param dto
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(AgencyOperationFeeDto dto) {
        validate(dto);
        validateUnique(dto);
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        List<AgencyOperationFeeTableVo> tableVoList = calculationFee(dto);
        if (!CollectionUtils.isEmpty(tableVoList)) {
            dto.setTotal(tableVoList.get(0).getTotal());
        }

        AgencyOperationFee entity = nebulaToolkitService.copyObjectByWhiteList(dto, AgencyOperationFee.class, HashSet.class, ArrayList.class);
        agencyOperationFeeRepository.save(entity);
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id
     * @return
     */
    @Override
    public AgencyOperationFeeVo findById(String id) {
        AgencyOperationFee entity = agencyOperationFeeRepository.getById(id);
        Validate.notNull(entity, "未找到对应的数据");
        return nebulaToolkitService.copyObjectByWhiteList(entity, AgencyOperationFeeVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 修改
     *
     * @param dto
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(AgencyOperationFeeDto dto) {
        Validate.notBlank(dto.getId(), "id不能为空");
        validate(dto);
        validateUnique(dto);
        dto.setTenantCode(TenantUtils.getTenantCode());
        List<AgencyOperationFeeTableVo> tableVoList = calculationFee(dto);
        if (!CollectionUtils.isEmpty(tableVoList)) {
            dto.setTotal(tableVoList.get(0).getTotal());
        }

        AgencyOperationFee entity = nebulaToolkitService.copyObjectByWhiteList(dto, AgencyOperationFee.class, HashSet.class, ArrayList.class);
        agencyOperationFeeRepository.saveOrUpdate(entity);
    }

    /**
     * 校验
     *
     * @param dto
     */
    public void validate(AgencyOperationFeeDto dto) {
        Validate.notBlank(dto.getPlatformCode(), "平台，必填！");
        Validate.notBlank(dto.getBusinessFormatCode(), "业态，必填！");
        Validate.notBlank(dto.getChannelCode(), "渠道，必填！");
        Validate.notBlank(dto.getSalesInstitutionCode(), "销售机构，必填！");
        Validate.notBlank(dto.getCustomerCode(), "客户，必填！");
        Validate.notBlank(dto.getCustomerErpCode(), "客户ERP，必填！");
        Validate.notBlank(dto.getSupplierCode(), "供应商，必填！");
        Validate.notBlank(dto.getYearMonthLy(), "年月，必填！");
    }

    /**
     * 唯一教育
     *
     * @param dto
     */
    public void validateUnique(AgencyOperationFeeDto dto) {
        StringBuffer sb = new StringBuffer();
        sb.append(dto.getPlatformCode());
        sb.append(dto.getBusinessFormatCode());
        sb.append(dto.getChannelCode());
        sb.append(dto.getSalesInstitutionErpCode());
        sb.append(dto.getCustomerErpCode());
        sb.append(dto.getSupplierCode());
        sb.append(dto.getYearMonthLy());
        dto.setUniqueKey(sb.toString());

        List<AgencyOperationFee> entities = agencyOperationFeeRepository.findByUniqueKey(dto.getUniqueKey(), dto.getId());
        Validate.isTrue(CollectionUtils.isEmpty(entities), "系统中存在重复数据，请检查");
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        agencyOperationFeeRepository.removeByIds(idList);
    }

    /**
     * 获取实际展示字段
     *
     * @param platformCode
     * @return
     */
    @Override
    public LinkedHashMap<String, String> showFields(String platformCode) {
        LinkedHashMap<String, String> fields = new LinkedHashMap<>();
        if (AgencyOperationConstant.TMALL.equals(platformCode)) {
            fields.put(AgencyOperationEnum.BRIEF.getFieldCoding(), AgencyOperationEnum.BRIEF.getFieldName());
            fields.put(AgencyOperationEnum.WORK_ACCURACY.getFieldCoding(), AgencyOperationEnum.WORK_ACCURACY.getFieldName());
            fields.put(AgencyOperationEnum.COMPLETION_TIME_POINT.getFieldCoding(), AgencyOperationEnum.COMPLETION_TIME_POINT.getFieldName());
            fields.put(AgencyOperationEnum.VIP_PROPORTION.getFieldCoding(), AgencyOperationEnum.VIP_PROPORTION.getFieldName());
            fields.put(AgencyOperationEnum.REPURCHASE_RATE.getFieldCoding(), AgencyOperationEnum.REPURCHASE_RATE.getFieldName());
            fields.put(AgencyOperationEnum.NEW_PRODUCT.getFieldCoding(), AgencyOperationEnum.NEW_PRODUCT.getFieldName());
            fields.put(AgencyOperationEnum.NEW_VIP.getFieldCoding(), AgencyOperationEnum.NEW_VIP.getFieldName());
            fields.put(AgencyOperationEnum.STORE_DISCOUNT.getFieldCoding(), AgencyOperationEnum.STORE_DISCOUNT.getFieldName());
            fields.put(AgencyOperationEnum.MARKETING_PLANNING.getFieldCoding(), AgencyOperationEnum.MARKETING_PLANNING.getFieldName());
        } else if (AgencyOperationConstant.JINGDONG2.equals(platformCode)) {
            fields.put(AgencyOperationEnum.BRIEF.getFieldCoding(), AgencyOperationEnum.BRIEF.getFieldName());
            fields.put(AgencyOperationEnum.WORK_ACCURACY.getFieldCoding(), AgencyOperationEnum.WORK_ACCURACY.getFieldName());
            fields.put(AgencyOperationEnum.VIP_PROPORTION.getFieldCoding(), AgencyOperationEnum.VIP_PROPORTION.getFieldName());
            fields.put(AgencyOperationEnum.STORE_DISCOUNT.getFieldCoding(), AgencyOperationEnum.STORE_DISCOUNT.getFieldName());
            fields.put(AgencyOperationEnum.MARKETING_PLANNING.getFieldCoding(), AgencyOperationEnum.MARKETING_PLANNING.getFieldName());
            fields.put(AgencyOperationEnum.MARKET_SHARE.getFieldCoding(), AgencyOperationEnum.MARKET_SHARE.getFieldName());
            fields.put(AgencyOperationEnum.NEW_PRODUCT_ACHIEVE.getFieldCoding(), AgencyOperationEnum.NEW_PRODUCT_ACHIEVE.getFieldName());
            fields.put(AgencyOperationEnum.NET_INCOME_ACHIEVE.getFieldCoding(), AgencyOperationEnum.NET_INCOME_ACHIEVE.getFieldName());
            fields.put(AgencyOperationEnum.INVENTORY_TURNOVER_RATE.getFieldCoding(), AgencyOperationEnum.INVENTORY_TURNOVER_RATE.getFieldName());

        } else if (AgencyOperationConstant.PINDUODUO.equals(platformCode)) {
            fields.put(AgencyOperationEnum.GMV.getFieldCoding(), AgencyOperationEnum.GMV.getFieldName());
            fields.put(AgencyOperationEnum.PAYMENT_CONVERSION_RATE.getFieldCoding(), AgencyOperationEnum.PAYMENT_CONVERSION_RATE.getFieldName());
            fields.put(AgencyOperationEnum.PROMOTE_ROI.getFieldCoding(), AgencyOperationEnum.PROMOTE_ROI.getFieldName());
            fields.put(AgencyOperationEnum.VIP.getFieldCoding(), AgencyOperationEnum.VIP.getFieldName());
            fields.put(AgencyOperationEnum.DISCOUNT_RATE.getFieldCoding(), AgencyOperationEnum.DISCOUNT_RATE.getFieldName());
            fields.put(AgencyOperationEnum.BRIEF.getFieldCoding(), AgencyOperationEnum.BRIEF.getFieldName());
            fields.put(AgencyOperationEnum.WORK_ACCURACY.getFieldCoding(), AgencyOperationEnum.WORK_ACCURACY.getFieldName());
        }
        return fields;
    }

    /**
     * 费用计算
     *
     * @param dto
     * @return
     */
    @Override
    public List<AgencyOperationFeeTableVo> calculationFee(AgencyOperationFeeDto dto) {
        List<AgencyOperationFeeTableVo> list = new ArrayList<>();

        if (!AgencyOperationConstant.TMALL.equals(dto.getPlatformCode()) &&
                !AgencyOperationConstant.JINGDONG2.equals(dto.getPlatformCode()) &&
                !AgencyOperationConstant.PINDUODUO.equals(dto.getPlatformCode())) {
            return list;
        }
        validate(dto);
        StringBuffer sb = new StringBuffer();
        sb.append(dto.getBusinessFormatCode());
        sb.append(dto.getChannelCode());
        sb.append(dto.getSalesInstitutionErpCode());
        sb.append(dto.getCustomerErpCode());
        sb.append(dto.getSupplierCode());
        sb.append(dto.getYearMonthLy());
        String uniqueKey = sb.toString();

        List<AgencyOperationServiceWeight> swList = agencyOperationServiceWeightRepository.findByUniqueKey(Arrays.asList(uniqueKey));
        Validate.notEmpty(swList, "未找到对应服务费&权重表的数据");
        List<AgencyOperationGoal> gList = agencyOperationGoalRepository.findByUniqueKey(Arrays.asList(uniqueKey));
        Validate.notEmpty(gList, "未找到对应目标表的数据");
        AgencyOperationServiceWeight sw = swList.get(0);
        AgencyOperationGoal g = gList.get(0);

        Map<String, Object> swMap = fieldToMap(sw);
        Map<String, Object> gMap = fieldToMap(g);
        Map<String, Object> fMap = fieldToMap(dto);
        LinkedHashMap<String, String> fields = showFields(dto.getPlatformCode());
        for (Map.Entry<String, String> entry : fields.entrySet()) {
            String fieldCode = entry.getKey();
            AgencyOperationFeeTableVo tableVo = new AgencyOperationFeeTableVo();
            tableVo.setProjectName(entry.getValue());
            tableVo.setMonthService(sw.getServiceFee());
            tableVo.setWeight(objToBD(swMap.get(fieldCode)));
            tableVo.setGoal(objToBD(gMap.get(fieldCode)));
            tableVo.setActual(objToBD(fMap.get(fieldCode)));
            Validate.isTrue(tableVo.getGoal().compareTo(BigDecimal.ZERO) != 0 , "【"+ tableVo.getProjectName() +"】，不能为空或为0");
            tableVo.setKpi(tableVo.getActual().divide(tableVo.getGoal(), 2, BigDecimal.ROUND_HALF_UP));
            //京东平台特殊处理
            if (AgencyOperationConstant.JINGDONG2.equals(dto.getPlatformCode())) {
                BigDecimal kpi = tableVo.getKpi();
                BigDecimal ratio = kpi;
                //1、KPI<0.85，系数=0；
                //得分=100*权重*系数
                //2、0.85=<KPI<=1.00，系数=1.00
                //得分=100*权重*系数
                //3、1.00<KPI<=1.05，系数=1.05；
                //得分=100*权重*系数
                //4、1.05<KPI<=1.10，系数=1.10；
                //得分=100*权重*系数
                //5、1.10<KPI<=1.15，系数=1.15；
                //得分=100*权重*系数
                //6、1.15<KPI<=1.20，系数=1.20；
                //得分=100*权重*系数
                //7、1.20<KPI，系数=1.25；
                //得分=100*权重*系数
                if (AgencyOperationEnum.MARKET_SHARE.getFieldCoding().equals(fieldCode) || AgencyOperationEnum.NET_INCOME_ACHIEVE.getFieldCoding().equals(fieldCode)) {
                    if (kpi.compareTo(new BigDecimal("0.85")) < 0) {
                        ratio = BigDecimal.ZERO;
                    } else if (kpi.compareTo(new BigDecimal("0.85")) >= 0 && kpi.compareTo(BigDecimal.ONE) <= 0) {
                        ratio = BigDecimal.ONE;
                    } else if (kpi.compareTo(BigDecimal.ONE) > 0 && kpi.compareTo(new BigDecimal("1.05")) <= 0) {
                        ratio = new BigDecimal("1.05");
                    } else if (kpi.compareTo(new BigDecimal("1.05")) > 0 && kpi.compareTo(new BigDecimal("1.1")) <= 0) {
                        ratio = new BigDecimal("1.1");
                    } else if (kpi.compareTo(new BigDecimal("1.1")) > 0 && kpi.compareTo(new BigDecimal("1.15")) <= 0) {
                        ratio = new BigDecimal("1.15");
                    } else if (kpi.compareTo(new BigDecimal("1.15")) > 0 && kpi.compareTo(new BigDecimal("1.2")) <= 0) {
                        ratio = new BigDecimal("1.2");
                    } else {
                        ratio = new BigDecimal("1.25");
                    }
                //1、KPI<1;
                //得分=0
                //2、KPI>=1；
                //得分=100*权重*KPI
                } else if (AgencyOperationEnum.NEW_PRODUCT_ACHIEVE.getFieldCoding().equals(fieldCode)) {
                    if (kpi.compareTo(BigDecimal.ONE) < 0) {
                        ratio = BigDecimal.ZERO;
                    }
                //1、KPI<1.00
                //得分=0;
                //2、1.00<=KPI<1.20
                //得分=100*权重*KPI
                //3、1.20<=KPI
                //得分=100*权重*1.20
                } else if (AgencyOperationEnum.INVENTORY_TURNOVER_RATE.getFieldCoding().equals(fieldCode)) {
                    if (kpi.compareTo(BigDecimal.ONE) < 0) {
                        ratio = BigDecimal.ZERO;
                    } else if (kpi.compareTo(BigDecimal.ONE) >= 0 && kpi.compareTo(new BigDecimal("1.2")) < 0) {

                    } else {
                        ratio = new BigDecimal("1.2");
                    }
                }
                tableVo.setScore(new BigDecimal(100).multiply(tableVo.getWeight()).multiply(ratio));
            } else {
                tableVo.setScore(new BigDecimal(100).multiply(tableVo.getWeight()).multiply(tableVo.getKpi()));
            }
            list.add(tableVo);
        }
        //京东：
        // 总得分<=120
        // 总费用=月固定服务费*总得分/100；
        // 总得分>120
        // 总费用=月固定服务费*1.2
        //其他：
        // 总费用=月固定服务费*总分/100
        BigDecimal feeTotal;
        BigDecimal scoreTotal = list.stream().map(e -> e.getScore()).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        if (AgencyOperationConstant.JINGDONG2.equals(dto.getPlatformCode()) && scoreTotal.compareTo(new BigDecimal("120")) > 0) {
            feeTotal = Optional.ofNullable(list.get(0).getMonthService()).orElse(BigDecimal.ZERO).multiply(new BigDecimal("1.2"));
        } else {
            feeTotal = Optional.ofNullable(list.get(0).getMonthService()).orElse(BigDecimal.ZERO).multiply(scoreTotal).divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP);
        }
        BigDecimal finalFeeTotal = feeTotal;
        list.forEach(e -> e.setTotal(finalFeeTotal));
        return list;
    }

    /**
     * 变量查询
     *
     * @param dto
     * @return
     */
    @Override
    public BigDecimal findForFormulaVariable(AgencyOperationFeeDto dto) {
        return agencyOperationFeeRepository.findForFormulaVariable(dto);
    }

    private Map<String, Object> fieldToMap(Object obj) {
        Map map = JSON.parseObject(JSON.toJSONString(obj), Map.class);
        return map;
    }

    private BigDecimal objToBD(Object obj) {
        return obj == null ? BigDecimal.ZERO : new BigDecimal(String.valueOf(obj));
    }
}
