package com.biz.crm.tpm.business.agency.operation.local.service.process;

import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mdm.business.supplier.sdk.service.SupplierVoService;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierVo;
import com.biz.crm.tpm.business.agency.operation.sdk.dto.AgencyOperationGoalDto;
import com.biz.crm.tpm.business.agency.operation.sdk.service.AgencyOperationGoalService;
import com.biz.crm.tpm.business.agency.operation.sdk.vo.AgencyOperationGoalImportVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: yaoyongming
 * @date: 2023/8/19 17:04
 */
@Slf4j
@Component
public class AgencyOperationGoalImportProcess implements ImportProcess<AgencyOperationGoalImportVo> {


    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private AgencyOperationGoalService agencyOperationGoalService;

    @Autowired(required = false)
    private SupplierVoService supplierVoService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    private static final String MDM_BUSINESS_FORMAT = "mdm_business_format";

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, AgencyOperationGoalImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        log.info("开始导入......");
        Map<Integer, String> errorMap = Maps.newHashMap();
        try {
            Validate.notEmpty(data, "导入数据不能为空！");

            List<AgencyOperationGoalDto> importList = this.validate(errorMap, data);

            if (errorMap.size() == 0){
                log.info("开始数据保存......");
                agencyOperationGoalService.importSave(importList);
                log.info("结束数据保存......");
            }
        } catch (Exception e){
            log.error(e.getMessage(),e);
            String errorMsg = e.getMessage();
            if (StringUtils.isEmpty(e.getMessage())){
                errorMsg = "数据处理失败！";
            }
            throw new IllegalArgumentException(errorMsg);
        }
        log.info("结束导入......");
        return errorMap;
    }

    List<AgencyOperationGoalDto> validate(Map<Integer, String> errorMap, LinkedHashMap<Integer, AgencyOperationGoalImportVo> data) {
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> channelCodeSet = new HashSet<>();
        Set<String> saleOrgCodeSet = new HashSet<>();
        Set<String> supplierCodeSet = new HashSet<>();
        Set<String> uniqueKeySet = new HashSet<>();
        log.info("开始非空校验......");
        //非空校验
        for (Map.Entry<Integer, AgencyOperationGoalImportVo> entry : data.entrySet()) {
            try {
                AgencyOperationGoalImportVo importsVo = entry.getValue();
                Validate.notBlank(importsVo.getBusinessFormatCode(), "业态不能为空");
                Validate.notBlank(importsVo.getChannelCode(), "渠道编码不能为空");
                Validate.notBlank(importsVo.getSalesInstitutionCode(), "销售机构编码不能为空");
                Validate.notBlank(importsVo.getCustomerCode(), "客户编码不能为空");
                Validate.notBlank(importsVo.getSupplierCode(), "供应商编码不能为空");
                Validate.notBlank(importsVo.getYearMonthLy(), "年月不能为空");

                customerCodeSet.add(importsVo.getCustomerCode());
                channelCodeSet.add(importsVo.getChannelCode());
                saleOrgCodeSet.add(importsVo.getSalesInstitutionCode());
                supplierCodeSet.add(importsVo.getSupplierCode());

                StringBuffer sb = new StringBuffer();
                sb.append(importsVo.getBusinessFormatCode());
                sb.append(importsVo.getChannelCode());
                sb.append(importsVo.getSalesInstitutionCode());
                sb.append(importsVo.getCustomerCode());
                sb.append(importsVo.getSupplierCode());
                sb.append(importsVo.getYearMonthLy());

                if (uniqueKeySet.contains(sb.toString())) {
                    throw new IllegalArgumentException("当前导入文件中存在重复数据");
                }
                uniqueKeySet.add(sb.toString());
            } catch (Exception e){
                log.error(e.getMessage(),e);
                errorMap.put(entry.getKey(),null == e.getMessage() ? "数据处理异常" : e.getMessage());
            }
        }
        log.info("结束非空校验......");

        if (!errorMap.isEmpty()) {
            return new ArrayList<>();
        }

        Map<String, List<CustomerVo>> customerMap = new HashMap<>();
        Map<String, String> channelVoMap = new HashMap<>();
        Map<String, List<SalesOrgVo>> salesMap = new HashMap<>();
        Map<String, String> supplierMap = new HashMap<>();

        log.info("开始获取基础数据......");
        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(MDM_BUSINESS_FORMAT);
        Validate.isTrue(ObjectUtils.isNotEmpty(formatMap), "未查询到业态数据字典!");
        List<SupplierVo> supplierVos = supplierVoService.findBySupplierCodes(new ArrayList<>(supplierCodeSet));
        if (CollectionUtils.isNotEmpty(supplierVos)) {
            Map<String, String> map = supplierVos.stream().collect(Collectors.toMap(SupplierVo::getSupplierCode, SupplierVo::getSupplierName, (a, b) -> a));
            supplierMap.putAll(map);
        }
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (CollectionUtils.isNotEmpty(channelList)) {
            Map<String, String> map = channelList.stream().collect(Collectors.toMap(CustomerChannelVo::getCustomerChannelCode, CustomerChannelVo::getCustomerChannelName, (a, b) -> a));
            channelVoMap.putAll(map);
        }
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (CollectionUtils.isNotEmpty(customerVos)) {
            Map<String, List<CustomerVo>> map = customerVos.stream().collect(Collectors.groupingBy(CustomerVo::getErpCode));
            customerMap.putAll(map);
        }
        List<SalesOrgVo> salesOrgVos = salesOrgVoService.findByErpCodeList(new ArrayList<>(saleOrgCodeSet));
        if (CollectionUtils.isNotEmpty(salesOrgVos)) {
            Map<String, List<SalesOrgVo>> map = salesOrgVos.stream().collect(Collectors.groupingBy(SalesOrgVo::getErpCode));
            salesMap.putAll(map);
        }
        log.info("结束获取基础数据......");


        log.info("开始封装数据......");
        List<AgencyOperationGoalDto> dtoList = new ArrayList<>();
        for (Map.Entry<Integer, AgencyOperationGoalImportVo> entry : data.entrySet()) {
            try {
                AgencyOperationGoalImportVo importsVo = entry.getValue();
                if (formatMap.containsKey(importsVo.getBusinessFormatCode())) {
                    importsVo.setBusinessFormatCode(formatMap.get(importsVo.getBusinessFormatCode()));
                } else {
                    throw new IllegalArgumentException("业态未能识别");
                }

                if (channelVoMap.containsKey(importsVo.getChannelCode())) {
                    importsVo.setChannelName(channelVoMap.get(importsVo.getChannelCode()));
                } else {
                    throw new IllegalArgumentException("渠道未能识别");
                }

                if (supplierMap.containsKey(importsVo.getSupplierCode())) {
                    importsVo.setSupplierName(supplierMap.get(importsVo.getSupplierCode()));
                } else {
                    throw new IllegalArgumentException("供应商编码【"+ importsVo.getSupplierCode() +"】错误，请检查！");
                }

                List<SalesOrgVo> salesOrgVoList = salesMap.get(importsVo.getSalesInstitutionCode());
                Validate.isTrue(CollectionUtils.isNotEmpty(salesOrgVoList) && salesOrgVoList.size() > 0, "销售组织编码【" + importsVo.getSalesInstitutionCode() + "】错误，请检查！");
                SalesOrgVo salesOrgVo = salesOrgVoList.stream().filter(e -> importsVo.getChannelCode().equals(e.getChannelCode()) && importsVo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                Validate.notNull(salesOrgVo, "销售组织编码【" + importsVo.getSalesInstitutionCode() + "】错误，请检查！");
                importsVo.setSalesInstitutionName(salesOrgVo.getSalesOrgName());

                List<CustomerVo> baseByCustomerCodes = customerMap.get(importsVo.getCustomerCode());
                Validate.isTrue(CollectionUtils.isNotEmpty(baseByCustomerCodes), "客户编码【" + importsVo.getCustomerCode() + "】错误，请检查！");
                CustomerVo customerVo = baseByCustomerCodes.stream().filter(e -> importsVo.getChannelCode().equals(e.getCustomerChannelCode()) && salesOrgVo.getErpCode().equals(e.getSalesInstitutionErpCode()) && importsVo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                Validate.notNull(customerVo, "客户编码【" + importsVo.getCustomerCode() + "】错误，请检查！");
                importsVo.setCustomerName(customerVo.getCustomerName());

                StringBuffer sb = new StringBuffer();
                sb.append(importsVo.getBusinessFormatCode());
                sb.append(importsVo.getChannelCode());
                sb.append(importsVo.getSalesInstitutionCode());
                sb.append(importsVo.getCustomerCode());
                sb.append(importsVo.getSupplierCode());
                sb.append(importsVo.getYearMonthLy());
                importsVo.setUniqueKey(sb.toString());

                AgencyOperationGoalDto dto = nebulaToolkitService.copyObjectByWhiteList(importsVo, AgencyOperationGoalDto.class, LinkedHashSet.class, ArrayList.class);
                dto.setTenantCode(TenantUtils.getTenantCode());
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());

                dtoList.add(dto);
            } catch (Exception e){
                log.error(e.getMessage(),e);
                errorMap.put(entry.getKey(), null == e.getMessage() ? "数据处理异常" : e.getMessage());
            }
        }
        log.info("结束封装数据......");

        return dtoList;
    }

    @Override
    public Class<AgencyOperationGoalImportVo> findCrmExcelVoClass() {
        return AgencyOperationGoalImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_AGENCY_OPERATION_GOAL_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-代运营目标导入";
    }
}
