package com.biz.crm.tpm.business.approval.collect.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.approval.collect.local.entity.MnApprovalCollect;
import com.biz.crm.tpm.business.approval.collect.local.repository.MnApprovalCollectRepository;
import com.biz.crm.tpm.business.approval.collect.sdk.constant.MnApprovalCollectConstant;
import com.biz.crm.tpm.business.approval.collect.sdk.dto.MnApprovalCollectDto;
import com.biz.crm.tpm.business.approval.collect.sdk.dto.MnApprovalCollectImageDto;
import com.biz.crm.tpm.business.approval.collect.sdk.dto.MnApprovalCollectLogEventDto;
import com.biz.crm.tpm.business.approval.collect.sdk.enums.MnApprovalCollectType;
import com.biz.crm.tpm.business.approval.collect.sdk.event.MnApprovalCollectEventListener;
import com.biz.crm.tpm.business.approval.collect.sdk.event.log.MnApprovalCollectLogEventListener;
import com.biz.crm.tpm.business.approval.collect.sdk.service.MnApprovalCollectImageVoService;
import com.biz.crm.tpm.business.approval.collect.sdk.service.MnApprovalCollectVoService;
import com.biz.crm.tpm.business.approval.collect.sdk.vo.MnApprovalCollectImageVo;
import com.biz.crm.tpm.business.approval.collect.sdk.vo.MnApprovalCollectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 核销采集信息(ApprovalCollect)表服务实现类
 *
 * @author hc
 * @since 2022-05-17 12:53:55
 */
@Slf4j
@Service("mnApprovalCollectService")
public class MnApprovalCollectServiceImpl implements MnApprovalCollectVoService {

    @Autowired(required = false)
    private MnApprovalCollectRepository mnApprovalCollectRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MnApprovalCollectImageVoService approvalCollectImageService;

    @Autowired(required = false)
    private List<MnApprovalCollectEventListener> mnApprovalCollectEventListeners;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    /**
     * 分页查询数据
     */
    @Override
    public Page<MnApprovalCollectVo> findByConditions(Pageable pageable, MnApprovalCollectDto dto) {
        if (pageable == null) {
            pageable = PageRequest.of(1, 50);
        }
        if (dto == null) {
            dto = new MnApprovalCollectDto();
        }
        if (StringUtils.isBlank(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        if (StringUtils.isBlank(dto.getDelFlag())) {
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        return this.mnApprovalCollectRepository.findByConditions(pageable, dto);
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public MnApprovalCollectVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        MnApprovalCollect approvalCollect = mnApprovalCollectRepository.getById(id);
        if (approvalCollect == null) {
            return null;
        }
        MnApprovalCollectVo result = nebulaToolkitService.copyObjectByWhiteList(approvalCollect, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
        List<MnApprovalCollectImageVo> imageVos = approvalCollectImageService.findByApprovalCollectCode(approvalCollect.getCode());
        result.setImages(imageVos);
        return result;
    }

    /**
     * 通过主键查询多条数据
     *
     * @param ids 主键集合
     * @return 多条数据
     */
    @Override
    public List<MnApprovalCollectVo> findByIds(Set<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        List<MnApprovalCollect> approvalCollects = mnApprovalCollectRepository.listByIds(ids);
        if (CollectionUtils.isEmpty(approvalCollects)) {
            return Lists.newArrayList();
        }
        return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(approvalCollects, MnApprovalCollect.class, MnApprovalCollectVo.class, HashSet.class, ArrayList.class));
    }

    /**
     * 根据主键查询详情
     */
    @Override
    public MnApprovalCollectVo findDetailsById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        MnApprovalCollect approvalCollect = mnApprovalCollectRepository.getById(id);
        if (approvalCollect == null) {
            return null;
        }
        MnApprovalCollectVo result = nebulaToolkitService.copyObjectByWhiteList(approvalCollect, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
        List<MnApprovalCollectImageVo> imageVos = approvalCollectImageService.findByApprovalCollectCode(approvalCollect.getCode());
        result.setImages(imageVos);
        return result;
    }

    /**
     * 根据编码查询信息
     */
    @Override
    public MnApprovalCollectVo findByCode(String code) {
        if (StringUtils.isBlank(code)) {
            return null;
        }
        MnApprovalCollect approvalCollect = mnApprovalCollectRepository.findByCode(code);
        if (approvalCollect == null) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(approvalCollect, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据编码查询详情
     */
    @Override
    public MnApprovalCollectVo findDetailsByCode(String code) {
        if (StringUtils.isBlank(code)) {
            return null;
        }
        MnApprovalCollect approvalCollect = mnApprovalCollectRepository.findByCode(code);
        if (approvalCollect == null) {
            return null;
        }
        MnApprovalCollectVo result = nebulaToolkitService.copyObjectByWhiteList(approvalCollect, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
        List<MnApprovalCollectImageVo> imageVos = approvalCollectImageService.findByApprovalCollectCode(approvalCollect.getCode());
        result.setImages(imageVos);
        return result;
    }

    /**
     * 通过编码集合查询数据详情
     */
    @Override
    public List<MnApprovalCollectVo> findDetailsByCodes(Set<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return Lists.newArrayList();
        }
        List<MnApprovalCollect> approvalCollects = mnApprovalCollectRepository.findByCodes(codes);
        if (CollectionUtils.isEmpty(approvalCollects)) {
            return Lists.newArrayList();
        }

        Collection<MnApprovalCollectVo> result = nebulaToolkitService.copyCollectionByWhiteList(approvalCollects, MnApprovalCollect.class, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
        for (MnApprovalCollectVo approvalCollect : result) {
            List<MnApprovalCollectImageVo> imageVos = approvalCollectImageService.findByApprovalCollectCode(approvalCollect.getCode());
            if (!CollectionUtils.isEmpty(imageVos)) {
                approvalCollect.setImages(imageVos);
            }
        }
        return Lists.newArrayList(result);
    }

    /**
     * 新增数据
     *
     * @param dto dto对象
     * @return 新增结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public MnApprovalCollectVo create(MnApprovalCollectDto dto) {
        this.createValidate(dto);
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        if (StringUtils.isBlank(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
//        String ruleCode = StringUtils.join(MnApprovalCollectConstant.MN_APPROVAL_COLLECT_FORMULA_PREFIX, DateFormatUtils.format(new Date(), "yyyyMMdd"));
        String code = this.generateCodeService.generateCode(MnApprovalCollectConstant.MN_APPROVAL_COLLECT_FORMULA_PREFIX, 1, 5, 2, TimeUnit.DAYS).get(0);
        dto.setCode(code);
        MnApprovalCollect entity = nebulaToolkitService.copyObjectByWhiteList(dto, MnApprovalCollect.class, HashSet.class, ArrayList.class, "images");
        mnApprovalCollectRepository.save(entity);

        //保存可能的图片信息
        if (!CollectionUtils.isEmpty(dto.getImages())) {
            approvalCollectImageService.create(dto, dto.getImages());
        }
        //新增业务日志
        MnApprovalCollectLogEventDto logEventDto = new MnApprovalCollectLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<MnApprovalCollectLogEventListener, MnApprovalCollectLogEventDto> onCreate =
                MnApprovalCollectLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, MnApprovalCollectLogEventListener.class, onCreate);
        return nebulaToolkitService.copyObjectByWhiteList(entity, MnApprovalCollectVo.class, HashSet.class, ArrayList.class, "images");
    }


    /**
     * 修改新据
     *
     * @param approvalCollect 实体对象
     * @return 修改结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public MnApprovalCollectVo update(MnApprovalCollectDto approvalCollect) {
        this.updateValidate(approvalCollect);
        MnApprovalCollect dbApprovalCollect = mnApprovalCollectRepository.getById(approvalCollect.getId());
        MnApprovalCollectDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(dbApprovalCollect, MnApprovalCollectDto.class, HashSet.class, ArrayList.class);
        Validate.notNull(dbApprovalCollect, "根据指定的id键值，未能获取到相应信息");
        MnApprovalCollectVo oldApprovalCollect = nebulaToolkitService.copyObjectByWhiteList(dbApprovalCollect, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
        dbApprovalCollect.setDescr(approvalCollect.getDescr());
        dbApprovalCollect.setName(approvalCollect.getName());
        dbApprovalCollect.setSortIndex(approvalCollect.getSortIndex());
        dbApprovalCollect.setType(approvalCollect.getType());
        this.mnApprovalCollectRepository.saveOrUpdate(dbApprovalCollect);

        List<MnApprovalCollectImageVo> imageVos = approvalCollectImageService.findByApprovalCollectCode(dbApprovalCollect.getCode());
        Set<String> currentIds = Sets.newHashSet();
        Set<String> dbIds = Sets.newHashSet();
        if (!CollectionUtils.isEmpty(approvalCollect.getImages())) {
            currentIds = approvalCollect.getImages().stream().map(MnApprovalCollectImageDto::getId).filter(StringUtils::isNotBlank).collect(Collectors.toSet());
        }
        if (!CollectionUtils.isEmpty(imageVos)) {
            dbIds = imageVos.stream().map(MnApprovalCollectImageVo::getId).collect(Collectors.toSet());
        }

        //处理图片删除情况
        Set<String> needDeletes = Sets.difference(dbIds, currentIds);
        List<MnApprovalCollectImageVo> oldImages = new ArrayList<>();
        if (!CollectionUtils.isEmpty(needDeletes)) {
            oldImages = approvalCollectImageService.findByIds(Lists.newArrayList(needDeletes));
            approvalCollectImageService.deleteByIds(needDeletes);
        }

        //处理图片新增情况
        List<MnApprovalCollectImageDto> needAdds = approvalCollect.getImages().stream().filter(e -> StringUtils.isBlank(e.getId())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(needAdds)) {
            approvalCollectImageService.create(approvalCollect, needAdds);
        }

        //通知活动明细，如果已关联活动细类的，需要更新相关信息
        if (!CollectionUtils.isEmpty(mnApprovalCollectEventListeners)) {
            for (MnApprovalCollectEventListener listener : mnApprovalCollectEventListeners) {
                MnApprovalCollectVo newApprovalCollect = nebulaToolkitService.copyObjectByWhiteList(dbApprovalCollect, MnApprovalCollectVo.class, HashSet.class, ArrayList.class);
                listener.onUpdate(oldApprovalCollect, newApprovalCollect);
            }
        }
        //编辑业务日志
        MnApprovalCollectLogEventDto logEventDto = new MnApprovalCollectLogEventDto();
        if (!CollectionUtils.isEmpty(oldImages)) {
            List<MnApprovalCollectImageDto> mnApprovalCollectImageDtos = (List<MnApprovalCollectImageDto>) this.nebulaToolkitService.copyCollectionByWhiteList(oldImages, MnApprovalCollectImageVo.class, MnApprovalCollectImageDto.class, HashSet.class, ArrayList.class);
            oldDto.setImages(mnApprovalCollectImageDtos);
        }
        logEventDto.setOriginal(oldDto);
        MnApprovalCollectDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(dbApprovalCollect, MnApprovalCollectDto.class, HashSet.class, ArrayList.class);
        newDto.setImages(needAdds);
        logEventDto.setNewest(newDto);
        SerializableBiConsumer<MnApprovalCollectLogEventListener, MnApprovalCollectLogEventDto> onUpdate =
                MnApprovalCollectLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, MnApprovalCollectLogEventListener.class, onUpdate);
        return this.findDetailsByCode(approvalCollect.getCode());
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(Set<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "主键集合不能为空！");
        List<MnApprovalCollectVo> mnApprovalCollectVos = this.findByIds(idList);
        Validate.notEmpty(mnApprovalCollectVos, "根据提供的主键集合信息，未能获取到相应数据");
        //通知活动明细，如果已关联活动细类的，不允许删除
        if (!CollectionUtils.isEmpty(mnApprovalCollectEventListeners)) {
            for (MnApprovalCollectEventListener listener : mnApprovalCollectEventListeners) {
                for (MnApprovalCollectVo mnApprovalCollectVo : mnApprovalCollectVos) {
                    listener.onDeleted(mnApprovalCollectVo);
                }
            }
        }
        //执行删除核销采集（主表逻辑删除，明细项信息不做变更）
        this.mnApprovalCollectRepository.delete(idList);
        //删除业务日志
        Collection<MnApprovalCollectDto> mnApprovalCollectDtos = this.nebulaToolkitService.copyCollectionByWhiteList(mnApprovalCollectVos, MnApprovalCollectVo.class, MnApprovalCollectDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<MnApprovalCollectLogEventListener, MnApprovalCollectLogEventDto> onDelete =
                MnApprovalCollectLogEventListener::onDelete;
        for (MnApprovalCollectDto mnApprovalCollectDto : mnApprovalCollectDtos) {
            MnApprovalCollectLogEventDto logEventDto = new MnApprovalCollectLogEventDto();
            logEventDto.setOriginal(mnApprovalCollectDto);
            this.nebulaNetEventClient.publish(logEventDto, MnApprovalCollectLogEventListener.class, onDelete);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateEnableStatus(Set<String> ids, String enableStatus) {
        Validate.notEmpty(ids, "主键id不能为空");
        Validate.notBlank(enableStatus, "启禁用状态不能为空");
        List<MnApprovalCollectVo> approvalCollects = this.findByIds(ids);
        Validate.notEmpty(approvalCollects, "根据指定的主键集合信息，未能获取到相应数据");
        Validate.isTrue(EnableStatusEnum.contains(enableStatus), "未知的启禁用状态");
        String tenantCode = TenantUtils.getTenantCode();
        mnApprovalCollectRepository.updateEnableStatus(approvalCollects.stream()
                .filter(e -> tenantCode.equals(e.getTenantCode()))
                .map(MnApprovalCollectVo::getId).collect(Collectors.toSet()), enableStatus);
        //更新状态业务日志
        List<MnApprovalCollectDto> mnApprovalCollectDtos = (List<MnApprovalCollectDto>) this.nebulaToolkitService.copyCollectionByWhiteList(approvalCollects, MnApprovalCollectVo.class, MnApprovalCollectDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<MnApprovalCollectLogEventListener, MnApprovalCollectLogEventDto> onUpdateEnable =
                MnApprovalCollectLogEventListener::onUpdate;
        for (MnApprovalCollectDto mnApprovalCollectDto : mnApprovalCollectDtos) {
            MnApprovalCollectLogEventDto logEventDto = new MnApprovalCollectLogEventDto();
            logEventDto.setOriginal(mnApprovalCollectDto);
            MnApprovalCollectDto newDto = new MnApprovalCollectDto();
            newDto.setEnableStatus(enableStatus);
            logEventDto.setNewest(newDto);
            this.nebulaNetEventClient.publish(logEventDto, MnApprovalCollectLogEventListener.class, onUpdateEnable);
        }
    }

    /**
     * 导入
     *
     * @param importList 对象集合
     */
    @Override
    public void importSave(List<MnApprovalCollectDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        //初始化
//        String ruleCode = StringUtils.join(MnApprovalCollectConstant.MN_APPROVAL_COLLECT_FORMULA_PREFIX, DateFormatUtils.format(new Date(), "yyyyMMdd"));
        List<String> codeList = this.generateCodeService.generateCode(MnApprovalCollectConstant.MN_APPROVAL_COLLECT_FORMULA_PREFIX, importList.size(), 5, 2, TimeUnit.DAYS);
        Collection<MnApprovalCollect> entities = this.nebulaToolkitService.copyCollectionByWhiteList(importList, MnApprovalCollectDto.class, MnApprovalCollect.class, LinkedHashSet.class, ArrayList.class);
        List<MnApprovalCollect> entityList = Lists.newArrayList(entities);
        String tenantCode = TenantUtils.getTenantCode();
        for (int i = 0; i < importList.size(); i++) {
            MnApprovalCollect e = entityList.get(i);
            e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            e.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            e.setTenantCode(tenantCode);
            e.setCode(codeList.get(i));
        }
        //保存
        this.mnApprovalCollectRepository.saveBatch(entities);
        // 业务日志创建
        importList.forEach(dto -> {
            MnApprovalCollectLogEventDto logEventDto = new MnApprovalCollectLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<MnApprovalCollectLogEventListener, MnApprovalCollectLogEventDto> onCreate =
                    MnApprovalCollectLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, MnApprovalCollectLogEventListener.class, onCreate);
        });

    }

    @Override
    public List<MnApprovalCollectVo> getList(MnApprovalCollectDto approvalCollect) {

        String tenantCode = TenantUtils.getTenantCode();

        List<MnApprovalCollect> list = this.mnApprovalCollectRepository.lambdaQuery()
                .eq(MnApprovalCollect::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MnApprovalCollect::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .eq(MnApprovalCollect::getTenantCode, tenantCode)
                .eq(StringUtils.isNotEmpty(approvalCollect.getBusinessFormatCode()), MnApprovalCollect::getBusinessFormatCode, approvalCollect.getBusinessFormatCode())
                .in(!CollectionUtils.isEmpty(approvalCollect.getBusinessUnitCode()), MnApprovalCollect::getBusinessUnitCode, approvalCollect.getBusinessUnitCode())
                .eq(StringUtils.isNotEmpty(approvalCollect.getType()), MnApprovalCollect::getType, approvalCollect.getType()).list();

        Collection<MnApprovalCollectVo> mnApprovalCollectVos = this.nebulaToolkitService.copyCollectionByBlankList(Optional.of(list).orElse(new ArrayList<>()), MnApprovalCollect.class, MnApprovalCollectVo.class, LinkedHashSet.class, ArrayList.class);
        return (List<MnApprovalCollectVo>) mnApprovalCollectVos;
    }

    /**
     * 创建验证
     */
    private void createValidate(MnApprovalCollectDto approvalCollect) {
        Validate.notNull(approvalCollect, "对象信息不能为空！");
        Validate.isTrue(StringUtils.isBlank(approvalCollect.getId()), "主键id不能有值");
        approvalCollect.setId(null);
        Validate.notBlank(approvalCollect.getName(), "名称不能为空");
        Validate.notBlank(approvalCollect.getType(), "核销采集信息类型不能为空");
        Validate.notNull(MnApprovalCollectType.findByCode(approvalCollect.getType()), "未知的核销采集类型");
        Validate.notNull(approvalCollect.getSortIndex(), "排序值不能为空");
    }

    /**
     * 修改验证
     */
    private void updateValidate(MnApprovalCollectDto approvalCollect) {
        Validate.notNull(approvalCollect, "对象信息不能为空！");
        Validate.notBlank(approvalCollect.getId(), "主键id不能为空！");
        Validate.notBlank(approvalCollect.getName(), "名称不能为空");
        Validate.notBlank(approvalCollect.getType(), "核销采集信息类型不能为空");
        Validate.notNull(MnApprovalCollectType.findByCode(approvalCollect.getType()), "未知的核销采集类型");
        Validate.notNull(approvalCollect.getSortIndex(), "排序值不能为空");
    }
}

