package com.biz.crm.tpm.business.audit.fee.local.helper;

import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.ledger.AuditFeeDiffLedgerDto;
import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @Description 费用差异台帐助手
 * @Author YangWei
 * @Date 2023/3/24 下午2:00
 */
@Slf4j
@Component
public class AuditFeeDiffLedgerHelper {

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 锁前缀
     */
    private static final String LOCK_PREFIX = "audit_fee_diff_ledger:lock_code:";

    /**
     * 创建费用差异台帐非空校验
     * @param dto
     */
    public void verifyEmpty(AuditFeeDiffLedgerDto dto) {
        Assert.notNull(dto, "参数不能为空");
        Assert.hasText(dto.getAuditFeeCheckCode(), "核销费用核对表扣费匹配单号不能为空");
        Assert.hasText(dto.getBusinessFormatCode(), "业态不能为空");
        Assert.hasText(dto.getBusinessUnitCode(), "业务单元不能为空");
        Assert.hasText(dto.getYear(), "年份不能为空");
        Assert.hasText(dto.getMonth(), "月份不能为空");
        Assert.hasText(dto.getFeeYearMonth(), "年月不能为空");
        Assert.notNull(dto.getDiffAmount(), "差异金额不能为空");
    }

    /**
     * 批量加锁
     * @param feeDiffLedgerCodes
     * @param seconds
     * @param time
     * @return {@link boolean}
     */
    public boolean batchTryLock(List<String> codes, TimeUnit timeUnit, int time) {
        Assert.notEmpty(codes, "加锁编码不能为空");
        if (ObjectUtils.isEmpty(timeUnit)) {
            timeUnit = TimeUnit.MILLISECONDS;
        }
        if (time <= 0) {
            time = BudgetLockConstant.DEFAULT_LOCK_TIME;
        }
        boolean isLock = true;
        List<String> successKeys = new ArrayList<>();
        try {
            // 循环加锁，并记录成功的key
            for (String code : codes) {
                isLock = this.redisLockService.tryLock(LOCK_PREFIX + code, timeUnit, time);
                if (!isLock) {
                    return false;
                }
                successKeys.add(code);
            }
        } finally {
            // 存在加锁失败的情况，则先将成功的解锁
            if (!isLock && !CollectionUtils.isEmpty(successKeys)) {
                successKeys.forEach(key -> {
                    redisLockService.unlock(LOCK_PREFIX + key);
                });
            }
        }
        return true;
    }

    /**
     * 批量释放锁
     * @param feeDiffLedgerCodes
     */
    public void batchUnLock(List<String> codes) {
        Assert.notEmpty(codes, "加锁编码不能为空");
        codes.forEach(code -> {
            redisLockService.unlock(LOCK_PREFIX + code);
        });
    }
}