package com.biz.crm.tpm.business.audit.fee.local.listener;

import cn.hutool.core.lang.Snowflake;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.rocketmq.event.RocketMqProducerEvent;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeDiffTrackConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.AuditFeeDiffTrackPassMqTagEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.service.ledger.AuditFeeDiffLedgerLockService;
import com.biz.crm.tpm.business.audit.fee.sdk.service.track.AuditFeeDiffTrackVoService;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.listener.ProcessCompleteListener;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * 差异费用追踪审批通过回调
 * @author wanghaojia
 * @date 2022/11/30 20:57
 */
@Slf4j
@Component
public class AuditFeeDIffTrackProcessCompleteListener implements ProcessCompleteListener {
    @Autowired(required = false)
    private AuditFeeDiffTrackVoService auditFeeDiffTrackVoService;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Resource
    private ApplicationEventPublisher eventPublisher;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisService redisService;
    @Autowired(required = false)
    private AuditFeeDiffLedgerLockService auditFeeDiffLedgerLockService;

    @Override
    public String getBusinessCode() {
        return AuditFeeDiffTrackConstants.AUDIT_FEE_DIFF_TRACK;
    }

    @Override
    public void onProcessComplete(ProcessStatusDto dto) {
        log.info("差异费用追踪审批结束,{}",JSON.toJSONString(dto));
        if (StringUtils.isEmpty(dto.getBusinessNo())){
            return;
        }
        String operateId = new Snowflake().nextIdStr();
        try {
            //多策略提交
            String processStatus = dto.getProcessStatus();
            Validate.notEmpty(dto.getProcessNo(), "流程编码不能为空！");
            //审批通过
            if (ProcessStatusEnum.PASS.getDictCode().equals(processStatus)) {
                auditFeeDiffTrackVoService.processPass(dto);
                // 推送细案预测 推送牛人管家
                this.passPushMq(dto.getProcessNo(), dto.getBusinessNoList());
            }
            //审批驳回|流程追回
            if (ProcessStatusEnum.REJECT.getDictCode().equals(processStatus) || ProcessStatusEnum.RECOVER.getDictCode().equals(processStatus)) {
                log.info("追踪驳回啦～～");
                auditFeeDiffTrackVoService.processRejectAndRecover(dto,operateId);
            }
        }finally {
            String key = String.format(AuditFeeDiffTrackConstants.AUDIT_FEE_DIFF_TRACK_OPERATE,operateId);
            if(this.redisService.hasKey(key)){
                this.auditFeeDiffLedgerLockService.unlock((List<String>) this.redisService.get(key));
                this.redisService.del(key);
            }
        }
    }

    /**
     * 审批通过推送mq处理
     * @param processNo 流程编号
     * @param businessNoList 业务编码集合
     */
    private void passPushMq(String processNo,List<String> businessNoList){
        log.info("差异费用追踪审批通过推送MQ{}",JSON.toJSONString(businessNoList));
        if (CollectionUtils.isEmpty(businessNoList)){
            return;
        }

        String uuid = UUID.randomUUID().toString().replace("-", "");
        String accountJson = "";
        String currentAccount = "";
        if (loginUserService != null) {
            currentAccount = loginUserService.getLoginAccountName();
            AbstractCrmUserIdentity userIdentity = loginUserService.getAbstractLoginUser();
            accountJson = Objects.isNull(userIdentity) ? null : JSON.toJSONString(userIdentity);
        }
        // 分枚举发送到不同目标
        for (AuditFeeDiffTrackPassMqTagEnum mqTagEnum : AuditFeeDiffTrackPassMqTagEnum.values()) {
            RocketMqProducerEvent rocketMqProducerEvent = new RocketMqProducerEvent(this, currentAccount,  accountJson, false,
                uuid, AuditFeeDiffTrackConstants.TPM_AUDIT_FEE_DIFF_TRACK_PROCESS_PASS_TOPIC + RocketMqUtil.mqEnvironment(),
                    mqTagEnum.getCode(), JSON.toJSONString(businessNoList));
            eventPublisher.publishEvent(rocketMqProducerEvent);
        }
    }

}
