package com.biz.crm.tpm.business.audit.fee.local.service.imports;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheckPos;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckPosRepository;
import com.biz.crm.tpm.business.audit.fee.local.service.AuditFeeCheckPosService;
import com.biz.crm.tpm.business.audit.fee.local.service.imports.vo.AuditFeeCheckPosImportVo;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.check.AuditFeeCheckPosDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.prediction.AuditFeePredictionDto;
import com.biz.crm.tpm.business.audit.fee.sdk.service.check.AuditFeeCheckDetailPlanVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.check.AuditFeeCheckDetailPlanVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * POS活动核对批量导入调整
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-12-29 21:29
 */
@Component
@Slf4j
public class AuditFeeCheckPosImportProcess implements ImportProcess<AuditFeeCheckPosImportVo> {


    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private AuditFeeCheckPosService auditFeeCheckPosService;

    @Autowired(required = false)
    private AuditFeeCheckPosRepository auditFeeCheckPosRepository;

    @Autowired(required = false)
    private AuditFeeCheckDetailPlanVoService auditFeeCheckDetailPlanVoService;

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<Integer, String> execute(LinkedHashMap<Integer, AuditFeeCheckPosImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errorMap = Maps.newHashMap();
        Assert.notEmpty(data, "导入的数据为空");
        Assert.notEmpty(data.values(), "导入的数据为空");
        try {
            this.baseVal(data, errorMap);
            if (CollectionUtil.isNotEmpty(errorMap)) {
                return errorMap;
            }
            this.dataVal(data, errorMap);
            if (CollectionUtil.isEmpty(errorMap)) {
                this.saveData(new ArrayList<>(data.values()));
            }
        } catch (Exception e) {
            log.error("POS活动核对批量导入调整");
            log.error("", e);
            throw new IllegalArgumentException(e.getMessage());
        }
        return errorMap;
    }

    /**
     * 保存数据
     *
     * @param auditFeeCheckPosImportVos
     */
    private void saveData(ArrayList<AuditFeeCheckPosImportVo> auditFeeCheckPosImportVos) {
        List<List<AuditFeeCheckPosImportVo>> posImportVosGroupList = Lists.partition(auditFeeCheckPosImportVos, 400);
        Map<String, String> codeMap = auditFeeCheckPosImportVos.stream()
                .collect(Collectors.toMap(AuditFeeCheckPosImportVo::getMatchCode, AuditFeeCheckPosImportVo::getActivityDetailItemCode, (n, o) -> n));
        posImportVosGroupList.forEach(list -> {
            List<String> matchCodeList = list.stream().map(AuditFeeCheckPosImportVo::getMatchCode)
                    .distinct().collect(Collectors.toList());
            List<AuditFeeCheckPos> checkPosList = auditFeeCheckPosRepository.findByMatchCodeList(matchCodeList);
            if (CollectionUtil.isEmpty(checkPosList)) {
                return;
            }
            List<AuditFeeCheckPosDto> dtoList = (List<AuditFeeCheckPosDto>) nebulaToolkitService.copyCollectionByWhiteList(checkPosList, AuditFeeCheckPos.class, AuditFeeCheckPosDto.class, HashSet.class, ArrayList.class);
            dtoList.forEach(dto -> {
                dto.setActivityDetailItemCode(codeMap.get(dto.getMatchCode()));
                auditFeeCheckPosService.update(dto);
            });

        });
    }

    /**
     * @param data
     * @param errorMap
     */
    private void dataVal(LinkedHashMap<Integer, AuditFeeCheckPosImportVo> data, Map<Integer, String> errorMap) {
        List<String> matchCodeList = Lists.newArrayList();
        List<String> activityDetailItemCodeList = Lists.newArrayList();
        data.values().forEach(vo -> {
            if (StringUtil.isNotEmpty(vo.getMatchCode())) {
                matchCodeList.add(vo.getMatchCode());
            }
            if (StringUtil.isNotEmpty(vo.getActivityDetailItemCode())) {
                activityDetailItemCodeList.add(vo.getActivityDetailItemCode());
            }
        });
        int size = 400;
        if (CollectionUtil.isNotEmpty(matchCodeList)) {
            List<List<String>> matchCodeGroupList = Lists.partition(matchCodeList, size);
            Set<String> businessUnitCodeSet = Sets.newHashSet();
            AtomicReference<String> businessUnitCode = new AtomicReference<>();
            matchCodeGroupList.forEach(list -> {
                List<AuditFeeCheckPos> checkPosList = auditFeeCheckPosRepository.findByMatchCodeList(list);
                if (CollectionUtil.isNotEmpty(checkPosList)) {
                    Map<String, AuditFeeCheckPos> posMap = checkPosList.stream()
                            .collect(Collectors.toMap(AuditFeeCheckPos::getMatchCode, v -> v, (n, o) -> n));
                    data.forEach((index, vo) -> {
                        AuditFeeCheckPos pos = posMap.get(vo.getMatchCode());
                        if (Objects.nonNull(pos)) {
                            businessUnitCode.set(pos.getBusinessUnitCode());
                            businessUnitCodeSet.add(pos.getBusinessUnitCode());
                            this.validateIsTrue(businessUnitCodeSet.size() == 1, "单次只能导入一个业务的数据! || ");
                            if (BooleanEnum.TRUE.getCapital().equals(pos.getMatchStatus())) {
                                this.validateIsTrue(false, "POS核对已确认! || ");
                            } else {
                                vo.setMatch(true);
                            }
                            if (BooleanEnum.TRUE.getCapital().equals(pos.getIsMatchCost())) {
                                this.validateIsTrue(pos.getActivityDetailItemCode().equals(vo.getActivityDetailItemCode()),
                                        "POS核对单[" + pos.getMatchCode() + "]已关联费用核对["
                                                + pos.getFeeCheckMatchCode() + "]，请先对费用核对操作取消匹配后再处理POS数据!");
                            }
                            String errInfo = this.validateGetErrorInfo();
                            if (errInfo != null) {
                                errorMap.put(index, errorMap.getOrDefault(index, "") + errInfo);
                            }
                        }
                    });
                }
            });
            data.forEach((index, vo) -> {
                this.validateIsTrue(vo.isMatch(), "POS核对单号不存在! || ");
                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errorMap.put(index, errorMap.getOrDefault(index, "") + errInfo);
                }
            });
            if (CollectionUtil.isNotEmpty(errorMap)) {
                return;
            }
            if (CollectionUtil.isNotEmpty(activityDetailItemCodeList)) {
                AuditFeePredictionDto dto = new AuditFeePredictionDto();
                dto.setBusinessUnitCode(businessUnitCode.get());
                Pageable pageable = PageRequest.of(1, size);
                List<List<String>> activityDetailItemCodeGroupList = Lists.partition(activityDetailItemCodeList, size);
                activityDetailItemCodeGroupList.forEach(list -> {
                    dto.setDetailPlanItemCodes(list);
                    Page<AuditFeeCheckDetailPlanVo> detailPlanVoPage = auditFeeCheckDetailPlanVoService.findDetailPlanByConditionsForImport(pageable, dto);
                    if (CollectionUtil.isNotEmpty(detailPlanVoPage.getRecords())) {
                        Map<String, AuditFeeCheckDetailPlanVo> planVoMap = detailPlanVoPage.getRecords().stream()
                                .collect(Collectors.toMap(AuditFeeCheckDetailPlanVo::getDetailPlanItemCode, v -> v, (n, o) -> n));
                        data.forEach((index, vo) -> {
                            AuditFeeCheckDetailPlanVo detailPlanVo = planVoMap.get(vo.getActivityDetailItemCode());
                            if (Objects.nonNull(detailPlanVo)) {
                                vo.setActivityDetailCode(detailPlanVo.getDetailPlanCode());
                                vo.setActivityDetailName(detailPlanVo.getDetailPlanName());
                                vo.setActivityDetailItemCode(detailPlanVo.getDetailPlanItemCode());
                                vo.setActivityDetailItemName(detailPlanVo.getDetailPlanName());
                                vo.setActivityDetailItem(true);
                            }
                        });
                    }
                });
            }
            data.forEach((index, vo) -> {
                this.validateIsTrue(vo.isActivityDetailItem(), "活动编码不存在! || ");
                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errorMap.put(index, errorMap.getOrDefault(index, "") + errInfo);
                }
            });
        }

    }

    /**
     * 基础验证
     *
     * @param data
     * @param errorMap
     */
    private void baseVal(LinkedHashMap<Integer, AuditFeeCheckPosImportVo> data, Map<Integer, String> errorMap) {
        data.forEach((index, vo) -> {
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getMatchCode()), "POS核对单号不能为空! || ");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getActivityDetailItemCode()), "活动编码不能为空! || ");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errorMap.put(index, errInfo);
            }
        });
    }

    @Override
    public Class<AuditFeeCheckPosImportVo> findCrmExcelVoClass() {
        return AuditFeeCheckPosImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "AUDIT_FEE_CHECK_POS_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "POS活动核对批量导入调整";
    }
}
