package com.biz.crm.tpm.business.audit.fee.local.service.internal.check;

import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckDetailPlanRepository;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.service.check.AuditFeeCheckDetailPlanVoCacheService;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.check.AuditFeeCheckDetailPlanVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;


/**
 * 费用核对关联细案缓存ServiceImpl
 * @author liuyifan
 */
@Slf4j
@Service
public class AuditFeeCheckDetailPlanVoCacheServiceImpl implements AuditFeeCheckDetailPlanVoCacheService {

    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisService redisService;

    @Resource(shareable = false)
    private AuditFeeCheckDetailPlanRepository auditFeeCheckDetailPlanRepository;

    /**
     * 1、获取明细列表分页接口
     *
     * @param pageable       分页参数
     * @param cacheKey       缓存key
     * @param matchCode 策略编码
     * @return
     */
    @Override
    public Page<AuditFeeCheckDetailPlanVo> findCachePageList(Pageable pageable, String cacheKey, String matchCode) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        Page<AuditFeeCheckDetailPlanVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        if (redisService.hasKey(redisCacheKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheKey);
            page.setTotal(total);
            List<Object> objects = redisService.lRange(redisCacheKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(objects)) {
                List<AuditFeeCheckDetailPlanVo> voList = (List<AuditFeeCheckDetailPlanVo>) (List) objects;
                page.setRecords(voList);
            }
        } else {
            //redis里面没有
            if (StringUtils.isNotEmpty(matchCode)) {
                //有策略编码，从数据库里面查出来放到缓存里面
                List<AuditFeeCheckDetailPlanVo> dtoList = auditFeeCheckDetailPlanRepository.findDetailPlanByMatchCode(matchCode);
                log.info("商超费用核对-匹配活动细案明细数据：" + JSONArray.toJSONString(dtoList));
                if (!CollectionUtils.isEmpty(dtoList)){
                    redisService.lPushAll(redisCacheKey, AuditFeeConstants.CACHE_EXPIRE_TIME, dtoList.toArray());
                }
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<AuditFeeCheckDetailPlanVo> recordDtoList = dtoList.subList((int) page.offset(), (int) end);
                    page.setRecords(recordDtoList);
                }
            }
        }
        //更新下VO里面的字段值
        return page;
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, List<AuditFeeCheckDetailPlanVo> saveList) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        // 在缓存中排序
        Set<AuditFeeCheckDetailPlanVo> newSet = Sets.newTreeSet(Comparator.comparing(AuditFeeCheckDetailPlanVo::getDetailPlanItemCode));
        Map<String, AuditFeeCheckDetailPlanVo> saveItemMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(saveList)) {
            // 使用明细编码防止重复添加，报错说明这个字段无法区分出数据，需要更换字段或者数据重复
            saveItemMap = saveList.stream().collect(Collectors.toMap(AuditFeeCheckDetailPlanVo::getDetailPlanItemCode, Function.identity(), (a, b) -> a));
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        if (!CollectionUtils.isEmpty(objects)) {
            for (Object object : objects) {
                AuditFeeCheckDetailPlanVo item = (AuditFeeCheckDetailPlanVo) object;
                newSet.add(saveItemMap.getOrDefault(item.getDetailPlanItemCode(), item));
            }
        }
        newSet.addAll(saveList);

        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        if(!CollectionUtils.isEmpty(newSet)){
            redisService.lPushAll(redisCacheKey, AuditFeeConstants.CACHE_EXPIRE_TIME, newSet.toArray());
        }
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param saveList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<AuditFeeCheckDetailPlanVo> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, AuditFeeCheckDetailPlanVo> saveItemMap = saveList.stream().collect(Collectors.toMap(AuditFeeCheckDetailPlanVo::getId, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<AuditFeeCheckDetailPlanVo> newList = Lists.newArrayList();
        List<AuditFeeCheckDetailPlanVo> copyList = Lists.newArrayList();
        for (Object object : objects) {
            AuditFeeCheckDetailPlanVo item = (AuditFeeCheckDetailPlanVo) object;
            if (saveItemMap.containsKey(item.getId())) {
                AuditFeeCheckDetailPlanVo newItem = saveItemMap.get(item.getId());
                if (BooleanEnum.TRUE.getNumStr().equals(newItem.getChecked())) {
                    //选中状态，代表该行数据要复制
                    AuditFeeCheckDetailPlanVo copyItem = nebulaToolkitService.copyObjectByWhiteList(newItem, AuditFeeCheckDetailPlanVo.class, HashSet.class, ArrayList.class);
                    copyItem.setId(UUID.randomUUID().toString().replace("-", ""));
                    copyItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    copyItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

                    copyList.add(copyItem);
                }

                newList.add(newItem);
            } else {

                newList.add(item);
            }
        }
        newList.addAll(0, copyList);
        redisService.del(redisCacheKey);//先全部删掉再放进去吧
        redisService.lPushAll(redisCacheKey, AuditFeeConstants.CACHE_EXPIRE_TIME, newList.toArray());
    }


    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<AuditFeeCheckDetailPlanVo> saveList) {
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<String, AuditFeeCheckDetailPlanVo> saveItemMap = saveList.stream().collect(Collectors.toMap(AuditFeeCheckDetailPlanVo::getDetailPlanItemCode, Function.identity()));
        String redisCacheKey = getRedisCacheKey(cacheKey);
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);

        List<AuditFeeCheckDetailPlanVo> newList = Lists.newArrayList();
        for (Object object : objects) {
            AuditFeeCheckDetailPlanVo item = (AuditFeeCheckDetailPlanVo) object;

            newList.add(saveItemMap.getOrDefault(item.getDetailPlanItemCode(), item));
        }
        redisService.del(redisCacheKey);//删掉重新放
        redisService.lPushAll(redisCacheKey, AuditFeeConstants.CACHE_EXPIRE_TIME, newList.toArray());//设置过期时间1天
    }


    /**
     * 5、获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<AuditFeeCheckDetailPlanVo> findCacheList(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return Lists.newArrayList();
        }
        List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
        return (List<AuditFeeCheckDetailPlanVo>) (List) objects;
    }


    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        redisService.del(redisCacheKey);
    }


    /**
     * 7、多行删除并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void deleteCacheList(String cacheKey, List<String> deleteIds) {
        if (!CollectionUtils.isEmpty(deleteIds)) {
            String redisCacheKey = getRedisCacheKey(cacheKey);
            List<Object> objects = redisService.lRange(redisCacheKey, 0, -1);
            if (deleteIds.size() == objects.size()){
                this.clearCache(cacheKey);
            }else {
                for (Object object : objects) {
                    AuditFeeCheckDetailPlanVo item = (AuditFeeCheckDetailPlanVo) object;
                    if (deleteIds.contains(item.getId())) {
                        redisService.lRemove(redisCacheKey, 0, object);
                    }
                }
            }
        }
    }

    /**
     * 8、获取缓存中的所有明细行长度
     *
     * @param cacheKey 缓存key
     */
    @Override
    public long findCacheListSize(String cacheKey) {
        String redisCacheKey = getRedisCacheKey(cacheKey);
        if (!redisService.hasKey(redisCacheKey)) {
            return 0;
        }
        return redisService.lSize(redisCacheKey);
    }


    /**
     * 获取redis缓存key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    private String getRedisCacheKey(String cacheKey) {
        return AuditFeeConstants.AUDIT_FEE_CACHE_KEY_PREFIX_DETAIL_PLAN + cacheKey;
    }
}
