package com.biz.crm.tpm.business.audit.fee.local.service.internal.check;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.dto.InvoiceExpenseSheetSpecificationDto;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.service.InvoiceExpenseSheetSpecificationVoService;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.vo.InvoiceExpenseSheetSpecificationVo;
import com.biz.crm.kms.business.invoice.sales.data.sdk.dto.SalesDataDto;
import com.biz.crm.kms.business.invoice.sales.data.sdk.service.InvoiceSalesDataVoService;
import com.biz.crm.kms.business.invoice.sales.data.sdk.vo.SalesDataResultVo;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.biz.crm.mdm.business.dictionary.sdk.constant.DictConstant;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheckPos;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckPosRepository;
import com.biz.crm.tpm.business.audit.fee.local.service.AuditFeeCheckPosPullKmsDataService;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeCheckPosConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.PosDataSourceEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.RetailersSystemTypeEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.service.check.AuditFeeCheckPosMatchActivityService;
import com.biz.crm.tpm.business.audit.fee.sdk.template.dto.TpmDeductionMatchingTemplateDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.service.TpmDeductionMatchingTemplateService;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.service.TpmDeductionDetailMappingService;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import liquibase.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * 拉去KMS 数据
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-10-04 16:42
 */
@Slf4j
@Service
public class AuditFeeCheckPosPullKmsDataServiceImpl implements AuditFeeCheckPosPullKmsDataService {

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private InvoiceSalesDataVoService invoiceSalesDataVoService;

    @Autowired(required = false)
    private TpmDeductionMatchingTemplateService tpmDeductionMatchingTemplateService;

    @Autowired(required = false)
    private TpmDeductionDetailMappingService tpmDeductionDetailMappingService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private AuditFeeCheckPosRepository auditFeeCheckPosRepository;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private InvoiceExpenseSheetSpecificationVoService invoiceExpenseSheetSpecificationVoService;

    @Autowired(required = false)
    private AuditFeeCheckPosMatchActivityService auditFeeCheckPosMatchActivityService;

    /**
     * 拉取KMS 的POST数据
     *
     * @param userIdentity
     * @param beginDate
     * @param endDate
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-10-04 16:44
     */
    @Override
    @Async
    public void pullKmsDataAsync(AbstractCrmUserIdentity userIdentity, String beginDate, String endDate) {
        loginUserService.refreshAuthentication(userIdentity);
        String yearMonthDay = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        String lockKey = AuditFeeConstants.AUDIT_FEE_UPDATE_KMS_POS_LOCK + yearMonthDay;
        boolean lockSuccess = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 4);
        Assert.isTrue(lockSuccess, "上次操作[" + yearMonthDay + "]还未完成,本次不执行!");
        try {
            this.pullKmsData(beginDate, endDate);
        } finally {
            redisLockService.unlock(lockKey);
        }

    }

    /**
     * 拉取KMS数据
     *
     * @param beginDate
     * @param endDate
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-10-04 16:44
     */
    @Override
    public void pullKmsData(String beginDate, String endDate) {
        log.info("=====>    拉取KMS的数据参数beginDate[{}] endDate[{}]    <=====", beginDate, endDate);
        List<String> dayList = this.buildDayList(beginDate, endDate);
        Pageable templatePageable = PageRequest.of(1, 40);
        TpmDeductionMatchingTemplateDto templateDto = new TpmDeductionMatchingTemplateDto();
        templateDto.setTenantCode(TenantUtils.getTenantCode());
        templateDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        Page<TpmDeductionMatchingTemplateVo> templatePage = null;
        //KMS的POS数据查询条件
        List<SalesDataDto> salesDataDtoList = this.buildSalesDataDto(dayList);
        //KMS的费用数据查询条件
        List<InvoiceExpenseSheetSpecificationDto> specificationDtoList = this.buildInvoiceExpenseSheetSpecificationDataDto(dayList);
        Map<String, Boolean> onlyKeyMap = Maps.newHashMap();
        Map<String, String> productUnitConvertMap = dictToolkitService.findConvertMapByDictTypeCode(DictConstant.PRODUCT_BASE_UNIT);
        Map<String, String> productUnitMap = dictToolkitService.findMapByDictTypeCode(DictConstant.PRODUCT_BASE_UNIT);
        do {
            templatePage = this.tpmDeductionMatchingTemplateService.findByConditions(templatePageable, templateDto);
            log.info("=====>    扣费匹配模版[{}/{}]页 size[{}]    <=====", templatePage.getCurrent(), templatePage.getPages(), templatePage.getSize());
            List<TpmDeductionMatchingTemplateVo> templatePageRecords = templatePage.getRecords();
            if (CollectionUtil.isEmpty(templatePageRecords)) {
                return;
            }
            Map<String, TpmDeductionDetailMappingVo> detailMappingVoMap = this.buildDetailMappingVoMap(templatePageRecords);
            if (CollectionUtil.isEmpty(detailMappingVoMap)) {
                log.error("=====>   拉取KMS的数据,扣费匹配模版编码[{}]未配置商超映射信息   <=====", JSON.toJSON(templatePageRecords));
                return;
            }
            long currentPage = templatePage.getCurrent();
            long totalPage = templatePage.getPages();
            long pageSize = templatePage.getRecords().size();
            AtomicInteger index = new AtomicInteger(1);
            templatePageRecords.stream()
                    .filter(k -> detailMappingVoMap.containsKey(k.getApplyMappingCode()))
                    .forEach(item -> {
                        log.info("=====>   商超扣费隐射编码[{}] 扣费匹配模版编码[{}] [{}/{}]页 size[{}/{}]    <=====",
                                item.getApplyMappingCode(), item.getCode(),
                                currentPage, totalPage, index.getAndIncrement(), pageSize);
                        if (CollectionUtil.isEmpty(item.getPosAllowances())) {
                            log.error("=====>   拉取KMS的数据,商超扣费隐射编码[{}]未配置信息!扣费匹配模版编码[{}]   <=====",
                                    item.getApplyMappingCode(), item.getCode());
                            return;
                        }
                        TpmDeductionDetailMappingVo detailMappingVo = detailMappingVoMap.get(item.getApplyMappingCode());
                        if (Objects.isNull(detailMappingVo)) {
                            log.error("=====>   拉取KMS的数据,商超扣费隐射编码[{}]未找到数据!扣费匹配模版编码[{}]   <=====",
                                    item.getApplyMappingCode(), item.getCode());
                            return;
                        }
                        item.getPosAllowances().forEach(templateVo -> {
                            PosDataSourceEnum posDataSourceEnum = PosDataSourceEnum.getEnumByCode(templateVo.getDataSource());
                            if (Objects.isNull(posDataSourceEnum)) {
                                log.error("=====>   拉取KMS的数据,扣费匹配模版编码[{}]信息数据来源不合法!POS信息[{}]   <=====",
                                        item.getCode(), templateVo.getDataSource());
                                return;
                            }
                            String onlKey = StringUtil.trimToEmpty(detailMappingVo.getResaleCommercialCode()) + StringUtil.trimToEmpty(detailMappingVo.getSalesInstitutionCode()) +
                                    StringUtil.trimToEmpty(detailMappingVo.getBusinessFormatCode()) + StringUtil.trimToEmpty(detailMappingVo.getBusinessUnitCode()) +
                                    StringUtil.trimToEmpty(posDataSourceEnum.getCode());
                            if (onlyKeyMap.containsKey(onlKey)
                                    && onlyKeyMap.get(onlKey)) {
                                log.error("=====>   拉取KMS的数据,扣费匹配模版编码[{}]维度重复,跳过本次拉取;onlyKey[{}]   <=====",
                                        item.getCode(), onlKey);
                                return;
                            }
                            onlyKeyMap.putIfAbsent(onlKey, false);
                            switch (posDataSourceEnum) {
                                case RETAILER_POS:
                                    pullPosData(salesDataDtoList, detailMappingVo, item, productUnitConvertMap, onlKey, onlyKeyMap);
                                    break;
                                case FEE_DETAIL:
                                    pullFeeData(specificationDtoList, detailMappingVo, item, productUnitMap, onlKey, onlyKeyMap);
                                    break;
                                default:
                                    log.error("=====>   拉取KMS的数据,数据类型:{}[{}]不合法!   <=====",
                                            posDataSourceEnum.getCode(), posDataSourceEnum.getDesc());
                                    break;
                            }

                        });

                    });
            templatePageable = templatePageable.next();
        } while (templatePage.hasNext());
        log.info("=====>   拉取KMS的POS数据 end 开始时间[{}]结束时间[{}]   <=====", beginDate, endDate);
    }

    private List<String> buildDayList(String beginDateStr, String endDateStr) {
        if (StringUtil.isEmpty(beginDateStr)) {
            beginDateStr = DateUtil.formatDate(DateUtil.dateAddDay(new Date(), AuditFeeCheckPosConstants.AUDIT_FEE_KMS_POS_DAY), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        }
        if (StringUtil.isEmpty(endDateStr)) {
            endDateStr = DateUtil.formatDate(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        }
        Date beginDate = DateUtil.parseDate(beginDateStr, DateUtil.DEFAULT_YEAR_MONTH_DAY);
        Date endDate = DateUtil.parseDate(endDateStr, DateUtil.DEFAULT_YEAR_MONTH_DAY);
        List<String> dayList = DateUtil.getBetweenDays(beginDate, endDate);
        if (CollectionUtil.isEmpty(dayList)) {
            dayList.add(DateUtil.formatDate(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY));
        }
        return dayList.stream().distinct().collect(Collectors.toList());
    }

    /**
     * 构建 商超扣费映射 map
     *
     * @param templatePageRecords
     * @return
     */
    private Map<String, TpmDeductionDetailMappingVo> buildDetailMappingVoMap(List<TpmDeductionMatchingTemplateVo> templatePageRecords) {
        if (CollectionUtil.isEmpty(templatePageRecords)) {
            log.error("=====>   拉取KMS的POS数据,无扣费匹配模版信息  <=====");
            return Maps.newHashMap();
        }
        List<String> applyMappingCodeList = templatePageRecords.stream()
                .filter(k -> StringUtil.isNotEmpty(k.getApplyMappingCode()))
                .map(TpmDeductionMatchingTemplateVo::getApplyMappingCode)
                .distinct().collect(Collectors.toList());
        if (CollectionUtil.isEmpty(applyMappingCodeList)) {
            log.error("=====>   拉取KMS的POS数据,扣费匹配模版无商超扣费隐射信息[{}]   <=====", JSON.toJSON(templatePageRecords));
            return Maps.newHashMap();
        }
        List<TpmDeductionDetailMappingVo> detailMappingVoList = tpmDeductionDetailMappingService.findByCodes(applyMappingCodeList);
        if (CollectionUtil.isEmpty(detailMappingVoList)) {
            log.error("=====>   拉取KMS的POS数据,扣费匹配模版编码配置的商超扣费隐射信息{}未在商超扣费隐射处找到!   <=====", applyMappingCodeList);
            return Maps.newHashMap();
        }
        return detailMappingVoList.stream()
                .filter(k -> StringUtil.isNotEmpty(k.getCode()))
                .collect(Collectors.toMap(TpmDeductionDetailMappingVo::getCode, v -> v, (n, o) -> n));
    }

    /**
     * 构建 POS查询参数
     *
     * @param beginDate
     * @param endDate
     * @return
     */
    private List<SalesDataDto> buildSalesDataDto(List<String> dayList) {
        List<SalesDataDto> salesDataDtoList = Lists.newArrayList();
        if (CollectionUtil.isEmpty(dayList)) {
            SalesDataDto salesDataDto = new SalesDataDto();
            salesDataDto.setTenantCode(TenantUtils.getTenantCode());
            salesDataDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            salesDataDto.setOrderStatus(InvoicesStatus.S200.getDictCode());
            salesDataDto.setModifyTimeDay(DateUtil.formatDate(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY));
            salesDataDtoList.add(salesDataDto);
            return salesDataDtoList;
        }
        dayList.forEach(item -> {
            SalesDataDto salesDataDto = new SalesDataDto();
            salesDataDto.setTenantCode(TenantUtils.getTenantCode());
            salesDataDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            salesDataDto.setOrderStatus(InvoicesStatus.S200.getDictCode());
            salesDataDto.setModifyTimeDay(item);
            salesDataDtoList.add(salesDataDto);
        });
        return salesDataDtoList;
    }

    /**
     * 构建 费用明细查询参数
     *
     * @param dayList
     * @return
     */
    private List<InvoiceExpenseSheetSpecificationDto> buildInvoiceExpenseSheetSpecificationDataDto(List<String> dayList) {
        List<InvoiceExpenseSheetSpecificationDto> specificationDtoList = Lists.newArrayList();
        if (CollectionUtil.isEmpty(dayList)) {
            InvoiceExpenseSheetSpecificationDto specificationDto = new InvoiceExpenseSheetSpecificationDto();
            specificationDto.setTenantCode(TenantUtils.getTenantCode());
            specificationDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            specificationDto.setOrderStatus(InvoicesStatus.S200.getDictCode());
            specificationDto.setModifyTimeDay(DateUtil.formatDate(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY));
            specificationDtoList.add(specificationDto);
            return specificationDtoList;
        }
        dayList.forEach(item -> {
            InvoiceExpenseSheetSpecificationDto specificationDto = new InvoiceExpenseSheetSpecificationDto();
            specificationDto.setTenantCode(TenantUtils.getTenantCode());
            specificationDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            specificationDto.setOrderStatus(InvoicesStatus.S200.getDictCode());
            specificationDto.setModifyTimeDay(item);
            specificationDtoList.add(specificationDto);
        });


        return specificationDtoList;
    }

    /**
     * 拉取KMS的POS数据
     *
     * @param salesDataDto
     * @param detailMappingVo
     * @param templateVo
     * @param productUnitConvertMap
     */
    private void pullPosData(List<SalesDataDto> salesDataDtoList, TpmDeductionDetailMappingVo detailMappingVo,
                             TpmDeductionMatchingTemplateVo templateVo, Map<String, String> productUnitConvertMap,
                             String onlyKey, Map<String, Boolean> onlyKeyMap) {
        if (CollectionUtil.isEmpty(salesDataDtoList)) {
            return;
        }
        salesDataDtoList.forEach(salesDataDto -> {
            if (Objects.isNull(salesDataDto)) {
                log.error("=====>    拉取KMS的POS数据 销售数据查询参数为空 商超扣费映射模板[{}]  扣费匹配模板[{}]    <=====",
                        detailMappingVo.getCode(), templateVo.getCode());
                return;
            }
            if (Objects.isNull(detailMappingVo)) {
                log.error("=====>    拉取KMS的POS数据 销售数据参数[{}] 商超扣费映射参数为空  扣费匹配模板[{}]    <=====",
                        salesDataDto, templateVo.getCode());
                return;
            }
            if (Objects.isNull(templateVo)) {
                log.error("=====>    拉取KMS的POS数据 销售数据参数[{}] 商超扣费映射模板[{}]  扣费匹配模板参数为空    <=====",
                        salesDataDto, detailMappingVo.getCode());
                return;
            }
            Pageable posPageable = PageRequest.of(1, 600);
            Page<SalesDataResultVo> posPage = null;
            salesDataDto.setCustomerRetailerCode(detailMappingVo.getResaleCommercialCode());
            salesDataDto.setSalesOrgCode(detailMappingVo.getSalesInstitutionCode());
            salesDataDto.setBusinessFormatCode(detailMappingVo.getBusinessFormatCode());
            salesDataDto.setBusinessUnitCode(detailMappingVo.getBusinessUnitCode());
            do {
                try {
                    posPage = invoiceSalesDataVoService.pagePosByGroupTotal(posPageable, salesDataDto);
                    posPageable = posPageable.next();
                    if (Objects.isNull(posPage)) {
                        log.error("=====>    拉取KMS的POS数据参数[{}] [{}/{}]页 KMS返回null   <=====", JSON.toJSONString(salesDataDto),
                                posPageable.getPageNumber(), posPageable.getPageSize());
                        return;
                    }
                    log.info("=====>    拉取KMS的POS数据参数[{}] 商超扣费映射模板[{}]  扣费匹配模板[{}] [{}/{}]页    <=====", JSON.toJSONString(salesDataDto),
                            detailMappingVo.getCode(), templateVo.getCode(), posPage.getCurrent(), posPage.getPages());
                    List<AuditFeeCheckPos> entityList = this.buildKmsPosEntity(detailMappingVo, templateVo, posPage.getRecords(), productUnitConvertMap);
                    if (!onlyKeyMap.getOrDefault(onlyKey, false)
                            && CollectionUtil.isNotEmpty(entityList)) {
                        onlyKeyMap.put(onlyKey, true);
                    }
                    this.saveOrUpdateEntityList(entityList);
                } catch (Exception e) {
                    log.error("=====>    拉取KMS的POS数据参数[{}] [{}/{}]页 异常   <=====", JSON.toJSONString(salesDataDto),
                            posPageable.getPageNumber(), posPageable.getPageSize());
                    log.error("", e);

                }
            } while (Objects.nonNull(posPage) && posPage.hasNext());
        });
    }

    /**
     * 拉取KMS的费用数据
     *
     * @param sheetSpecificationDto
     * @param detailMappingVo
     * @param templateVo
     * @param productUnitMap
     */
    private void pullFeeData(List<InvoiceExpenseSheetSpecificationDto> specificationDtoList, TpmDeductionDetailMappingVo detailMappingVo,
                             TpmDeductionMatchingTemplateVo templateVo, Map<String, String> productUnitMap,
                             String onlyKey, Map<String, Boolean> onlyKeyMap) {
        if (CollectionUtil.isEmpty(specificationDtoList)) {
            return;
        }
        specificationDtoList.forEach(sheetSpecificationDto -> {
            if (Objects.isNull(sheetSpecificationDto)) {
                log.error("=====>    拉取KMS的费用数据 费用数据查询参数为空 商超扣费映射模板[{}]  扣费匹配模板[{}]    <=====",
                        detailMappingVo.getCode(), templateVo.getCode());
                return;
            }
            if (Objects.isNull(detailMappingVo)) {
                log.error("=====>    拉取KMS的费用数据 费用数据参数[{}] 商超扣费映射参数为空  扣费匹配模板[{}]    <=====",
                        sheetSpecificationDto, templateVo.getCode());
                return;
            }
            if (Objects.isNull(templateVo)) {
                log.error("=====>    拉取KMS的费用数据 费用数据参数[{}] 商超扣费映射模板[{}]  扣费匹配模板参数为空    <=====",
                        sheetSpecificationDto, detailMappingVo.getCode());
                return;
            }
            Pageable posPageable = PageRequest.of(1, 600);
            Page<InvoiceExpenseSheetSpecificationVo> posPage = null;
            sheetSpecificationDto.setCustomerRetailerCode(detailMappingVo.getResaleCommercialCode());
            sheetSpecificationDto.setSalesOrgCode(detailMappingVo.getSalesInstitutionCode());
            sheetSpecificationDto.setBusinessFormatCode(detailMappingVo.getBusinessFormatCode());
            sheetSpecificationDto.setBusinessUnitCode(detailMappingVo.getBusinessUnitCode());
            if (RetailersSystemTypeEnum.VANGUARD.getCode().equals(sheetSpecificationDto.getCustomerRetailerCode())
                    || RetailersSystemTypeEnum.TESCO.getCode().equals(sheetSpecificationDto.getCustomerRetailerCode())) {
                //零售商是华润或乐购时，只取“抓单路径”为“促销费用单”的数据；
                sheetSpecificationDto.setUrlList(Collections.singletonList("促销费用单"));

            } else if (RetailersSystemTypeEnum.REN_REN_LE.getCode().equals(sheetSpecificationDto.getCustomerRetailerCode())) {
                //零售商是人人乐时，只取“抓单路径”为“促销”的数据；
                sheetSpecificationDto.setUrlList(Collections.singletonList("促销"));
            }
            do {
                try {
                    posPage = invoiceExpenseSheetSpecificationVoService.pageInvoiceExpenseSheetSpecificationByGroupTotal(posPageable, sheetSpecificationDto);
                    posPageable = posPageable.next();
                    if (Objects.isNull(posPage)) {
                        log.error("=====>    拉取KMS的费用数据参数[{}] [{}/{}]页 KMS返回null   <=====", JSON.toJSONString(sheetSpecificationDto),
                                posPageable.getPageNumber(), posPageable.getPageSize());
                        return;
                    }
                    log.info("=====>    拉取KMS的费用数据参数[{}] 商超扣费映射模板[{}]  扣费匹配模板[{}] [{}/{}]页    <=====", JSON.toJSONString(sheetSpecificationDto),
                            detailMappingVo.getCode(), templateVo.getCode(), posPage.getCurrent(), posPage.getPages());
                    List<AuditFeeCheckPos> entityList = this.buildKmsFeeEntity(detailMappingVo, templateVo, posPage.getRecords(), productUnitMap);
                    if (!onlyKeyMap.getOrDefault(onlyKey, false)
                            && CollectionUtil.isNotEmpty(entityList)) {
                        onlyKeyMap.put(onlyKey, true);
                    }
                    this.saveOrUpdateEntityList(entityList);
                } catch (Exception e) {
                    log.error("=====>    拉取KMS的费用数据参数[{}] [{}/{}]页 异常   <=====", JSON.toJSONString(sheetSpecificationDto),
                            posPageable.getPageNumber(), posPageable.getPageSize());
                    log.error("", e);
                }
            } while (Objects.nonNull(posPage) && posPage.hasNext());
        });

    }

    /**
     * 区分新增还是保存
     *
     * @param entityList
     */
    private void saveOrUpdateEntityList(List<AuditFeeCheckPos> entityList) {
        if (CollectionUtil.isEmpty(entityList)) {
            return;
        }
        entityList = new ArrayList<>(entityList.stream()
                .filter(k -> StringUtil.isNotEmpty(k.getId()))
                .collect(Collectors.toMap(AuditFeeCheckPos::getId, v -> v, (n, o) -> n)).values());
        if (CollectionUtil.isEmpty(entityList)) {
            return;
        }
        List<String> idList = entityList.stream()
                .filter(k -> StringUtil.isNotEmpty(k.getId()))
                .map(AuditFeeCheckPos::getId)
                .collect(Collectors.toList());
        List<AuditFeeCheckPos> oldList = auditFeeCheckPosRepository.listIdsByIds(idList);

        List<AuditFeeCheckPos> saveList = new ArrayList<>();
        List<AuditFeeCheckPos> updateList = new ArrayList<>();
        if (CollectionUtil.isEmpty(oldList)) {
            saveList.addAll(entityList);
        } else {
            Map<String, AuditFeeCheckPos> idsMap = oldList.stream()
                    .filter(k -> StringUtil.isNotEmpty(k.getId()))
                    .collect(Collectors.toMap(AuditFeeCheckPos::getId, v -> v, (n, o) -> n));
            entityList.forEach(entity -> {
                BigDecimal productNumber = Optional.ofNullable(entity.getProductNumber()).orElse(BigDecimal.ZERO);
                BigDecimal promotionDeduction = Optional.ofNullable(entity.getPromotionDeduction()).orElse(BigDecimal.ZERO);
                entity.setOnePromotionDeduction(BigDecimal.ZERO);
                if (BigDecimal.ZERO.compareTo(productNumber) != 0) {
                    entity.setOnePromotionDeduction(promotionDeduction.divide(productNumber,
                            4, BigDecimal.ROUND_HALF_UP));
                }
                entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                AuditFeeCheckPos oldEntity = idsMap.get(entity.getId());
                if (Objects.nonNull(oldEntity)) {
                    if (!BooleanEnum.TRUE.getCapital().equals(oldEntity.getMatchStatus())) {
                        entity.setDiffType(auditFeeCheckPosMatchActivityService.oneDiffAmountGetDiffType(oldEntity.getTotalDiffFinalAmount(), oldEntity.getActivityDetailItemCode()));
                        updateList.add(entity);
                    }
                } else {
                    saveList.add(entity);
                }
            });
        }

        if (CollectionUtil.isNotEmpty(saveList)) {
            List<String> codeList = generateCodeService.generateCode(AuditFeeCheckPosConstants.POS_CODE, saveList.size());
            for (int i = 0; i < saveList.size(); i++) {
                saveList.get(i).setMatchCode(codeList.get(i));
                saveList.get(i).setIsMatchActivity(BooleanEnum.FALSE.getCapital());
                saveList.get(i).setIsMatchCost(BooleanEnum.FALSE.getCapital());
            }
            auditFeeCheckPosRepository.saveBatch(saveList);
        }
        if (CollectionUtil.isNotEmpty(updateList)) {
            updateList.forEach(entity -> {
                auditFeeCheckPosRepository.updateAmountById(entity);
            });
        }
    }

    /**
     * 保存或更新KMS的POS数据
     *
     * @param templateVo
     * @param records
     */
    private List<AuditFeeCheckPos> buildKmsPosEntity(TpmDeductionDetailMappingVo detailMappingVo, TpmDeductionMatchingTemplateVo templateVo,
                                                     List<SalesDataResultVo> records, Map<String, String> productUnitConvertMap) {
        if (CollectionUtil.isEmpty(records)) {
            return Collections.emptyList();
        }
        List<AuditFeeCheckPos> entityList = new ArrayList<>();
        String tenantCode = TenantUtils.getTenantCode();
        String enableCode = EnableStatusEnum.ENABLE.getCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        records.forEach(item -> {
            AuditFeeCheckPos entity = new AuditFeeCheckPos();
            //基础信息
            entity.setTenantCode(tenantCode);
            entity.setEnableStatus(enableCode);
            entity.setDelFlag(delFlag);
            entity.setMatchTemplateCode(templateVo.getCode());
            entity.setMatchTemplateName(templateVo.getName());
            entity.setCustomerRetailerCode(detailMappingVo.getResaleCommercialCode());
            entity.setCustomerRetailerName(detailMappingVo.getResaleCommercialName());

            //POS数据的销售组织  存的是  销售机构
            entity.setSalesInstitutionCode(item.getSalesOrgCode());
            entity.setSalesInstitutionName(item.getSalesOrgName());

            //KMS pos数据信息
            entity.setBusinessUnitCode(item.getBusinessUnitCode());
            entity.setBusinessFormatCode(item.getBusinessFormatCode());
            entity.setBusinessArea(item.getBusinessArea());
            entity.setProvinceCode(item.getProvinceCode());
            entity.setProvinceName(item.getProvinceName());
            String salesDateStr = item.getSalesDate();
            entity.setSalesDateStr(salesDateStr);
            try {
                entity.setSalesDate(DateUtil.parseDate(salesDateStr, DateUtil.DEFAULT_YEAR_MONTH_DAY));
                entity.setYearMonthLy(DateUtil.format(entity.getSalesDate(), DateUtil.DEFAULT_YEAR_MONTH));
            } catch (Exception e) {
                log.error("", e);
            }
            entity.setChannelCode(item.getChannelCode());
            entity.setChannelName(item.getChannelName());
            if (StringUtil.isNotEmpty(item.getCurCompanyUnit())) {
                entity.setUnitCode(productUnitConvertMap.getOrDefault(item.getCurCompanyUnit(), ""));
                if (StringUtil.isNotEmpty(entity.getUnitCode())) {
                    entity.setUnitName(item.getCurCompanyUnit());
                }
            }
            entity.setDirectCode(item.getDirectCode());
            entity.setDirectName(item.getDirectName());
            entity.setTerminalCode(item.getDeliveryPartyCode());
            entity.setTerminalName(item.getDeliveryPartyName());
            entity.setProductCode(item.getGoodsCode());
            entity.setProductName(item.getGoodsName());
            entity.setPromotionDeduction(Optional.ofNullable(item.getPromotionDeduction()).orElse(BigDecimal.ZERO));
            entity.setProductNumber(Optional.ofNullable(item.getCurCompanyUnitOrderQuantity()).orElse(BigDecimal.ZERO));
            entity.setOnePromotionDeduction(BigDecimal.ZERO);
            if (BigDecimal.ZERO.compareTo(entity.getProductNumber()) != 0) {
                entity.setOnePromotionDeduction(entity.getPromotionDeduction().divide(entity.getProductNumber(),
                        4, BigDecimal.ROUND_HALF_UP));
            }

            BigDecimal oneApplicationFee = Optional.ofNullable(entity.getOneApplicationFee()).orElse(BigDecimal.ZERO);
            //单件费用差异 = 单件促销扣款（元）- 单件申请费用（元）
            BigDecimal oneDiffAmount = entity.getOnePromotionDeduction().subtract(oneApplicationFee);
            entity.setOneDiffAmount(oneDiffAmount);
            //分摊促销扣款
            BigDecimal sharePromotionDeduction = Optional.ofNullable(entity.getSharePromotionDeduction()).orElse(BigDecimal.ZERO);
            entity.setSharePromotionDeduction(sharePromotionDeduction);
            //差异总费用
            entity.setTotalDiffAmount(oneDiffAmount.multiply(entity.getProductNumber()).add(sharePromotionDeduction));
            entity.setTotalDiffFinalAmount(entity.getTotalDiffAmount());

            entity.setMatchStatus(BooleanEnum.FALSE.getCapital());
            entity.setIsMatchActivity(BooleanEnum.FALSE.getCapital());
            entity.setIsMatchCost(BooleanEnum.FALSE.getCapital());
            entity.setDiffType(auditFeeCheckPosMatchActivityService.oneDiffAmountGetDiffType(entity.getTotalDiffFinalAmount(), entity.getActivityDetailItemCode()));
            entity.setDataSource(PosDataSourceEnum.RETAILER_POS.getCode());
            if (StringUtil.isEmpty(entity.getDataSource())
                    || StringUtil.isEmpty(entity.getDirectCode())
                    || StringUtil.isEmpty(entity.getTerminalCode())
                    || StringUtil.isEmpty(entity.getProductCode())
                    || StringUtil.isEmpty(entity.getChannelCode())
                    || StringUtil.isEmpty(entity.getSalesDateStr())) {
                return;
            }
            String onlyKey = entity.getDataSource() + entity.getDirectCode()
                    + entity.getTerminalCode() + entity.getProductCode()
                    + entity.getChannelCode() + entity.getSalesDateStr();
            entity.setOnlyKey(onlyKey);
            entity.setId(DigestUtils.md5Hex(onlyKey));
            entityList.add(entity);

        });
        return entityList;
    }

    /**
     * 保存或更新KMS的费用数据
     *
     * @param templateVo
     * @param records
     */
    private List<AuditFeeCheckPos> buildKmsFeeEntity(TpmDeductionDetailMappingVo detailMappingVo, TpmDeductionMatchingTemplateVo templateVo,
                                                     List<InvoiceExpenseSheetSpecificationVo> records, Map<String, String> productUnitMap) {
        if (CollectionUtil.isEmpty(records)) {
            return Collections.emptyList();
        }
        List<AuditFeeCheckPos> entityList = new ArrayList<>();
        String tenantCode = TenantUtils.getTenantCode();
        String enableCode = EnableStatusEnum.ENABLE.getCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        records.forEach(item -> {
            AuditFeeCheckPos entity = new AuditFeeCheckPos();
            //基础信息
            entity.setTenantCode(tenantCode);
            entity.setEnableStatus(enableCode);
            entity.setDelFlag(delFlag);
            entity.setMatchTemplateCode(templateVo.getCode());
            entity.setMatchTemplateName(templateVo.getName());

            entity.setCustomerRetailerCode(detailMappingVo.getResaleCommercialCode());
            entity.setCustomerRetailerName(detailMappingVo.getResaleCommercialName());

            entity.setSalesInstitutionCode(item.getSalesOrgCode());
            entity.setSalesInstitutionName(item.getSalesOrgName());

            //KMS pos数据信息
            entity.setBusinessUnitCode(item.getBusinessUnitCode());
            entity.setBusinessFormatCode(item.getBusinessFormatCode());
            entity.setBusinessArea(item.getBusinessArea());
            entity.setProvinceCode(item.getProvinceCode());
            entity.setProvinceName(item.getProvinceName());
            String orderDateStr = item.getOrderDate();
            entity.setSalesDateStr(orderDateStr);
            try {
                entity.setSalesDate(DateUtil.parseDate(orderDateStr, DateUtil.DEFAULT_YEAR_MONTH_DAY));
                entity.setYearMonthLy(DateUtil.format(entity.getSalesDate(), DateUtil.DEFAULT_YEAR_MONTH));
            } catch (Exception e) {
                log.error("", e);
            }
            if (StringUtil.isNotEmpty(item.getCurCompanyUnit())) {
                entity.setUnitCode(productUnitMap.getOrDefault(item.getCurCompanyUnit(), ""));
                if (StringUtil.isNotEmpty(entity.getUnitCode())) {
                    entity.setUnitName(item.getCurCompanyUnit());
                }
            }
            entity.setChannelCode("");
            entity.setChannelName("");

            entity.setDirectCode(item.getDirectCode());
            entity.setDirectName(item.getDirectName());
            entity.setTerminalCode(item.getDeliveryPartyCode());
            entity.setTerminalName(item.getDeliveryPartyName());
            entity.setProductCode(item.getProductCode());
            entity.setProductName(item.getProductName());
            entity.setPromotionDeduction(Optional.ofNullable(item.getAmount()).orElse(BigDecimal.ZERO));
            entity.setProductNumber(Optional.ofNullable(item.getCurCompanyUnitOrderQuantity()).orElse(BigDecimal.ZERO));
            entity.setOnePromotionDeduction(BigDecimal.ZERO);
            if (BigDecimal.ZERO.compareTo(entity.getProductNumber()) != 0) {
                entity.setOnePromotionDeduction(entity.getPromotionDeduction().divide(entity.getProductNumber(),
                        4, BigDecimal.ROUND_HALF_UP));
            }

            BigDecimal oneApplicationFee = Optional.ofNullable(entity.getOneApplicationFee()).orElse(BigDecimal.ZERO);
            //单件费用差异 = 单件促销扣款（元）- 单件申请费用（元）
            BigDecimal oneDiffAmount = entity.getOnePromotionDeduction().subtract(oneApplicationFee);
            entity.setOneDiffAmount(oneDiffAmount);
            //分摊促销扣款
            BigDecimal sharePromotionDeduction = Optional.ofNullable(entity.getSharePromotionDeduction()).orElse(BigDecimal.ZERO);
            entity.setSharePromotionDeduction(sharePromotionDeduction);
            //差异总费用
            entity.setTotalDiffAmount(oneDiffAmount.multiply(entity.getProductNumber()).add(sharePromotionDeduction));
            entity.setTotalDiffFinalAmount(entity.getTotalDiffAmount());

            entity.setMatchStatus(BooleanEnum.FALSE.getCapital());
            entity.setIsMatchActivity(BooleanEnum.FALSE.getCapital());
            entity.setIsMatchCost(BooleanEnum.FALSE.getCapital());
            entity.setDiffType(auditFeeCheckPosMatchActivityService.oneDiffAmountGetDiffType(entity.getTotalDiffFinalAmount(), entity.getActivityDetailItemCode()));
            entity.setDataSource(PosDataSourceEnum.FEE_DETAIL.getCode());
            if (StringUtil.isEmpty(entity.getDataSource())
                    || StringUtil.isEmpty(entity.getDirectCode())
                    || StringUtil.isEmpty(entity.getTerminalCode())
                    || StringUtil.isEmpty(entity.getProductCode())
                    || StringUtil.isEmpty(entity.getSalesDateStr())) {
                return;
            }
            String onlyKey = entity.getDataSource() + entity.getDirectCode()
                    + entity.getTerminalCode() + entity.getProductCode()
                    + entity.getChannelCode() + entity.getSalesDateStr();
            entity.setOnlyKey(onlyKey);
            entity.setId(DigestUtils.md5Hex(onlyKey));
            entityList.add(entity);

        });
        return entityList;
    }

}
