package com.biz.crm.tpm.business.audit.fee.local.service.internal.check.async;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.kms.business.audit.fee.sdk.dto.AuditFeeReqDto;
import com.biz.crm.kms.business.audit.fee.sdk.enums.AuditFeeMatchStatusEnum;
import com.biz.crm.kms.business.audit.fee.sdk.service.cost.AuditFeeCostService;
import com.biz.crm.kms.business.audit.fee.sdk.vo.AuditFeeRespVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheck;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheckCost;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckCostRepository;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.check.AuditFeeMatchDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.DeductionMatchingTemplateConditionEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateAllowanceVo;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.service.TpmDeductionDetailMappingService;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingCustomerVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingTextVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingVo;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import liquibase.util.MD5Util;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

@Slf4j
@Component
public class GenerateAuditFeeCheckCostAsync {

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 商超扣费映射Service
     */
    @Autowired(required = false)
    private TpmDeductionDetailMappingService deductionDetailMappingService;

    /**
     * KMS数据拉取
     */
    @Autowired(required = false)
    private AuditFeeCostService auditFeeCostService;

    @Autowired(required = false)
    private AuditFeeCheckCostRepository auditFeeCheckCostRepository;

    @Autowired(required = false)
    private GenerateAuditFeeCheckCostAsyncHelper generateAuditFeeCheckCostAsyncHelper;

    @Transactional(rollbackFor = Exception.class)
    public void generateForCostOrder(AuditFeeMatchDto auditFeeMatchDto, TpmDeductionMatchingTemplateVo feeMatchTemplate, AbstractCrmUserIdentity abstractLoginUser) {
        this.generateForCostOrderAsync(auditFeeMatchDto, feeMatchTemplate, abstractLoginUser);
    }

    /**
     * 拉取费用单
     *
     * @param auditFeeMatchDto
     * @param feeMatchTemplate
     * @param abstractLoginUser
     * @return
     */
    @Async("auditFeeThread")
    public void generateForCostOrderAsync(AuditFeeMatchDto auditFeeMatchDto, TpmDeductionMatchingTemplateVo feeMatchTemplate, AbstractCrmUserIdentity abstractLoginUser) {
        loginUserService.refreshAuthentication(abstractLoginUser);
        log.info("拉取费用单核对主数据:{}", JSON.toJSONString(auditFeeMatchDto));

        List<AuditFeeRespVo> respVoList = new ArrayList<>();
        List<AuditFeeCheckCost> costList;
        Set<String> costUniqueKeySet = new HashSet<>();

        String redisLockKey = AuditFeeConstants.AUDIT_FEE_PULL_KMS + feeMatchTemplate.getCode() + ":" + feeMatchTemplate.getName();
        boolean hasLock = false;
        try {
            hasLock = redisLockService.tryLock(redisLockKey, TimeUnit.MINUTES, 15);
            if (!hasLock) {
                throw new RuntimeException("正在使用模板【" + redisLockKey + "】拉取数据，请等待");
            }
            List<TpmDeductionMatchingTemplateAllowanceVo> feeAllowances = feeMatchTemplate.getFeeAllowances();
            if (CollectionUtils.isEmpty(feeAllowances)) {
                log.info("拉取费用单模板容差个数为0不拉取数据");
                return;
            }
            //任务总数
            AtomicInteger totalTasks = new AtomicInteger(0);
            AtomicInteger currentTasks = new AtomicInteger(0);

            Map<String, TpmDeductionMatchingTemplateAllowanceVo> feeAllowanceMap = feeAllowances.stream().collect(Collectors.toMap(e -> e.getDataSource() + "_" + e.getApplyBusinessAreaCode(), Function.identity(), (a,b) -> a));
            log.info("拉取费用单模板去重容差:{}个数:{}", JSON.toJSONString(feeAllowanceMap.keySet()), feeAllowanceMap.size());

            // 1.组装查询参数
            AuditFeeReqDto auditFeeReqDto = new AuditFeeReqDto();
            auditFeeReqDto.setCostIsMatch(AuditFeeMatchStatusEnum.WITH.getCode());
            auditFeeReqDto.setWithoutTime(true);
            // 抽取数据：送达方编码（非必填）+费用日期开始+费用日期结束+业务单元+业态+零售商编码+销售机构编码（非必填）+扣费明细项名称（非必填）+售达方编码（非必填）+数据来源
            auditFeeReqDto.setDeliveryPartyCode(auditFeeMatchDto.getTerminalCode());
            auditFeeReqDto.setOrderDateBegin(DateUtil.formatDate(auditFeeMatchDto.getStartTime()));
            auditFeeReqDto.setOrderDateEnd(DateUtil.formatDate(auditFeeMatchDto.getEndTime()));
            auditFeeReqDto.setBusinessUnitCode(feeMatchTemplate.getBusinessUnitCode());
            auditFeeReqDto.setBusinessFormatCode(feeMatchTemplate.getBusinessFormatCode());
            //获取商超映射数据
            if (StringUtils.isEmpty(feeMatchTemplate.getApplyMappingCode())) {
                throw new IllegalArgumentException("模板[" + feeMatchTemplate.getName() + "]商超映射编码不存在");
            }
            List<TpmDeductionDetailMappingVo> mappingVoList = deductionDetailMappingService.findByCodes(Lists.newArrayList(feeMatchTemplate.getApplyMappingCode()));
            if (CollectionUtils.isEmpty(mappingVoList)) {
                throw new IllegalArgumentException("模板[" + feeMatchTemplate.getName() + "]商超映射数据不存在");
            }
            TpmDeductionDetailMappingVo mappingVo = mappingVoList.get(0);
            // 从映射取的部分
            // 零售商从映射取
            auditFeeReqDto.setCustomerRetailerCode(mappingVo.getResaleCommercialCode());
            if (StringUtils.isNotBlank(mappingVo.getSalesInstitutionCode())) {
                auditFeeReqDto.setSalesOrgCode(mappingVo.getSalesInstitutionCode());
            }
            if (CollectionUtils.isNotEmpty(mappingVo.getDeductionDetailMappingTextList())) {
                auditFeeReqDto.setDeductionNameList(mappingVo.getDeductionDetailMappingTextList().stream().map(TpmDeductionDetailMappingTextVo::getText).filter(Objects::nonNull).distinct().collect(Collectors.toList()));
            }
            //适用客户编码不是mdg编码，需要转换
            if (CollectionUtils.isNotEmpty(mappingVo.getCustomerList())) {
                List<String> customerErpCodeList = mappingVo.getCustomerList().stream().map(TpmDeductionDetailMappingCustomerVo::getErpCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(customerErpCodeList)) {
                    auditFeeReqDto.setSoldToPartyCodeList(customerErpCodeList);
                }
            }

            String templateInfoStr = feeMatchTemplate.getName() + feeMatchTemplate.getCode();
            String mappingStr = feeMatchTemplate.getApplyMappingName() + feeMatchTemplate.getApplyMappingCode();
            // 2.遍历模板容差 查询数据, 按数据源+区域使用
            for (TpmDeductionMatchingTemplateAllowanceVo allowanceVo : feeAllowanceMap.values()) {
                String dataSource = allowanceVo.getDataSource();
                auditFeeReqDto.setDataSource(dataSource);
                String applyBusinessAreaCode = allowanceVo.getApplyBusinessAreaCode();

                String[] areaArray = null;
                if (StringUtils.isBlank(applyBusinessAreaCode)) {
                    areaArray = new String[1];
                    areaArray[0] = null;
                } else {
                    areaArray = applyBusinessAreaCode.split(",");
                }
                log.info("拉取费用单-当前容差:{}", JSON.toJSONString(allowanceVo));
                log.info("拉取费用单-区域数组:{}", JSON.toJSONString(areaArray));
                for (String areaCode : areaArray) {
                    auditFeeReqDto.setBusinessArea(areaCode);
                    // 3.查询总条数
                    log.info("拉取费用单参数:{}", JSON.toJSONString(auditFeeReqDto));
                    Pageable pageable = Pageable.ofSize(1);
                    Page<AuditFeeRespVo> costPages = auditFeeCostService.findByConditions(pageable, auditFeeReqDto);
                    int total = (int) costPages.getTotal();
                    log.info("使用模板:{},映射:{},容差数据来源{},费用单总条数:{}", templateInfoStr, mappingStr, dataSource,total);
                    if (total <= 0) {
                        continue;
                    }
                    int pageSize = 400;
                    int pageNum = (total / pageSize) + 1;
                    totalTasks.addAndGet(pageNum);
                    // 4.分页查询数据
                    for (int c = 1; c <= pageNum; c++) {
                        Pageable currentPage = PageRequest.of(c, pageSize);
                        //拉取费用单
                        log.info("拉取费用单模板{}数据源{} 开始第:{}页", templateInfoStr, dataSource, c);
                        long kmsPageTimeUsed = this.getKMSPage(templateInfoStr, mappingStr, currentTasks, currentPage, auditFeeReqDto, respVoList);
                        log.info("拉取费用单模板{}数据源{} 已完成第:{}页 耗时{}", templateInfoStr, dataSource, c, kmsPageTimeUsed);

                        log.info("拉取费用单模板{}映射{}数据源{} 第{}页后存储{}条", templateInfoStr, mappingStr, dataSource, c, respVoList.size());
                        // 5k条将处理一次。给每个费用单设置唯一值存入数据库，唯一值存入缓存。所有接口分页查询完成后，再分别根据唯一值查询数据，并剔除已确认的数据，聚合成一个费用核对主数据
                        costList = this.convertToAuditFeeCheckCost(respVoList, allowanceVo, feeMatchTemplate, mappingVo);
                        respVoList.clear();
                        log.info("拉取费用单模板{}映射{}数据源{} 第{}页后唯一计算完成", templateInfoStr, mappingStr, dataSource, c);

                        auditFeeCheckCostRepository.saveBatch(costList);
                        log.info("拉取费用单模板{}映射{}数据源{} 第{}页后存储完成", templateInfoStr, mappingStr, dataSource, c);

                        costUniqueKeySet.addAll(costList.stream().map(AuditFeeCheckCost::getMd5UniqueKey).filter(Objects::nonNull).collect(Collectors.toSet()));
                        costList.clear();
                        log.info("拉取费用单模板{}映射{}数据源{} 第{}页后本批处理完成,唯一键[{}]", templateInfoStr, mappingStr, dataSource, c, String.join(",", costUniqueKeySet));
                    }
                }
                log.info("拉取费用单模板{}映射{}数据源{}数据转存完成", templateInfoStr, mappingStr, dataSource);
            }
            log.info("费单获取完成,费单总任务数:{},已执行完成任务数:{}", totalTasks, currentTasks);

            // 6.聚合数据:将数据根据唯一值取出，并剔除已确认的数据，聚合成一个费用核对主数据
            List<AuditFeeCheck> newFeeCheckList = generateAuditFeeCheckCostAsyncHelper.doSummary(costUniqueKeySet, feeMatchTemplate, auditFeeReqDto, mappingVo);
            log.info("拉取费用单 模板{} 生成核对总条数:{}", templateInfoStr, JSON.toJSONString(newFeeCheckList));

        } catch (Exception e) {
            log.error("费用核对获取费用单数据失败！");
            log.error("", e);
        } finally {
            if (hasLock) {
                log.info("拉取模板{}费用单已解锁", feeMatchTemplate.getName());
                redisLockService.unlock(redisLockKey);
            }
        }
    }

    /**
     * 请求KMS数据
     * @param template
     * @param dataSource
     * @param currentTasks
     * @param currentPage
     * @param auditFeeReqDto
     * @param respVoList
     * @return
     */
    private long getKMSPage(String template, String dataSource, AtomicInteger currentTasks,
        Pageable currentPage, AuditFeeReqDto auditFeeReqDto, List<AuditFeeRespVo> respVoList) {

        long timeUsed = 0;
        try {
            long before = System.currentTimeMillis();
            Page<AuditFeeRespVo> costPages = auditFeeCostService.findByConditions(currentPage, auditFeeReqDto);
            timeUsed = System.currentTimeMillis() - before;
            List<AuditFeeRespVo> records = costPages.getRecords();
            if (CollectionUtils.isNotEmpty(records)) {
                List<BigDecimal> amounts = records.stream().map(AuditFeeRespVo::getAmount).collect(Collectors.toList());
                log.info("获取费用数据金额:{}", JSON.toJSONString(amounts));
                // 金额空或0的数据不要，不浪费内存
                records = records.stream().filter(e -> e.getAmount() != null && !BigDecimal.ZERO.equals(e.getAmount())).collect(Collectors.toList());
                // 记录数据 5k条将处理一次
                respVoList.addAll(records);
                currentTasks.getAndIncrement();
            }
        } catch (Exception e) {
            log.info("费用核对获取费用单数据失败!,[当前模板:{},数据源:{},当前页:{}, 错误{}]",template,dataSource,currentPage.getPageNumber(), e.getMessage());
            throw e;
        }
        return timeUsed;
    }

    private List<AuditFeeCheckCost> convertToAuditFeeCheckCost(List<AuditFeeRespVo> respVoList, TpmDeductionMatchingTemplateAllowanceVo allowanceVo, TpmDeductionMatchingTemplateVo templateVo, TpmDeductionDetailMappingVo mappingVo) {
        if (CollectionUtils.isEmpty(respVoList)) return new ArrayList<>(0);
        List<AuditFeeCheckCost> costList = new ArrayList<>();
        for (AuditFeeRespVo respVo : respVoList) {

            AuditFeeCheckCost cost = auditFeeCheckCostRepository.initOne();
            costList.add(cost);
            // 因为分片不能为空，设置专用值标识
            cost.setAuditFeeCheckCode(AuditFeeCheckCost.MATCH_CODE_NULL);
            cost.setBusinessFormatCode(respVo.getBusinessFormatCode());
            cost.setBusinessUnitCode(respVo.getBusinessUnitCode());
            cost.setBusinessArea(respVo.getBusinessArea());
            cost.setCustomerRetailerCode(respVo.getCustomerRetailerCode());
            cost.setCustomerRetailerName(respVo.getCustomerRetailerName());
            cost.setProvinceCode(respVo.getProvinceCode());
            cost.setProvinceName(respVo.getProvinceName());
            cost.setSoldToPartyCode(respVo.getSoldToPartyCode());
            cost.setSoldToPartyName(respVo.getSoldToPartyName());
            cost.setDeliveryPartyCode(respVo.getDeliveryPartyCode());
            cost.setDeliveryPartyName(respVo.getDeliveryPartyName());
            cost.setProductCode(respVo.getProductCode());
            cost.setProductName(respVo.getProductName());
            cost.setSlotDateName(respVo.getSlotDateName());
            cost.setActualYearMonth(DateUtil.format(respVo.getCreateTime(), "yyyy-MM"));
            cost.setOrderDate(respVo.getOrderDate());
            if (StringUtils.isNotBlank(respVo.getOrderDate())) {
                cost.setOrderYearMonth(DateUtil.format(DateUtil.parseDate(respVo.getOrderDate()), "yyyy-MM"));
            }
            cost.setAuditWay(respVo.getAuditWay());
            cost.setCashingType(respVo.getCashingType());
            cost.setDeductionAmountTax(respVo.getAmount());
            cost.setSalesOrgCode(respVo.getSalesOrgCode());
            cost.setSalesOrgName(respVo.getSalesOrgName());
            cost.setRemark(respVo.getRemark());
            cost.setDataSource(respVo.getDataSource());
            // 直接存拼好的扣费明细编码，不根据类型存
            cost.setCompanyCostCode(respVo.getTpmDeductionCode());
            // 扣费项
            cost.setDeductionCode(respVo.getDeductionCode());
            cost.setDeductionName(respVo.getDeductionName());
            // 取自容差
            cost.setAllowanceKey(allowanceVo.getDeductionUniqueKey());
            cost.setActivityFormDesc(respVo.getActivityFormDesc());
            this.generateCostSummaryUnique(cost, templateVo, mappingVo);
        }
        return costList;
    }

    /**
     * 根据扣费模板规则生成唯一编码，作为聚合依据
     * @param cost
     * @param templateVo
     * @param mappingVo
     */
    public void generateCostSummaryUnique(AuditFeeCheckCost cost, TpmDeductionMatchingTemplateVo templateVo, TpmDeductionDetailMappingVo mappingVo) {

        // 聚合规则：业态+业务单元+销售机构（非必填）+零售商+客户（非必填）+扣费明细项+费用单匹配条件+费用单适用区域
        StringBuilder keyBuilder = new StringBuilder();
        // 公共key部分
        String separator = "_";
        keyBuilder.append(cost.getBusinessUnitCode());
        keyBuilder.append(separator).append(cost.getBusinessFormatCode());

        List<String> dimensionList = Arrays.asList(templateVo.getFeeMatchingCondition().split(","));
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.PRODUCT_CODE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getProductCode()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.TERMINAL_CODE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getDeliveryPartyCode()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getOrderYearMonth()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.FEE_CODE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getCompanyCostCode()).split("-")[0]);
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.FEE_ITEM_CODE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getCompanyCostCode()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.PROVINCE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getProvinceCode()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.AREA_CODE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getBusinessArea()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.SCHEDULE.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getSlotDateName()));
        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.CHANNEL.getCode())){

        }
        if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.ACTIVITY_FORM_DESC.getCode())){
            keyBuilder.append(separator).append(ifNull(cost.getActivityFormDesc()));
        }

        keyBuilder.append(separator).append(ifNull(cost.getDataSource()));
        keyBuilder.append(separator).append(ifNull(mappingVo.getSalesInstitutionErpCode()));
        // 数据的零售商
        keyBuilder.append(separator).append(ifNull(cost.getCustomerRetailerCode()));
        // 客户
        if (CollectionUtils.isNotEmpty(mappingVo.getCustomerList())) {
            List<String> customerCodeList = mappingVo.getCustomerList().stream().sorted(Comparator.comparing(TpmDeductionDetailMappingCustomerVo::getCustomerCode)).map(TpmDeductionDetailMappingCustomerVo::getCustomerCode).filter(Objects::nonNull).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(customerCodeList)) {
                String customerCodeListStr = String.join(separator, customerCodeList);
                keyBuilder.append(separator).append(ifNull(customerCodeListStr));
            }
        }
        // 扣费明细项：将一个映射中的所有扣费明细项拼接在一起，所对应的费用单都聚合为一个
        if (CollectionUtils.isNotEmpty(mappingVo.getDeductionDetailMappingTextList())) {
            List<String> deductionDetailMappingTextCodeList = mappingVo.getDeductionDetailMappingTextList().stream()
                .sorted(Comparator.comparing(TpmDeductionDetailMappingTextVo::getCode))
                .map(TpmDeductionDetailMappingTextVo::getCode).filter(Objects::nonNull).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(deductionDetailMappingTextCodeList)) {
                String deductionDetailMappingTextCodeListStr = String.join(separator, deductionDetailMappingTextCodeList);
                keyBuilder.append(separator).append(ifNull(deductionDetailMappingTextCodeListStr));
            }
        }
        cost.setMd5UniqueKey(MD5Util.computeMD5(keyBuilder.toString()));
    }

    private String ifNull(String s){
        return s == null ? "" : s;
    }
}
