package com.biz.crm.tpm.business.audit.fee.local.service.internal.check.async;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.kms.business.audit.fee.sdk.dto.AuditFeeReqDto;
import com.biz.crm.kms.business.audit.fee.sdk.enums.AuditFeeMatchStatusEnum;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheck;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheckCost;
import com.biz.crm.tpm.business.audit.fee.local.entity.check.AuditFeeCheckPos;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckCostRepository;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckDetailPlanRepository;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckDiffRepository;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckPosRepository;
import com.biz.crm.tpm.business.audit.fee.local.repository.check.AuditFeeCheckRepository;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.DiffTypeEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.MatchStatusEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.DeductionMatchingTemplateConditionEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingRelationActivityConfigVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingVo;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/**
 * @Description:
 * @Author qiancheng
 * @Date 2023/6/13
 */
@Slf4j
@Component
public class GenerateAuditFeeCheckCostAsyncHelper {

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;
    @Autowired(required = false)
    private AuditFeeCheckRepository auditFeeCheckRepository;

    @Autowired(required = false)
    private AuditFeeCheckDetailPlanRepository auditFeeCheckDetailPlanRepository;

    @Autowired(required = false)
    private AuditFeeCheckDiffRepository auditFeeCheckDiffRepository;

    @Autowired(required = false)
    private AuditFeeCheckCostRepository auditFeeCheckCostRepository;

    @Autowired(required = false)
    private AuditFeeCheckPosRepository auditFeeCheckPosRepository;

    @Autowired(required = false)
    private KMSFeeUpdateAsync kmsFeeUpdateAsync;

    /**
     * cost聚合成auditFee
     * 剔除已确认的auditFee数据
     * @param costUniqueKeySet
     * @param feeMatchTemplate
     * @param auditFeeReqDto
     * @param mappingVo
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public List<AuditFeeCheck> doSummary(Set<String> costUniqueKeySet, TpmDeductionMatchingTemplateVo feeMatchTemplate, AuditFeeReqDto auditFeeReqDto, TpmDeductionDetailMappingVo mappingVo) {
        if (CollectionUtils.isEmpty(costUniqueKeySet)) return new ArrayList<>(0);
        log.info("模板编号{}拉取费用单开始汇总数据", feeMatchTemplate.getCode());

        // 清理同模板未确认数据
        this.removeUnconfirmedAuditFeeCheck(feeMatchTemplate);
        // 删除拉取到，但已被确认/差异确认的费用单
        auditFeeCheckCostRepository.deleteUsedCosts(Lists.newArrayList(costUniqueKeySet));

        List<String> dimensionList = Arrays.asList(feeMatchTemplate.getFeeMatchingCondition().split(","));

        List<AuditFeeCheck> feeCheckList = new ArrayList<>();
        for (String uniqueKey : costUniqueKeySet) {
            // 查询一条，必须取matchCode为空的数据，否则会统计到拉取前确认的数据
            AuditFeeCheckCost costOne = auditFeeCheckCostRepository.findOneByMd5UniqueKey(uniqueKey);
            AuditFeeCheckCost sumOne = auditFeeCheckCostRepository.findSumAmountByMd5UniqueKey(uniqueKey);
            if (costOne == null || sumOne == null) continue;
            BigDecimal deductionAmountTax = sumOne.getDeductionAmountTax();
            log.info("模板编号{}汇总数据MD5Key{} 对应的数据存在", feeMatchTemplate.getCode(), uniqueKey);

            AuditFeeCheck feeCheck = auditFeeCheckRepository.initOne();
            feeCheck.setBusinessFormatCode(costOne.getBusinessFormatCode());
            feeCheck.setBusinessUnitCode(costOne.getBusinessUnitCode());

            // 如果模板规则含有按区域聚合，就用费用明细数据的区域；未按区域聚合，费用明细数据会出现各种区域，此时置空
            if(dimensionList.contains(DeductionMatchingTemplateConditionEnum.AREA_CODE.getCode())) {
                // 聚合条件含有区域
                feeCheck.setAreaCode(costOne.getBusinessArea());
            }
            if (null != costOne.getActualYearMonth()) feeCheck.setActualYearMonth(costOne.getActualYearMonth());
            if (null != costOne.getOrderYearMonth()) feeCheck.setOrderYearMonthStr(costOne.getOrderYearMonth());
            if (null != costOne.getOrderYearMonth()) feeCheck.setOrderYearMonth(DateUtil.parse(costOne.getOrderYearMonth(), "yyyy-MM"));
            if (null != costOne.getOrderDate()) feeCheck.setOrderDate(DateUtil.parse(costOne.getOrderDate(), "yyyy-MM-dd"));
            feeCheck.setCustomerRetailerCode(costOne.getCustomerRetailerCode());
            feeCheck.setCustomerRetailerName(costOne.getCustomerRetailerName());
            feeCheck.setTerminalCode(costOne.getDeliveryPartyCode());
            feeCheck.setTerminalName(costOne.getDeliveryPartyName());
            feeCheck.setCustomerErpCode(costOne.getSoldToPartyCode());
            feeCheck.setCustomerName(costOne.getSoldToPartyName());
            feeCheck.setProvinceCode(costOne.getProvinceCode());
            feeCheck.setProvinceName(costOne.getProvinceName());
            feeCheck.setProductCode(costOne.getProductCode());
            feeCheck.setProductName(costOne.getProductName());
            feeCheck.setMd5UniqueKey(costOne.getMd5UniqueKey());
            feeCheck.setAllowanceKey(costOne.getAllowanceKey());
            feeCheck.setMatchTemplateCode(feeMatchTemplate.getCode());
            feeCheck.setMatchTemplateName(feeMatchTemplate.getName());
            feeCheck.setMatchTemplateType(feeMatchTemplate.getDeductionMatchingTemplateType());
            feeCheck.setSalesOrgCode(costOne.getSalesOrgCode());
            feeCheck.setSalesOrgName(costOne.getSalesOrgName());
            feeCheck.setMatchStatus(MatchStatusEnum.WAIT_MATCH.getCode());
            feeCheck.setIsConfirm(BooleanEnum.FALSE.getCapital());
            feeCheck.setIsConfirmDiff(BooleanEnum.FALSE.getCapital());
            feeCheck.setMatchCode(generateCodeService.generateCode(AuditFeeConstants.AUDIT_FEE_CODE, 1).get(0));
            // 汇总金额
            feeCheck.setCostDeductionAmount(deductionAmountTax);
            feeCheck.setCostDetailDiffAmount(feeCheck.getCostDeductionAmount());
            feeCheck.setDiffConfirmAmount(feeCheck.getCostDetailDiffAmount());
            feeCheck.setIsAddUp(feeMatchTemplate.getIsAddUpMapping());
            feeCheck.setDataSource(costOne.getDataSource());
            feeCheck.setActivityFormDesc(costOne.getActivityFormDesc());
            if (mappingVo != null && CollectionUtils.isNotEmpty(mappingVo.getDeductionDetailMappingRelationActivityConfigList())) {
                TpmDeductionDetailMappingRelationActivityConfigVo activityConfigVo = mappingVo.getDeductionDetailMappingRelationActivityConfigList().get(0);
                feeCheck.setActivityTypeCode(activityConfigVo.getActivityTypeCode());
                feeCheck.setActivityTypeName(activityConfigVo.getActivityTypeName());
            }
            int diffAmount = feeCheck.getDiffConfirmAmount().compareTo(BigDecimal.ZERO);
            if (diffAmount == 0) {
                feeCheck.setDiffType(DiffTypeEnum.NO_DIFF.getCode());
            }
            if (diffAmount > 0) {
                feeCheck.setDiffType(DiffTypeEnum.MORE_DEDUCTION_NO_DEDUCTION.getCode());
            }
            if (diffAmount < 0) {
                feeCheck.setDiffType(DiffTypeEnum.LESS_DEDUCTION_HAD_DEDUCTION.getCode());
            }

            log.info("模板编号{}汇总数据MD5Key{} 对应的数据{}", feeMatchTemplate.getCode(), uniqueKey, JSON.toJSONString(feeCheck));

            // 直接按条件更新cost，只更新matchCode为空的数据
            auditFeeCheckCostRepository.updateMatchedByMd5UniqueKey(uniqueKey, feeCheck.getMatchCode());
            feeCheckList.add(feeCheck);
        }
        // 保存费用核对主数据
        auditFeeCheckRepository.saveBatch(feeCheckList);
        log.info("模板编号{}拉取费用单汇总数据完成汇总及写入", feeMatchTemplate.getCode());
        return feeCheckList;
    }

    /**
     * 按模板回退历史抽取的待确认数据
     * 目前是写入了本次数据后再进行的回退，根据本次模板，查询待确认的聚合数据，再根据其matchCode清理
     * @param template 模板
     */
    private void removeUnconfirmedAuditFeeCheck(TpmDeductionMatchingTemplateVo template){
        if (template == null) {
            return;
        }

        List<AuditFeeCheck> feeChecks = this.auditFeeCheckRepository.findNotConfirmedDiff(template.getCode());
        if (CollectionUtils.isNotEmpty(feeChecks)) {
            List<String> matchCodes = feeChecks.stream().map(AuditFeeCheck::getMatchCode).distinct().collect(Collectors.toList());
            // 确认差异/确认 的数据都不能回退
            this.deleteByMatchCodes(matchCodes);
        }
        log.info("模板编号{}拉取费用单已清理未确认数据", template.getCode());
    }

    /**
     * 更新KMS费用单状态
     * @param tpmDeductionCodeList 费用单编码，含行号
     * @param costIsMatch 费用单已使用
     */
    public void updateKMSStatus(List<String> tpmDeductionCodeList, String costIsMatch) {
        if (CollectionUtils.isEmpty(tpmDeductionCodeList)) return;
        // 只打已确认标记
        if (!AuditFeeMatchStatusEnum.CONFIRM.getCode().equals(costIsMatch)) return;
        AuditFeeReqDto reqDto = new AuditFeeReqDto();
        reqDto.setTpmDeductionCodeList(tpmDeductionCodeList);
        reqDto.setCostIsMatch(costIsMatch);
        AuditFeeCheckCost cost = auditFeeCheckCostRepository.findByCompanyCode(tpmDeductionCodeList.get(0));
        reqDto.setDataSource(cost.getDataSource());
        log.info("费用核对-更新KMS费用单状态参数：{}", JSON.toJSONString(reqDto));
        kmsFeeUpdateAsync.updateKMSStatusAsync(reqDto);
    }

    /**
     * 回退核对数据，为了重新拉取
     * @param matchCodes
     */
    @Transactional
    public void deleteByMatchCodes(List<String> matchCodes) {
        if (CollectionUtils.isEmpty(matchCodes)) {
            return;
        }
        matchCodes = matchCodes.stream().distinct().collect(Collectors.toList());
        auditFeeCheckRepository.updateDelFlagByMatchCodes(matchCodes);
        auditFeeCheckCostRepository.updateDelFlagByMatchCodes(matchCodes);
        auditFeeCheckDetailPlanRepository.updateDelFlagByMatchCodes(matchCodes);
        auditFeeCheckDiffRepository.updateDelFlagByMatchCodes(matchCodes);
        this.cancelRelatePos(matchCodes);
    }

    /**
     * 取消费用与活动的匹配
     * 取消只需要更新主数据，cost费用不动，活动/差异要清理，pos要回退
     * @param matchCodes
     */
    @Transactional
    public void cancelByMatchCodes(List<String> matchCodes) {
        if (CollectionUtils.isEmpty(matchCodes)) return;
        auditFeeCheckDetailPlanRepository.updateDelFlagByMatchCodes(matchCodes);
        auditFeeCheckDiffRepository.updateDelFlagByMatchCodes(matchCodes);
        this.cancelRelatePos(matchCodes);
    }

    private void cancelRelatePos(List<String> matchCodes) {
        List<AuditFeeCheckPos> posList = auditFeeCheckPosRepository.findByFeeCheckMatchCodes(matchCodes);
        if (CollectionUtils.isNotEmpty(posList)) {
            List<String> posIdList = new ArrayList<>(posList.size());
            for (AuditFeeCheckPos pos : posList) {
                pos.setIsMatchCost(BooleanEnum.FALSE.getCapital());
                pos.setSharePromotionDeduction(BigDecimal.ZERO);
                pos.setFeeCheckMatchCode(null);
                posIdList.add(pos.getId());
            }
            auditFeeCheckPosRepository.updateBatchById(posList);
            auditFeeCheckPosRepository.computeTotalDiffAmount(posIdList);
        }
    }
}
