package com.biz.crm.tpm.business.audit.fee.local.service.internal.ledger;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.mn.common.page.cache.service.MnPageCacheHelper;
import com.biz.crm.tpm.business.audit.fee.local.entity.ledger.AuditFeeDiffLedgerPlan;
import com.biz.crm.tpm.business.audit.fee.local.repository.ledger.AuditFeeDiffLedgerPlanRepository;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.dispose.AuditFeeDiffDisposeDetailDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.ledger.AuditFeeDiffLedgerPlanDto;
import com.biz.crm.tpm.business.audit.fee.sdk.service.dispose.AuditFeeDiffDisposeDetailVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.dispose.AuditFeeDiffDisposeDetailVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.ledger.AuditFeeDiffLedgerPlanVo;
import com.biz.crm.tpm.business.budget.cal.config.sdk.dto.BudgetCalConfigProductRatioDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wanghaojia
 * @date 2022/12/8 11:19
 */
@Component
public class AuditFeeDiffLedgerPlanCacheHelper extends MnPageCacheHelper<AuditFeeDiffLedgerPlanVo, AuditFeeDiffLedgerPlanDto> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private AuditFeeDiffLedgerPlanRepository auditFeeDiffLedgerPlanRepository;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    @Override
    public String getCacheKeyPrefix() {
        return AuditFeeConstants.AUDIT_FEE_DIFF_LEDGER_PLAN_CACHE_KEY;
    }

    @Override
    public Class<AuditFeeDiffLedgerPlanDto> getDtoClass() {
        return AuditFeeDiffLedgerPlanDto.class;
    }

    @Override
    public Class<AuditFeeDiffLedgerPlanVo> getVoClass() {
        return AuditFeeDiffLedgerPlanVo.class;
    }

    /**
     * 从数据库查询初始化数据
     * @param dto
     * @param cacheKey
     * @return {@link List}<{@link AuditFeeDiffLedgerPlanDto}>
     */
    @Override
    public List<AuditFeeDiffLedgerPlanDto> findDtoListFromRepository(AuditFeeDiffLedgerPlanDto dto, String cacheKey) {
        Assert.hasText(cacheKey, "cacheKey不能为空");
        Assert.hasText(dto.getFeeDiffLedgerCode(), "差异费用台帐编码不能为空");
        //查询明细
        List<AuditFeeDiffLedgerPlan> entities = this.auditFeeDiffLedgerPlanRepository
            .findByFeeDiffLedgerCode(dto);
        List<AuditFeeDiffLedgerPlanDto> detailDtos =
            (List<AuditFeeDiffLedgerPlanDto>) this.nebulaToolkitService
                .copyCollectionByWhiteList(entities, AuditFeeDiffLedgerPlan.class,
                    AuditFeeDiffLedgerPlanDto.class, LinkedHashSet.class, ArrayList.class);
        if(!CollectionUtils.isEmpty(detailDtos)){
            //获取缓存key
            String redisCacheIdKey = getRedisCacheIdKey(cacheKey);
            String redisCacheDataKey = getRedisCacheDataKey(cacheKey);
            Map<String, AuditFeeDiffLedgerPlanDto> ratioDtoMap = detailDtos.stream()
                .collect(Collectors.toMap(AuditFeeDiffLedgerPlanDto::getId, Function.identity()));
            //保存到缓存中
            redisService.lPushAll(redisCacheIdKey, this.getExpireTime(), ratioDtoMap.keySet().toArray());
            redisTemplate.opsForHash().putAll(redisCacheDataKey, ratioDtoMap);
            redisService.expire(redisCacheDataKey, this.getExpireTime());
        }
        return detailDtos;
    }

    /**
     * 是否初始化数据到缓存，默认为true
     * @param
     * @return {@link boolean}
     */
    @Override
    public boolean initToCacheFromRepository() {
        return false;
    }

    /**
     * 新增数据
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link AuditFeeDiffLedgerPlanDto}>
     */
    @Override
    public List<AuditFeeDiffLedgerPlanDto> newItem(String cacheKey, List<AuditFeeDiffLedgerPlanDto> itemList) {
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        //获取已缓存的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        //过滤掉已经存在的数据
        List<AuditFeeDiffLedgerPlanDto> newList = itemList.stream()
            .filter(item -> !idList.contains(item.getId())).collect(Collectors.toList());
        if(newList.size() == 0){
            AuditFeeDiffLedgerPlanDto detailDto = new AuditFeeDiffLedgerPlanDto();
            detailDto.setId(UuidCrmUtil.general());
            detailDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            detailDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            detailDto.setTenantCode(TenantUtils.getTenantCode());
            newList.add(detailDto);
        }
        return newList;
    }

    /**
     * 更新数据
     */
    @Override
    public void updateItem(String cacheKey, List<AuditFeeDiffLedgerPlanDto> itemList){
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        itemList.forEach(item -> {
            if (!Objects.isNull(item.getPlanRecoveredAmount())) {
                if (BigDecimal.ZERO.compareTo(item.getPlanRecoveredAmount()) > 0) {
                    throw new IllegalArgumentException("计划追回金额必须 ≥ 0 ！");
                }
            }
        });
    }

    /**
     * 复制数据
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link AuditFeeDiffDisposeDetailDto}>
     */
    @Override
    public List<AuditFeeDiffLedgerPlanDto> copyItem(String cacheKey, List<AuditFeeDiffLedgerPlanDto> itemList) {
        List<AuditFeeDiffLedgerPlanDto> newList = (List<AuditFeeDiffLedgerPlanDto>) this.nebulaToolkitService
            .copyCollectionByBlankList(itemList, AuditFeeDiffLedgerPlanDto.class,
                AuditFeeDiffLedgerPlanDto.class, HashSet.class, ArrayList.class);
        for (AuditFeeDiffLedgerPlanDto newItem : newList) {
            newItem.setId(UuidCrmUtil.general());
            newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            newItem.setTenantCode(TenantUtils.getTenantCode());
        }
        return newList;
    }

    /**
     * 获取唯一key
     * @param dto
     * @return {@link Object}
     */
    @Override
    public Object getDtoKey(AuditFeeDiffLedgerPlanDto dto) {
        return dto.getId();
    }

    /**
     * 获取是否选中状态
     * @param dto
     * @return {@link String}
     */
    @Override
    public String getCheckedStatus(AuditFeeDiffLedgerPlanDto dto) {
        return dto.getChecked();
    }

    /**
     * 导入数据存入缓存
     * @param cacheKey
     * @param dto
     */
    public void importSave(String cacheKey, AuditFeeDiffLedgerPlanDto dto) {
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        dto.setId(UuidCrmUtil.general());
        //获取已缓存的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        //判断是否已经存在
        if(!idList.contains(dto.getId())){
            //缓存ID
            Object[] newIdArr = {dto.getId()};
            redisService.lPushAll(redisCacheIdKey, this.getExpireTime(),newIdArr );
        }

        //导入新增数据
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());

        //缓存数据
        Map<Object, AuditFeeDiffLedgerPlanDto> ratioDtoMap = Maps.newHashMap();
        ratioDtoMap.put(dto.getId(), dto);

        redisTemplate.opsForHash().putAll(redisCacheDataKey, ratioDtoMap);
        redisService.expire(redisCacheDataKey, this.getExpireTime());
    }
}
