package com.biz.crm.tpm.business.audit.fee.local.service.internal.ledger;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.audit.fee.local.entity.ledger.AuditFeeDiffLedger;
import com.biz.crm.tpm.business.audit.fee.local.entity.ledger.AuditFeeDiffLedgerDeduction;
import com.biz.crm.tpm.business.audit.fee.local.repository.ledger.AuditFeeDiffLedgerDeductionRepository;
import com.biz.crm.tpm.business.audit.fee.local.repository.ledger.AuditFeeDiffLedgerRepository;
import com.biz.crm.tpm.business.audit.fee.local.service.AuditFeeDiffLedgerService;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.dispose.AuditFeeDiffDisposeLedgerItemDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import jdk.nashorn.internal.runtime.logging.DebugLogger;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 台帐内部接口实现
 * @Author YangWei
 * @Date 2023/3/30 下午5:00
 */
@Slf4j
@Service("auditFeeDiffLedgerService")
public class AuditFeeDiffLedgerServiceImpl implements AuditFeeDiffLedgerService {

    @Autowired(required = false)
    private AuditFeeDiffLedgerRepository auditFeeDiffLedgerRepository;

    @Autowired
    private AuditFeeDiffLedgerDeductionRepository auditFeeDiffLedgerDeductionRepository;

    /**
     * 处理台帐信息
     * @param feeDiffDisposeCode
     * @param ledgerItems
     */
    @Override
    public void handle(String feeDiffDisposeCode, List<AuditFeeDiffDisposeLedgerItemDto> ledgerItems) {
        if (CollectionUtils.isEmpty(ledgerItems)) {
            return;
        }
        //每条台帐申请金额
        Map<String, BigDecimal> ledgerAmountMap = Maps.newHashMap();
        //费用处理申请明细编码
        Set<String> detailPlanItemCodeSet = Sets.newHashSet();
        //扣减明细
        List<AuditFeeDiffLedgerDeduction> thisDeductionList = Lists.newArrayList();
        ledgerItems.forEach(ledgerItemDto -> {
            //台帐编码
            String feeDiffLedgerCode = ledgerItemDto.getFeeDiffLedgerCode();
            BigDecimal ledgerAmount = Optional.ofNullable(ledgerAmountMap.get(feeDiffLedgerCode)).orElse(BigDecimal.ZERO);
            BigDecimal thisUseAmount = ledgerItemDto.getThisUseAmount();
            ledgerAmountMap.put(feeDiffLedgerCode, ledgerAmount.add(thisUseAmount));
            detailPlanItemCodeSet.add(ledgerItemDto.getDetailPlanItemCode());
            this.buildThisDeductionList(thisDeductionList, ledgerItemDto);
        });
        //查询已经存在的台帐扣减明细信息
        List<AuditFeeDiffLedgerDeduction> histDeductionEntities = this.auditFeeDiffLedgerDeductionRepository
            .findByFeeDiffDisposeCodeAndDetailPlanItemCodes(feeDiffDisposeCode, Lists.newArrayList(detailPlanItemCodeSet));
        Map<String, BigDecimal> histDeductionAmountMap = histDeductionEntities.stream()
            .collect(Collectors.groupingBy(AuditFeeDiffLedgerDeduction::getFeeDiffLedgerDisposeCode
                , Collectors.reducing(BigDecimal.ZERO, AuditFeeDiffLedgerDeduction::getRecoveredAmount, BigDecimal::add)));
        //查询所有台帐信息
        List<AuditFeeDiffLedger> ledgerEntities = this.auditFeeDiffLedgerRepository
            .findDetailByCodes(Lists.newArrayList(ledgerAmountMap.keySet()));
        ledgerEntities.forEach(entity -> {
            //历史金额需要先还回去
            BigDecimal histAmount = Optional.ofNullable(histDeductionAmountMap.get(entity.getFeeDiffLedgerCode())).orElse(BigDecimal.ZERO);
            //最新
            BigDecimal thisAmount = Optional.ofNullable(ledgerAmountMap.get(entity.getFeeDiffLedgerCode())).orElse(BigDecimal.ZERO);
            //已追回金额
            entity.setRecoveredAmount(entity.getRecoveredAmount().subtract(histAmount).add(thisAmount));
            //待追回金额
            entity.setBeRecoveredAmount(entity.getDiffAmount().subtract(entity.getRecoveredAmount()));
            Assert.isTrue(entity.getBeRecoveredAmount().compareTo(BigDecimal.ZERO) >= 0, "台帐[" + entity.getFeeDiffLedgerCode() + "]费用超额使用");
        });
        //删除历史扣减明细
        this.auditFeeDiffLedgerDeductionRepository.deleteByFeeDiffDisposeCode(feeDiffDisposeCode);
        //保存扣减明细
        this.auditFeeDiffLedgerDeductionRepository.saveBatch(thisDeductionList);
        //保存台帐
        this.auditFeeDiffLedgerRepository.updateBatchById(ledgerEntities);
    }

    /**
     * 构建扣减明细
     * @param thisDeductionList
     * @param ledgerItemDto
     */
    private void buildThisDeductionList(List<AuditFeeDiffLedgerDeduction> thisDeductionList, AuditFeeDiffDisposeLedgerItemDto ledgerItemDto) {
        AuditFeeDiffLedgerDeduction deduction = new AuditFeeDiffLedgerDeduction();
        deduction.setFeeDiffLedgerDisposeCode(ledgerItemDto.getFeeDiffDisposeCode());
        deduction.setBusinessCode(ledgerItemDto.getDetailPlanItemCode());
        deduction.setRecoveredAmount(ledgerItemDto.getThisUseAmount());
        deduction.setRecoveredTime(ledgerItemDto.getRecoveredTime());
        deduction.setTenantCode(TenantUtils.getTenantCode());
        deduction.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        deduction.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        thisDeductionList.add(deduction);
    }
}